#pragma once

#include "box_tree_generator.h"

#include <infra/pod_agent/libs/behaviour/template_bt_storage/template_bt_storage.h>
#include <infra/pod_agent/libs/network_client/client.h>
#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/status_and_ticker_holder/status_and_ticker_holder.h>
#include <infra/pod_agent/libs/porto_client/async_client.h>
#include <infra/pod_agent/libs/posix_worker/posix_worker.h>

namespace NInfra::NPodAgent {

/**
    Creates behavior tree for each new workload from provided TreeTemplate
    Removes unneeded trees
    Ticks trees via TMtpPeriodTicker
    Method Wait() waits for signal from Shutdown() and waits the TMtpPeriodTicker
    Method Stop() calls {Shutdown(); Wait();}
*/
class TWorkloadTreeGenerator {
public:
    TWorkloadTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& workloadTreeTemplate
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TNetworkClientPtr networkClient
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& podAgentBinaryFilePathInBox
        , ui32 maxLogsFileSize
        , bool isBoxAgentMode = false
    )
        : Logger_(logger)
        , PathHolder_(pathHolder)
        , WorkloadTreeTemplate_(workloadTreeTemplate)
        , Porto_(porto)
        , PosixWorker_(posixWorker)
        , NetworkClient_(networkClient)
        , StatusNTickerHolder_(statusNTickerHolder)
        , TemplateBTStorage_(templateBTStorage)
        , PodAgentBinaryFilePathInBox_(podAgentBinaryFilePathInBox)
        , MaxLogsFileSize_(maxLogsFileSize)
        , IsBoxAgentMode_(isBoxAgentMode)
    {
    }

    ~TWorkloadTreeGenerator() {}

public:
    struct TWorkloadToAdd {
        TUpdateHolder::TWorkloadTarget Target_;
        API::EWorkloadTargetState TargetState_;
        // need for tests
        TMap<TString, TString> TreeReplaceMap_;
    };

    struct TWorkloadsToAdd {
        TMap<TString, TWorkloadToAdd> Workloads_;
    };

private:
    struct TWorkloadHookData {
        TWorkloadMeta::THookInfo Info_;
        TMap<TString, TString> TreeReplaceMap_;
    };

public:
    TWorkloadTreeGenerator::TWorkloadsToAdd UpdateSpec(
        const google::protobuf::RepeatedPtrField<API::TWorkload>& workloads
        , API::EPodAgentTargetState podAgentTargetState
        , const google::protobuf::RepeatedPtrField<API::TMutableWorkload>& mutableWorkloads
        , const NSecret::TSecretMap& secretMap
        , const TBoxTreeGenerator::TBoxesToAdd& boxes
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , bool useEnvSecret = false
        , bool autoDecodeBase64Secrets = false
    );

    void RemoveWorkloads(const TWorkloadsToAdd& toAdd);
    void AddWorkloads(const TWorkloadsToAdd& toAdd);

protected:
    TWorkloadMeta::THookInfo GetWorkloadHookInfo(
        const TString& containerName
        , bool isContainer
        , bool isHttp
        , bool isTcp
        , bool isUnixSignal
        , bool isStartContainerAliveCheck
    ) const;

    TMap<TString, TString> FillHookTypeReplaceMap(
        const TString& name
        , bool isContainer
        , bool isHttp
        , bool isTcp
        , bool isUnixSignal
        , bool isStartContainerAliveCheck
    ) const;

    TMap<TString, TString> FillHookTriesThresholdsReplaceMap(
        const TString& name
        , ui32 failureThreshold
        , ui32 successThreshold
    ) const;

    TMap<TString, TString> FillHttpGetReplaceMap(
        const API::THttpGet& httpGet
        , const TString& name
        , bool needValidation
        , bool needDefaultInitialDelay
    ) const;

    TMap<TString, TString> FillTcpCheckReplaceMap(
        const API::TTcpCheck& tcpCheck
        , const TString& name
        , bool needValidation
        , bool needDefaultInitialDelay
    ) const;

    TMap<TString, TString> FillUnixSignalReplaceMap(
        const API::TUnixSignal& unixSignal
        , const TString& name
    ) const;

    TWorkloadHookData GetReadinessCheckData(
        const API::TReadinessCheck& readinessCheck
        , const TString& workloadId
        , const TString& boxId
        , const NSecret::TSecretMap& secretMap
        , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
        , const double cpuToVcpuFactor
        , bool useEnvSecret = false
    ) const;

    TWorkloadHookData GetLivenessCheckData(
        const API::TLivenessCheck& livenessCheck
        , const TString& workloadId
        , const TString& boxId
        , const NSecret::TSecretMap& secretMap
        , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
        , const double cpuToVcpuFactor
        , bool useEnvSecret = false
    ) const;

    TWorkloadHookData GetStopPolicyData(
        const API::TStopPolicy& stopPolicy
        , const TString& workloadId
        , const TString& boxId
        , const NSecret::TSecretMap& secretMap
        , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
        , const double cpuToVcpuFactor
        , bool needValidation
        , bool useEnvSecret = false
    ) const;

    TWorkloadHookData GetDestroyPolicyData(
        const API::TDestroyPolicy& destroyPolicy
        , const TString& workloadId
        , const TString& boxId
        , const NSecret::TSecretMap& secretMap
        , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
        , const double cpuToVcpuFactor
        , bool needValidation
        , bool useEnvSecret = false
    ) const;

private:
    TWorkloadToAdd GetWorkloadToAdd(
        const API::TWorkload& workload
        , API::EPodAgentTargetState podAgentTargetState
        , const TMap<TString, const API::TMutableWorkload&>& mutableWorkloadMap
        , const TBoxTreeGenerator::TBoxesToAdd& boxes
        , const NSecret::TSecretMap& secretMap
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , bool useEnvSecret
        , bool autoDecodeBase64Secrets
    ) const;

private:
    static const i32 PORTO_LOG_FILE_SIZE_LIMIT_MULTIPLIER = 2;
    static const ui32 DEFAULT_HOOK_TRIES_THRESHOLD = 1;

    TLogger& Logger_;
    TPathHolderPtr PathHolder_;
    const TBehavior3 WorkloadTreeTemplate_;

    TAsyncPortoClientPtr Porto_;
    TPosixWorkerPtr PosixWorker_;
    TNetworkClientPtr NetworkClient_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;
    TTemplateBTStoragePtr TemplateBTStorage_;

    const TString PodAgentBinaryFilePathInBox_;
    const ui32 MaxLogsFileSize_;
    const bool IsBoxAgentMode_;
};

} // namespace NInfra::NPodAgent
