#include "workload_tree_generator.h"

#include "support_functions.h"
#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

class TMockWorkloadTreeGenerator : public TWorkloadTreeGenerator {
public:
    TMockWorkloadTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& treeTemplate
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TNetworkClientPtr networkClient
        , TStatusNTickerHolderPtr statusNTicketHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& podAgentBinaryFilePathInBox
        , ui32 maxLogsFileSize
        , const bool isBoxAgentMode
    ) : TWorkloadTreeGenerator(
        logger
        , pathHolder
        , treeTemplate
        , porto
        , posixWorker
        , networkClient
        , statusNTicketHolder
        , templateBTStorage
        , podAgentBinaryFilePathInBox
        , maxLogsFileSize
        , isBoxAgentMode
    )
    {
    }

    TWorkloadMeta::THookInfo GetWorkloadHookInfo(
        const TString& containerName
        , bool isContainer
        , bool isHttp
        , bool isTcp
        , bool isUnixSignal
        , bool isStartContainerAliveCheck
    ) const {
        return TWorkloadTreeGenerator::GetWorkloadHookInfo(containerName, isContainer, isHttp, isTcp, isUnixSignal, isStartContainerAliveCheck);
    }

    TMap<TString, TString> FillHookTypeReplaceMap(
        const TString& name
        , bool isContainer
        , bool isHttp
        , bool isTcp
        , bool isUnixSignal
        , bool isStartContainerAliveCheck
    ) {
        return TWorkloadTreeGenerator::FillHookTypeReplaceMap(name, isContainer, isHttp, isTcp, isUnixSignal, isStartContainerAliveCheck);
    }

    TMap<TString, TString> FillHookTriesThresholdsReplaceMap(
        const TString& name
        , ui32 failureThreshold
        , ui32 successThreshold
    ) const {
        return TWorkloadTreeGenerator::FillHookTriesThresholdsReplaceMap(name, failureThreshold, successThreshold);
    }

    TMap<TString, TString> FillUnixSignalReplaceMap(
        const API::TUnixSignal& unixSignal
        , const TString& name
    ) const {
        return TWorkloadTreeGenerator::FillUnixSignalReplaceMap(unixSignal, name);
    }
};

using TMockWorkloadTreeGeneratorPtr = TSimpleSharedPtr<TMockWorkloadTreeGenerator>;

class ITestWorkloadTreeGeneratorCanon: public ITestTreeGeneratorCanon {
protected:
    ITestWorkloadTreeGeneratorCanon(const bool isBoxAgentMode = false)
        : ITestTreeGeneratorCanon(isBoxAgentMode)
        , IsBoxAgentMode_(isBoxAgentMode)
    {}

    void CreateWorker() final {
        TMap<TString, TString> properties;
        properties["object_id_or_hash"] = "<RSLV:WORKLOAD_ID>";
        properties["state"] = "EWorkloadState_ACTIVE";
        properties["object_type"] = "workload";

        Worker_ = new TMockWorkloadTreeGenerator(
            logger
            , PathHolder_
            , GetTestTemplateTree(
                properties
                , "FeedbackObjectState"
                , "FeedbackObjectState(workload, EWorkloadState_ACTIVE)"
            )
            , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
            , new TPosixWorker(new TFakeThreadPool())
            , new TMockNetworkClient()
            , StatusNTickerHolder_
            , nullptr
            , "/pod_agent_public/pod_agent" /* podAgentBinaryFilePathInBox */
            , 0
            , IsBoxAgentMode_
        );
    }

    TWorkloadTreeGenerator::TWorkloadsToAdd UpdateSpec(
        const ::google::protobuf::RepeatedPtrField<API::TWorkload>& workloads
        , API::EPodAgentTargetState podAgentTargetState
        , const google::protobuf::RepeatedPtrField<API::TMutableWorkload>& mutableWorkloads
        , const NSecret::TSecretMap& secretMap
        , const TBoxTreeGenerator::TBoxesToAdd& boxes
        , ui64 specTimestamp
        , ui32 revision
        , bool autoDecodeBase64Secrets = false
    ) {
        TWorkloadTreeGenerator::TWorkloadsToAdd newSpec = Worker_->UpdateSpec(
            workloads
            , podAgentTargetState
            , mutableWorkloads
            , secretMap
            , boxes
            , 1.0 /* CpuToVcpuFactor */
            , specTimestamp
            , revision
            , false /* useEnvSecret */
            , autoDecodeBase64Secrets
        );
        Worker_->RemoveWorkloads(newSpec);
        Worker_->AddWorkloads(newSpec);
        return newSpec;
    }

protected:
    TMockWorkloadTreeGeneratorPtr Worker_;
    const bool IsBoxAgentMode_;
};

Y_UNIT_TEST_SUITE(WorkloadWorkerSuite) {

Y_UNIT_TEST(TestSimpleTree) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    public:
        TTest(const bool isBoxAgentMode)
            : ITestWorkloadTreeGeneratorCanon(isBoxAgentMode)
        {
        }

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");
            workload->mutable_readiness_check()->mutable_container()->set_command_line("echo readiness");
            workload->mutable_liveness_check()->mutable_container()->set_command_line("echo liveness");
            workload->mutable_stop_policy()->mutable_container()->set_command_line("echo stop");

            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_stop_policy()->set_max_tries(1000);


            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            auto workloadCopy = spec.add_workloads();
            workloadCopy->CopyFrom(*workload);

            auto mutableWorkloadCopy = spec.add_mutable_workloads();
            mutableWorkloadCopy->CopyFrom(*mutableWorkload);

            TBoxTreeGenerator::TBoxesToAdd boxes;
            if (!IsBoxAgentMode_) {
                BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
                BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId + "_other"));
                boxes.Boxes_.insert(
                    {
                        boxId
                        , {
                            NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash"), 0, {}
                        }
                    }
                );
                boxes.Boxes_.insert(
                 {
                        boxId + "_other"
                        , {
                            NObjectTargetTestLib::CreateBoxTargetSimple(boxId + "_other", "box_tree_hash"), 0, {}
                        }
                    }
                );
            }

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "workloads' ids are not unique: workload 'my_workload' occurs twice"
            );

            *workloadCopy->mutable_id() += "_other";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "Mutable workload not found at workload 'my_workload_other'"
            );

            *mutableWorkloadCopy->mutable_workload_ref() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));

            *workloadCopy->mutable_id() = "";
            *mutableWorkloadCopy->mutable_workload_ref() = "";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "One of workloads has empty id"
            );

            *workloadCopy->mutable_id() = "workload_other";
            *mutableWorkloadCopy->mutable_workload_ref() = "workload_other";
            auto workloadsToAdd = UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0);

            spec.add_mutable_workloads()->set_workload_ref("None");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "Different size of mutableWorkloads (3) and workloads (2)"
            );

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, WorkloadStatusRepository_->GetObjectStatus("my_workload").state());
            UNIT_ASSERT(workloadsToAdd.Workloads_.at("my_workload").Target_.Tree_->GetUseLongTickPeriod());
        }
    };

    for (const bool isBoxAgentMode: {false, true}) {
        TTest test(isBoxAgentMode);
        test.DoTest();
    }

}

Y_UNIT_TEST(TestRemoveTree) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));

            workload->set_id("other_workload");
            mutableWorkload->set_workload_ref("other_workload");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveWorkloadTarget("my_workload").TargetRemove_);
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("other_workload"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestEmptyCommandLine) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_stop_policy()->set_max_tries(1000);
            workload->mutable_destroy_policy()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_destroy_policy()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_destroy_policy()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
            workload->mutable_destroy_policy()->set_max_tries(1000);
            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "readiness container command_line is empty at workload 'my_workload'"
            );
            workload->mutable_readiness_check()->mutable_container()->set_command_line("echo readiness");

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "liveness container command_line is empty at workload 'my_workload'"
            );
            workload->mutable_liveness_check()->mutable_container()->set_command_line("echo liveness");

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
            , "stop container command_line is empty at workload 'my_workload'"
            );
            workload->mutable_stop_policy()->mutable_container()->set_command_line("echo stop");

           UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
            , "destroy container command_line is empty at workload 'my_workload'"
            );
            workload->mutable_destroy_policy()->mutable_container()->set_command_line("echo destroy");

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "start container command_line is empty at workload 'my_workload'"
            );
            workload->mutable_start()->set_command_line("echo start");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTargetState) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");
            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectTargetState("my_workload") == API::EWorkloadTarget_ACTIVE);

            mutableWorkload->set_target_state(API::EWorkloadTarget_REMOVED);

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectTargetState("my_workload") == API::EWorkloadTarget_REMOVED);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTarget) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");
            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, i + 1, i + 2));
                UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectTargetState("my_workload") == API::EWorkloadTarget_ACTIVE);
                UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").spec_timestamp() == i + 1);
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").revision() == i + 2);
            }

            // change workload
            workload->mutable_readiness_check()->mutable_container()->set_command_line("echo readiness");
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
            workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 4, 5));

            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectTargetState("my_workload") == API::EWorkloadTarget_ACTIVE);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").spec_timestamp() == 3);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").revision() == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddWithTargetCheck) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");

            Holder_->SetRevision(1);
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "hash", 0, 1));
            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "other_hash", 0, 4));

            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "other_hash", 0, 4)
                        , 0
                        , {}
                    }
                }
            );

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 5, 6));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").state() == API::EWorkloadState_REMOVED);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").spec_timestamp() == 0);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("my_workload").revision() == 0);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));
            TUpdateHolder::TWorkloadTarget workloadTarget = UpdateHolder_->GetAndRemoveWorkloadTarget("my_workload");
            UNIT_ASSERT(workloadTarget.Meta_.SpecTimestamp_ == 5);
            UNIT_ASSERT(workloadTarget.Meta_.Revision_ == 6);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithWorkloadTree) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    public:
        TTest() : ITestWorkloadTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepository_->GetObjectTargetState("my_workload")
                , API::EWorkloadTarget_ACTIVE
                , API::EWorkloadTargetState_Name(WorkloadStatusRepository_->GetObjectTargetState("my_workload"))
            );
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_SUSPENDED, spec.Getmutable_workloads(), {}, boxes, 1, 1));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepository_->GetObjectTargetState("my_workload")
                , API::EWorkloadTarget_REMOVED
                , API::EWorkloadTargetState_Name(WorkloadStatusRepository_->GetObjectTargetState("my_workload"))
            );
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_REMOVED, spec.Getmutable_workloads(), {}, boxes, 2, 2));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("my_workload"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveWorkloadTarget("my_workload").TargetRemove_);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSecrets) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");
            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            API::TPodAgentRequest podAgentRequest;

            auto* envVar = workload->add_env();
            envVar->set_name("TestSecret");
            envVar->mutable_value()->mutable_secret_env()->set_id("SecretKey");
            envVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");

            auto* secret = podAgentRequest.add_secrets();
            secret->set_id("SecretAlias");
            auto* value = secret->add_values();
            value->set_key("SecretKey");
            value->set_value("SomeValue");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), NSecret::GetSecretMap(podAgentRequest.secrets()), boxes, 0, 0));

            value->set_value("SomeOtherValue");
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), NSecret::GetSecretMap(podAgentRequest.secrets()), boxes, 0, 0));

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));

            value->set_value("U29tZU90aGVyVmFsdWU=");
            value->set_encoding("base64");

            auto workloadsToAdd = UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), NSecret::GetSecretMap(podAgentRequest.secrets()), boxes, 0, 0, /* autoDecodeBase64Secrets = */ true);
            auto envList = workloadsToAdd.Workloads_.at("my_workload").TreeReplaceMap_.at("START_ENVIRONMENT");
            UNIT_ASSERT_STRING_CONTAINS_C(envList, "TestSecret=SomeOtherValue", envList);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("my_workload"));

        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestTreeWithHttpAndTcp) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TBox* box = spec.add_boxes();
            TString boxId = "my_box";
            box->set_id(boxId);
            API::TWorkload* workload = spec.add_workloads();
            workload->set_id("my_workload");
            workload->set_box_ref(boxId);

            workload->mutable_start()->set_command_line("echo start");
            workload->mutable_readiness_check()->mutable_http_get()->set_port(20);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));

            API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
            mutableWorkload->set_workload_ref("my_workload");
            mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

            BoxStatusRepository_->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId));
            TBoxTreeGenerator::TBoxesToAdd boxes;
            boxes.Boxes_.insert(
                {
                    boxId
                    , {
                        NObjectTargetTestLib::CreateBoxTargetSimple(boxId, "box_tree_hash")
                        , 0
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));

            workload->mutable_readiness_check()->mutable_http_get()->set_port(0);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "port is zero"
            );

            workload->clear_readiness_check();
            workload->mutable_readiness_check()->mutable_tcp_check()->set_port(20);
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0));

            workload->mutable_readiness_check()->mutable_tcp_check()->set_port(0);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.workloads(), API::EPodAgentTargetState_ACTIVE, spec.Getmutable_workloads(), {}, boxes, 0, 0)
                , yexception
                , "port is zero"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestFillHookTypeReplaceMap) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            TMap<TString, TString> result;

            for (ui32 i = 0; i < 32; ++i) {
                if (i == 0 || (i & (i - 1)) == 0) {
                    // i is a power of 2 or equal 0
                    UNIT_ASSERT_NO_EXCEPTION(result = Worker_->FillHookTypeReplaceMap("test", (i & 1), (i & 2), (i & 4), (i & 8), (i & 16)));

                    if (i == 0) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "no_hook");
                    } else if (i == 1) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "container");
                    } else if (i == 2) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "http");
                    } else if (i == 4) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "tcp");
                    } else if (i == 8) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "unix_signal");
                    } else if (i == 16) {
                        UNIT_ASSERT(result.contains("TEST_HOOK_TYPE") && result["TEST_HOOK_TYPE"] == "no_hook");
                    } else {
                        UNIT_ASSERT_C(false, "This case must be unreachable");
                    }
                } else {
                    UNIT_ASSERT_EXCEPTION_CONTAINS(
                        Worker_->FillHookTypeReplaceMap("test", (i & 1), (i & 2), (i & 4), (i & 8), (i & 16))
                        , yexception
                        , "at least two of four hooks are set to true"
                    );
                }
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetWorkloadHookInfo) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            for (ui32 i = 0; i < 32; ++i) {
                if (i == 0 || (i & (i - 1)) == 0) {
                    // i is a power of 2 or equal 0
                    TWorkloadMeta::THookInfo result = Worker_->GetWorkloadHookInfo("test", (i & 1), (i & 2), (i & 4), (i & 8), (i & 16));

                    if (i == 0) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::TEmptyInfo>(result));
                    } else if (i == 1) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::TContainerInfo>(result));
                        UNIT_ASSERT_EQUAL(std::get<TWorkloadMeta::TContainerInfo>(result), TWorkloadMeta::TContainerInfo("test"));
                    } else if (i == 2) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::THttpGetInfo>(result));
                    } else if (i == 4) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::TTcpCheckInfo>(result));
                    } else if (i == 8) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::TUnixSignalInfo>(result));
                    } else if (i == 16) {
                        UNIT_ASSERT(std::holds_alternative<TWorkloadMeta::TStartContainerAliveCheckInfo>(result));
                    } else {
                        UNIT_ASSERT_C(false, "This case must be unreachable");
                    }
                } else {
                    UNIT_ASSERT_EXCEPTION_CONTAINS(
                        Worker_->GetWorkloadHookInfo("test", (i & 1), (i & 2), (i & 4), (i & 8), (i & 16))
                        , yexception
                        , "at least two of four hooks are set to true"
                    );
                }
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestFillHookTriesThresholdsReplaceMap) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {
            {
                TMap<TString, TString> result = Worker_->FillHookTriesThresholdsReplaceMap("readiness", 2, 3);
                UNIT_ASSERT_STRINGS_EQUAL("2", result["READINESS_FAILURE_THRESHOLD"]);
                UNIT_ASSERT_STRINGS_EQUAL("3", result["READINESS_SUCCESS_THRESHOLD"]);
            }

            {
                TMap<TString, TString> result = Worker_->FillHookTriesThresholdsReplaceMap("liveness", 0, 0);
                UNIT_ASSERT_STRINGS_EQUAL("1", result["LIVENESS_FAILURE_THRESHOLD"]);
                UNIT_ASSERT_STRINGS_EQUAL("1", result["LIVENESS_SUCCESS_THRESHOLD"]);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestFillUnixSignalReplaceMap) {
    class TTest : public ITestWorkloadTreeGeneratorCanon {
    protected:
        void Test() override {

            API::TUnixSignal unixSignal;

            {
                TMap<TString, TString> result = Worker_->FillUnixSignalReplaceMap(unixSignal, "test");
                UNIT_ASSERT_STRINGS_EQUAL(ToString(SIGTERM), result["UNIX_SIGNAL_TEST_SIGNAL"]);
                UNIT_ASSERT_STRINGS_EQUAL("5000", result["UNIX_SIGNAL_TEST_INITIAL_DELAY"]);
                UNIT_ASSERT_STRINGS_EQUAL("0", result["UNIX_SIGNAL_TEST_RESTART_PERIOD_SCALE"]);
                UNIT_ASSERT_STRINGS_EQUAL("0", result["UNIX_SIGNAL_TEST_RESTART_PERIOD_BACKOFF"]);
                UNIT_ASSERT_STRINGS_EQUAL("30000", result["UNIX_SIGNAL_TEST_MIN_RESTART_PERIOD"]);
                UNIT_ASSERT_STRINGS_EQUAL("18446744073709551615", result["UNIX_SIGNAL_TEST_MAX_RESTART_PERIOD"]);
            }

            {
                unixSignal.set_signal(API::EUnixSignalType_SIGUSR1);

                API::TTimeLimit& timeLimit = *unixSignal.mutable_time_limit();
                timeLimit.set_initial_delay_ms(1);
                timeLimit.set_restart_period_scale_ms(2);
                timeLimit.set_restart_period_back_off(3);
                timeLimit.set_min_restart_period_ms(4);
                timeLimit.set_max_restart_period_ms(5);

                TMap<TString, TString> result = Worker_->FillUnixSignalReplaceMap(unixSignal, "test");
                UNIT_ASSERT_STRINGS_EQUAL(ToString(SIGUSR1), result["UNIX_SIGNAL_TEST_SIGNAL"]);
                UNIT_ASSERT_STRINGS_EQUAL("1", result["UNIX_SIGNAL_TEST_INITIAL_DELAY"]);
                UNIT_ASSERT_STRINGS_EQUAL("2", result["UNIX_SIGNAL_TEST_RESTART_PERIOD_SCALE"]);
                UNIT_ASSERT_STRINGS_EQUAL("3", result["UNIX_SIGNAL_TEST_RESTART_PERIOD_BACKOFF"]);
                UNIT_ASSERT_STRINGS_EQUAL("4", result["UNIX_SIGNAL_TEST_MIN_RESTART_PERIOD"]);
                UNIT_ASSERT_STRINGS_EQUAL("5", result["UNIX_SIGNAL_TEST_MAX_RESTART_PERIOD"]);
            }
        }
    };

    TTest test;
    test.DoTest();
}

}

} // NInfra::NPodAgent::NTreeGeneratorTest
