#include "trees_update_job.h"

namespace NInfra::NPodAgent {

void TTreesUpdateJob::Run() {
    try {
        TGuard<TMutex> gMutex(UpdateSpecMutex_);

        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TTreesUpdateJobStart{});

        // Note: the order is important for optimize update
        // We need to update in both optimal orders

        // Optimal order for TargetRemove
        UpdateWorkloadTrees();
        UpdateBoxTrees();
        UpdateVolumeTrees();
        UpdateLayerTrees();
        UpdateStaticResourceTrees();

        // Optimal order for Target
        UpdateStaticResourceTrees();
        UpdateLayerTrees();
        UpdateVolumeTrees();
        UpdateBoxTrees();
        UpdateWorkloadTrees();

        SystemLogsStatisticsPrinter_->Print();

        LogFrame_->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TTreesUpdateJobFinish{});
    } catch (const yexception& e) {
        NLogEvent::TTreesUpdateJobException ev;
        ev.SetMessage(CurrentExceptionMessage());
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ev);
    }
}

void TTreesUpdateJob::UpdateLayerTrees() {
    const auto layerIds = StatusNTickerHolder_->GetLayerIds();

    for (const auto& layerId : layerIds) {
        try {
            TStatusNTickerHolder::TUpdateObjectResult objectResult = StatusNTickerHolder_->TryToUpdateLayerFromTarget(layerId);
            if (objectResult.Result_ != TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET) {
                NLogEvent::TTreesUpdateJobUpdateTree event;
                event.SetObjectType(NLogEvent::EPodAgentObjectType::LAYER);
                event.SetId(layerId);
                event.SetResult(ConvertUpdateResult(objectResult.Result_));
                event.SetMessage(objectResult.Message_);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, event);
            }
        } catch (const yexception& e) {
            NLogEvent::TTreesUpdateJobException event;
            event.SetMessage(CurrentExceptionMessage());
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, event);
        }
    }
}

void TTreesUpdateJob::UpdateStaticResourceTrees() {
    const auto staticResourceIds = StatusNTickerHolder_->GetStaticResourceIds();

    for (const auto& staticResourceId : staticResourceIds) {
        try {
            TStatusNTickerHolder::TUpdateObjectResult objectResult = StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget(staticResourceId);
            if (objectResult.Result_ != TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET) {
                NLogEvent::TTreesUpdateJobUpdateTree event;
                event.SetObjectType(NLogEvent::EPodAgentObjectType::STATIC_RESOURCE);
                event.SetId(staticResourceId);
                event.SetResult(ConvertUpdateResult(objectResult.Result_));
                event.SetMessage(objectResult.Message_);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, event);
            }
        } catch (const yexception& e) {
            NLogEvent::TTreesUpdateJobException event;
            event.SetMessage(CurrentExceptionMessage());
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, event);
        }
    }
}

void TTreesUpdateJob::UpdateVolumeTrees() {
    const auto volumeIds = StatusNTickerHolder_->GetVolumeIds();

    for (const auto& volumeId : volumeIds) {
        try {
            TStatusNTickerHolder::TUpdateObjectResult objectResult = StatusNTickerHolder_->TryToUpdateVolumeFromTarget(volumeId);
            if (objectResult.Result_ != TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET) {
                NLogEvent::TTreesUpdateJobUpdateTree event;
                event.SetObjectType(NLogEvent::EPodAgentObjectType::VOLUME);
                event.SetId(volumeId);
                event.SetResult(ConvertUpdateResult(objectResult.Result_));
                event.SetMessage(objectResult.Message_);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, event);
            }
        } catch (const yexception& e) {
            NLogEvent::TTreesUpdateJobException event;
            event.SetMessage(CurrentExceptionMessage());
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, event);
        }
    }
}

void TTreesUpdateJob::UpdateBoxTrees() {
    const auto boxIds = StatusNTickerHolder_->GetBoxIds();

    for (const auto& boxId : boxIds) {
        try {
            TStatusNTickerHolder::TUpdateObjectResult objectResult = StatusNTickerHolder_->TryToUpdateBoxFromTarget(boxId);
            if (objectResult.Result_ != TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET) {
                NLogEvent::TTreesUpdateJobUpdateTree event;
                event.SetObjectType(NLogEvent::EPodAgentObjectType::BOX);
                event.SetId(boxId);
                event.SetResult(ConvertUpdateResult(objectResult.Result_));
                event.SetMessage(objectResult.Message_);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, event);
            }
        } catch (const yexception& e) {
            NLogEvent::TTreesUpdateJobException event;
            event.SetMessage(CurrentExceptionMessage());
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, event);
        }
    }
}

void TTreesUpdateJob::UpdateWorkloadTrees() {
    const auto workloadIds = StatusNTickerHolder_->GetWorkloadIds();

    for (const auto& workloadId : workloadIds) {
        try {
            TStatusNTickerHolder::TUpdateObjectResult objectResult = StatusNTickerHolder_->TryToUpdateWorkloadFromTarget(workloadId);
            if (objectResult.Result_ != TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET) {
                NLogEvent::TTreesUpdateJobUpdateTree event;
                event.SetObjectType(NLogEvent::EPodAgentObjectType::WORKLOAD);
                event.SetId(workloadId);
                event.SetResult(ConvertUpdateResult(objectResult.Result_));
                event.SetMessage(objectResult.Message_);
                LogFrame_->LogEvent(ELogPriority::TLOG_INFO, event);
            }
        } catch (const yexception& e) {
            NLogEvent::TTreesUpdateJobException event;
            event.SetMessage(CurrentExceptionMessage());
            LogFrame_->LogEvent(ELogPriority::TLOG_ERR, event);
        }
    }
}

NLogEvent::TTreesUpdateJobUpdateTree::EUpdateResult TTreesUpdateJob::ConvertUpdateResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult updateResult) {
    switch (updateResult) {
        case TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET:
            return NLogEvent::TTreesUpdateJobUpdateTree::EUpdateResult_NO_TARGET;
        case TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED:
            return NLogEvent::TTreesUpdateJobUpdateTree::EUpdateResult_REMOVED;
        case TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED:
            return NLogEvent::TTreesUpdateJobUpdateTree::EUpdateResult_UPDATED;
        case TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE:
            return NLogEvent::TTreesUpdateJobUpdateTree::EUpdateResult_WAITING_UPDATE;
    }

    ythrow yexception() << "Unknown update result '" << ToString(updateResult) << "'";
}

} // namespace NInfra::NPodAgent
