#include "trees_update_job.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/period_job_worker/period_job_worker.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_sender.h>
#include <infra/pod_agent/libs/system_logs_sender/statistics/system_logs_statistics_printer_mock.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTreesUpdateJobTest  {

static TLogger logger({});

static constexpr TDuration DEFAULT_PERIOD = TDuration::MilliSeconds(50);

class ITestTreesUpdateJobCanon {
public:
    ITestTreesUpdateJobCanon()
        : Ticker_(new TMtpPeriodTicker(TBehaviorTickerConfig{}, logger.SpawnFrame(), logger.SpawnFrame()))
        , UpdateHolder_(new TUpdateHolder())
        , BoxStatusRepository_(new TBoxStatusRepository())
        , LayerStatusRepository_(new TLayerStatusRepository())
        , StaticResourceStatusRepository_(new TStaticResourceStatusRepository())
        , VolumeStatusRepository_(new TVolumeStatusRepository())
        , WorkloadStatusRepository_(new TWorkloadStatusRepository())
        , WorkloadStatusRepositoryInternal_(new TWorkloadStatusRepositoryInternal())
    {
        StatusRepository_ = new TStatusRepository(
            UpdateHolder_
            , BoxStatusRepository_
            , LayerStatusRepository_
            , StaticResourceStatusRepository_
            , VolumeStatusRepository_
            , WorkloadStatusRepository_
        );
        StatusNTickerHolder_ = new TStatusNTickerHolder(
            Ticker_
            , StatusRepository_
            , WorkloadStatusRepositoryInternal_
            , new TMockSystemLogsSender()
            , new TMockSystemLogsSender()
            , /* isBoxAgentMode = */ false
        );
    }

    virtual ~ITestTreesUpdateJobCanon() = default;

    TTreesUpdateJobPtr GetJob() {
        return new TTreesUpdateJob(
            DEFAULT_PERIOD
            , logger.SpawnFrame()
            , logger.SpawnFrame()
            , StatusNTickerHolder_
            , new TSystemLogsStatisticsPrinterMock()
            , UpdateSpecMutex_
        );
    }

    void RunJob() {
        StatusNTickerHolder_->StartTicker();
        GetJob()->Run();
        StatusNTickerHolder_->StopTicker();
    }

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    TMtpPeriodTickerPtr Ticker_;
    TUpdateHolderPtr UpdateHolder_;

    TBoxStatusRepositoryPtr BoxStatusRepository_;
    TLayerStatusRepositoryPtr LayerStatusRepository_;
    TStaticResourceStatusRepositoryPtr StaticResourceStatusRepository_;
    TVolumeStatusRepositoryPtr VolumeStatusRepository_;
    TWorkloadStatusRepositoryPtr WorkloadStatusRepository_;

    TWorkloadStatusRepositoryInternalPtr WorkloadStatusRepositoryInternal_;

    TStatusRepositoryPtr StatusRepository_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;

    TMutex UpdateSpecMutex_;
};

Y_UNIT_TEST_SUITE(TreesUpdateJobSuite) {

Y_UNIT_TEST(TestNoRun) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            TTreesUpdateJobPtr job = GetJob();
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestNoChange) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")) == "static_resource1_download_hash");
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")) == "layer1_download_hash");
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "hash");
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "hash");
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "hash");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeLayer) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            TUpdateHolder::TLayerTarget newTarget = NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other", 3);
            StatusNTickerHolder_->UpdateLayerTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            RunJob();

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            TUpdateHolder::TLayerTarget layerTarget = UpdateHolder_->GetAndRemoveLayerTarget("layer1");
            UNIT_ASSERT(layerTarget == newTarget);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")) == "layer1_download_hash");

            StatusNTickerHolder_->UpdateLayerTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash_other")) == "layer1_download_hash_other");
            UNIT_ASSERT(!LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("layer1_download_hash_other"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeStaticResource) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            TUpdateHolder::TStaticResourceTarget newTarget = NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other", 3);
            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            RunJob();

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            TUpdateHolder::TStaticResourceTarget staticResourceTarget = UpdateHolder_->GetAndRemoveStaticResourceTarget("static_resource1");
            UNIT_ASSERT(staticResourceTarget == newTarget);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")) == "static_resource1_download_hash");

            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash_other")) == "static_resource1_download_hash_other");
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash_other"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectCheckPeriodMs("static_resource1") == 5);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeVolume) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {}, {"volume1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            TUpdateHolder::TVolumeTarget newTarget = NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer2"});
            StatusNTickerHolder_->UpdateVolumeTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            RunJob();

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            TUpdateHolder::TVolumeTarget volumeTarget = UpdateHolder_->GetAndRemoveVolumeTarget("volume1");
            UNIT_ASSERT(volumeTarget == newTarget);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "hash");

            StatusNTickerHolder_->UpdateVolumeTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "other_hash");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeBox) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume2", "hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}, {"volume1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            TUpdateHolder::TBoxTarget newTarget = NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "other_hash", {"layer2"}, {"static_resource2"}, {"volume2"});
            StatusNTickerHolder_->UpdateBoxTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            RunJob();

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            TUpdateHolder::TBoxTarget boxTarget = UpdateHolder_->GetAndRemoveBoxTarget("box1");
            UNIT_ASSERT(boxTarget == newTarget);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "hash");

            StatusNTickerHolder_->UpdateBoxTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "other_hash");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeWorkload) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef_other"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "boxRef"));

            TUpdateHolder::TWorkloadTarget newTarget = NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "boxRef_other");
            StatusNTickerHolder_->UpdateWorkloadTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            RunJob();

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            TUpdateHolder::TWorkloadTarget workloadTarget = UpdateHolder_->GetAndRemoveWorkloadTarget("workload1");
            UNIT_ASSERT(workloadTarget == newTarget);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "hash");

            StatusNTickerHolder_->UpdateWorkloadTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            RunJob();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "other_hash");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestWithPeriodJobWorker) {
    class TTest : public ITestTreesUpdateJobCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef_other"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "boxRef"));

            TUpdateHolder::TWorkloadTarget newTarget = NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "boxRef_other");
            StatusNTickerHolder_->UpdateWorkloadTarget(newTarget);
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);

            TTreesUpdateJobPtr job = GetJob();

            StatusNTickerHolder_->StartTicker();
            TPeriodJobWorker periodJobWorker(TDuration::Seconds(5), logger.SpawnFrame());
            periodJobWorker.AddJob(job);

            periodJobWorker.Start();
            Sleep(TDuration::MilliSeconds(350));
            periodJobWorker.Stop();

            StatusNTickerHolder_->StopTicker();

            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "other_hash");
        }
    };

    TTest test;
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreesUpdateJobTest
