#include "unistat_object_helper.h"

#include <util/string/cast.h>
#include <util/string/builder.h>

namespace NInfra::NPodAgent {

void TUnistatObjectHelper::AddCacheObject(
    const TObjectMeta& objectMeta
) {
    TWriteGuardBase<TLightRWLock> guard(SignalLock_);

    TString objectId;
    ui32 revision;
    NStatusRepositoryTypes::EObjectType objectType;

    if (std::holds_alternative<TLayerMeta>(objectMeta)) {
        objectId = std::get<TLayerMeta>(objectMeta).Id_;
        revision = std::get<TLayerMeta>(objectMeta).Revision_;
        objectType = NStatusRepositoryTypes::EObjectType::LAYER;
    } else if (std::holds_alternative<TStaticResourceMeta>(objectMeta)) {
        objectId = std::get<TStaticResourceMeta>(objectMeta).Id_;
        revision = std::get<TStaticResourceMeta>(objectMeta).Revision_;
        objectType = NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE;
    } else if (
        std::holds_alternative<TBoxMeta>(objectMeta)
        || std::holds_alternative<TVolumeMeta>(objectMeta)
        || std::holds_alternative<TWorkloadMeta>(objectMeta)
    ) {
        ythrow yexception() << "Add cache object not implemented for: " << objectMeta.index();
    } else {
        ythrow yexception() << "Cannot parse object meta: " << objectMeta.index();
    }

    AddAllCacheObjectConditionSignals(
        objectId
        , revision
        , objectType
    );
}

void TUnistatObjectHelper::RemoveCacheObject(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
) {
    TWriteGuardBase<TLightRWLock> guard(SignalLock_);

    switch (objectType) {
        case NStatusRepositoryTypes::EObjectType::LAYER:
            break;
        case NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE:
            break;
        case NStatusRepositoryTypes::EObjectType::BOX:
        case NStatusRepositoryTypes::EObjectType::VOLUME:
        case NStatusRepositoryTypes::EObjectType::WORKLOAD:
            ythrow yexception() << "Add cache object not implemented for: " << ToString(objectType);
    }

    RemoveAllCacheObjectConditionSignals(
        objectId
        , revision
        , objectType
    );
}

void TUnistatObjectHelper::AddObject(
    const TObjectMeta& objectMeta
) {
    TWriteGuardBase<TLightRWLock> guard(SignalLock_);

    TString objectId;
    NStatusRepositoryTypes::EObjectType objectType;

    if (std::holds_alternative<TBoxMeta>(objectMeta)) {
        objectId = std::get<TBoxMeta>(objectMeta).Id_;
        objectType = NStatusRepositoryTypes::EObjectType::BOX;

        AddAllContainerSignals(
            objectId
            , objectType
            , BOX_CONTAINER_TYPES
            , std::get<TBoxMeta>(objectMeta).InitContainers_.size()
        );
    } else if (std::holds_alternative<TLayerMeta>(objectMeta)) {
        objectId = std::get<TLayerMeta>(objectMeta).Id_;
        objectType = NStatusRepositoryTypes::EObjectType::LAYER;
    } else if (std::holds_alternative<TStaticResourceMeta>(objectMeta)) {
        objectId = std::get<TStaticResourceMeta>(objectMeta).Id_;
        objectType = NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE;
    } else if (std::holds_alternative<TVolumeMeta>(objectMeta)) {
        objectId = std::get<TVolumeMeta>(objectMeta).Id_;
        objectType = NStatusRepositoryTypes::EObjectType::VOLUME;
    } else if (std::holds_alternative<TWorkloadMeta>(objectMeta)) {
        objectId = std::get<TWorkloadMeta>(objectMeta).Id_;
        objectType = NStatusRepositoryTypes::EObjectType::WORKLOAD;

        AddAllContainerSignals(
            objectId
            , objectType
            , ExtractWorkloadContainerTypes(std::get<TWorkloadMeta>(objectMeta))
            , std::get<TWorkloadMeta>(objectMeta).InitContainers_.size()
        );

        AddAllNetworkSignals(
            objectId
            , NStatusRepositoryTypes::EHookBackend::HTTP
            , ExtractWorkloadHttpHookTypes(std::get<TWorkloadMeta>(objectMeta))
            , HTTP_SIGNAL_TYPES
        );

        AddAllNetworkSignals(
            objectId
            , NStatusRepositoryTypes::EHookBackend::TCP
            , ExtractWorkloadTcpHookTypes(std::get<TWorkloadMeta>(objectMeta))
            , TCP_SIGNAL_TYPES
        );
    } else {
        ythrow yexception() << "Cannot parse object meta: " << objectMeta.index();
    }

    AddAllObjectConditionSignals(
        objectId
        , objectType
    );
}

void TUnistatObjectHelper::RemoveObject(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
) {
    TWriteGuardBase<TLightRWLock> guard(SignalLock_);

    switch (objectType) {
        case NStatusRepositoryTypes::EObjectType::BOX:
            RemoveAllContainerSignals(
                objectId
                , objectType
                , BOX_CONTAINER_TYPES
            );
            break;
        case NStatusRepositoryTypes::EObjectType::LAYER:
            break;
        case NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE:
            break;
        case NStatusRepositoryTypes::EObjectType::VOLUME:
            break;
        case NStatusRepositoryTypes::EObjectType::WORKLOAD:
            RemoveAllContainerSignals(
                objectId
                , objectType
                , WORKLOAD_CONTAINER_TYPES
            );

            RemoveAllNetworkSignals(
                objectId
                , NStatusRepositoryTypes::EHookBackend::HTTP
                , HTTP_HOOK_TYPES
                , HTTP_SIGNAL_TYPES
            );

            RemoveAllNetworkSignals(
                objectId
                , NStatusRepositoryTypes::EHookBackend::TCP
                , TCP_HOOK_TYPES
                , TCP_SIGNAL_TYPES
            );
            break;
    }

    RemoveAllObjectConditionSignals(
        objectId
        , objectType
    );
}

bool TUnistatObjectHelper::PushPodConditionSignal(
    EConditionSignalType signalType
    , EConditionStatus status
) {
    TReadGuardBase<TLightRWLock> guard(SignalLock_);
    return PushConditionSignal(
        // All pod condition are INFRA signals
        TMultiUnistat::ESignalNamespace::INFRA
        , GetPodConditionSignalName(
            signalType
        )
        , status
    );
}

bool TUnistatObjectHelper::PushCacheObjectConditionSignal(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
    , EConditionSignalType signalType
    , EConditionStatus status
) {
    TReadGuardBase<TLightRWLock> guard(SignalLock_);
    return PushConditionSignal(
        TMultiUnistat::ESignalNamespace::USER
        , GetCacheObjectConditionSignalName(
            objectId
            , revision
            , objectType
            , signalType
        )
        , status
    );
}

bool TUnistatObjectHelper::PushObjectConditionSignal(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
    , EConditionSignalType signalType
    , EConditionStatus status
) {
    TReadGuardBase<TLightRWLock> guard(SignalLock_);
    return PushConditionSignal(
        TMultiUnistat::ESignalNamespace::USER
        , GetObjectConditionSignalName(
            objectId
            , objectType
            , signalType
        )
        , status
    );
}

bool TUnistatObjectHelper::PushContainerSignal(
    const NStatusRepositoryTypes::TContainerDescription& container
    , EContainerSignalType signalType
    , ui32 value
) {
    TReadGuardBase<TLightRWLock> guard(SignalLock_);
    bool result = true;
    result &= TMultiUnistat::Instance().PushSignalUnsafe(
        TMultiUnistat::ESignalNamespace::USER
        , GetContainerSignalName(container, signalType)
        , value
    );

    return result;
}

bool TUnistatObjectHelper::PushNetworkSignal(
    const TString& objectId
    , NStatusRepositoryTypes::EHookBackend backend
    , NStatusRepositoryTypes::ENetworkHookType hookType
    , TUnistatObjectHelper::ENetworkSignalType signalType
    , ui32 value
) {
    TReadGuardBase<TLightRWLock> guard(SignalLock_);
    return TMultiUnistat::Instance().PushSignalUnsafe(
        TMultiUnistat::ESignalNamespace::USER
        , GetNetworkSignalName(
            objectId
            , backend
            , hookType
            , signalType
        )
        , value
    );
}

void TUnistatObjectHelper::AddFloatHoles(
    const TMultiUnistat::ESignalNamespace signalNamespace
    , const TVector<TString>& names
    , const TString& suffix
    , NUnistat::TPriority priority
    , NUnistat::TStartValue startValue
    , EAggregationType type
    , bool alwaysVisible
) {
    for (const TString& signal : names) {
        TMultiUnistat::Instance().DrillFloatHole(
            signalNamespace
            , signal
            , suffix
            , priority
            , startValue
            , type
            , alwaysVisible
        );
    }
}

void TUnistatObjectHelper::AddHistogramHoles(
    const TMultiUnistat::ESignalNamespace signalNamespace
    , const TVector<TString>& names
    , const TString& suffix
    , NUnistat::TPriority priority
    , const NUnistat::TIntervals& intervals
    , EAggregationType type
    , bool alwaysVisible
) {
    for (const TString& signal : names) {
        TMultiUnistat::Instance().DrillHistogramHole(
            signalNamespace
            , signal
            , suffix
            , priority
            , intervals
            , type
            , alwaysVisible
        );
    }
}

void TUnistatObjectHelper::EraseHoles(
    const TMultiUnistat::ESignalNamespace signalNamespace
    , const TVector<TString>& names
) {
    for (const TString& signal : names) {
        TMultiUnistat::Instance().EraseHole(
            signalNamespace
            , signal
        );
    }
}

bool TUnistatObjectHelper::PushConditionSignal(
    const TMultiUnistat::ESignalNamespace signalNamespace
    , const TString& signal
    , EConditionStatus status
) {
    TMultiUnistat::Instance().ResetSignalUnsafe(
        signalNamespace
        , signal
    );

    return TMultiUnistat::Instance().PushSignalUnsafe(
        signalNamespace
        , signal
        , double(status)
    );
}

void TUnistatObjectHelper::AddConditionSignals(
    const TMultiUnistat::ESignalNamespace signalNamespace
    , const TVector<TString>& names
    , NUnistat::TPriority priority
) {
    AddHistogramHoles(
        signalNamespace
        , names
        , "ahhh"
        , priority
        , CONDITION_SIGNAL_INTERVALS
        , EAggregationType::HostHistogram
        , false
    );

    for (const TString& signal : names) {
        PushConditionSignal(
            signalNamespace
            , signal
            , EConditionStatus::UNKNOWN
        );
    }
}

TString TUnistatObjectHelper::GetPodConditionSignalName(
    EConditionSignalType signalType
) const {
    return TStringBuilder()
        << CONDITION_POD_SIGNAL_PREFIX
        << ToString(signalType)
    ;
}

TVector<TString> TUnistatObjectHelper::GenerateAllPodConditionSignals() {
    TVector<TString> result;

    for (const auto& signalType : CONDITION_SIGNAL_TYPES) {
        result.emplace_back(
            GetPodConditionSignalName(
                signalType
            )
        );
    }

    return result;
}

void TUnistatObjectHelper::AddAllPodConditionSignals() {
    AddConditionSignals(
        // All pod condition are INFRA signals
        TMultiUnistat::ESignalNamespace::INFRA
        , GenerateAllPodConditionSignals()
        // Pod conditions have INFRA priority
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
    );
}

TString TUnistatObjectHelper::GetCacheObjectConditionSignalName(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
    , EConditionSignalType signalType
) const {
    return TStringBuilder()
        << CONDITION_CACHE_OBJECT_SIGNAL_PREFIX
        << ToString(objectType) << "_"
        << objectId << "_"
        << revision << "_"
        << ToString(signalType)
    ;
}

TVector<TString> TUnistatObjectHelper::GenerateAllCacheObjectConditionSignals(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
) const {
    TVector<TString> result;

    for (const auto& signalType : CONDITION_SIGNAL_TYPES) {
        result.emplace_back(
            GetCacheObjectConditionSignalName(
                objectId
                , revision
                , objectType
                , signalType
            )
        );
    }

    return result;
}

void TUnistatObjectHelper::AddAllCacheObjectConditionSignals(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
) {
    AddConditionSignals(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllCacheObjectConditionSignals(
            objectId
            , revision
            , objectType
        )
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::USER_INFO)
    );
}

void TUnistatObjectHelper::RemoveAllCacheObjectConditionSignals(
    const TString& objectId
    , ui32 revision
    , NStatusRepositoryTypes::EObjectType objectType
) {
    EraseHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllCacheObjectConditionSignals(
            objectId
            , revision
            , objectType
        )
    );
}

TString TUnistatObjectHelper::GetObjectConditionSignalName(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
    , EConditionSignalType signalType
) const {
    return TStringBuilder()
        << CONDITION_OBJECT_SIGNAL_PREFIX
        << ToString(objectType) << "_"
        << objectId << "_"
        << ToString(signalType)
    ;
}

TVector<TString> TUnistatObjectHelper::GenerateAllObjectConditionSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
) const {
    TVector<TString> result;

    for (const auto& signalType : CONDITION_SIGNAL_TYPES) {
        result.emplace_back(
            GetObjectConditionSignalName(
                objectId
                , objectType
                , signalType
            )
        );
    }

    return result;
}

void TUnistatObjectHelper::AddAllObjectConditionSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
) {
    AddConditionSignals(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllObjectConditionSignals(
            objectId
            , objectType
        )
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::USER_INFO)
    );
}

void TUnistatObjectHelper::RemoveAllObjectConditionSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
) {
    EraseHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllObjectConditionSignals(
            objectId
            , objectType
        )
    );
}

TString TUnistatObjectHelper::GetContainerSignalName(
    const NStatusRepositoryTypes::TContainerDescription& container
    , EContainerSignalType signalType
) const {
    return TStringBuilder()
        << CONTAINER_SIGNAL_PREFIX
        << ToString(container.ObjectType_) << "_"
        << container.ObjectIdOrHash_ << "_"
        << ToString(container.ContainerType_) << "_"
        << (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT ? ToString(container.InitNum_) + "_" : "")
        << ToString(signalType)
    ;
}

TVector<TString> TUnistatObjectHelper::GenerateAllContainerSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
    , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
    , ui32 initSize
) const {
    TVector<TString> result;

    for (const auto& containerType : containerTypes) {
        for (const auto& signalType : CONTAINER_SIGNAL_TYPES) {
            result.emplace_back(
                GetContainerSignalName(
                    NStatusRepositoryTypes::TContainerDescription(
                        objectId
                        , objectType
                        , containerType
                    )
                    , signalType
                )
            );
        }
    }

    for (ui32 initNum = 0; initNum < initSize; ++initNum) {
        for (const auto& signalType : CONTAINER_SIGNAL_TYPES) {
            result.emplace_back(
                GetContainerSignalName(
                    NStatusRepositoryTypes::TContainerDescription(
                        objectId
                        , objectType
                        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
                        , initNum
                    )
                    , signalType
                )
            );
        }
    }

    return result;
}

void TUnistatObjectHelper::AddAllContainerSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
    , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
    , ui32 initSize
) {
    AddFloatHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllContainerSignals(
            objectId
            , objectType
            , containerTypes
            , initSize
        )
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::USER_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
        , false
    );
}

void TUnistatObjectHelper::RemoveAllContainerSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EObjectType objectType
    , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
) {
    EraseHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllContainerSignals(
            objectId
            , objectType
            , containerTypes
            , 0 // INIT is special
        )
    );

    // INIT is special
    for (ui32 initNum = 0; true; ++initNum) {
        bool hasSignal = false;

        for (const auto& signalType : CONTAINER_SIGNAL_TYPES) {
            hasSignal |= TMultiUnistat::Instance().EraseHole(
                TMultiUnistat::ESignalNamespace::USER
                , GetContainerSignalName(
                    NStatusRepositoryTypes::TContainerDescription(
                        objectId
                        , objectType
                        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
                        , initNum
                    )
                    , signalType
                )
            );
        }

        if (!hasSignal) {
            break;
        }
    }
}

TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> TUnistatObjectHelper::ExtractWorkloadContainerTypes(
    const TWorkloadMeta& objectMeta
) {
    TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> result;

    result.push_back(NStatusRepositoryTypes::TContainerDescription::EContainerType::START);
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Readiness_)) {
        result.push_back(NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Liveness_)) {
        result.push_back(NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Stop_)) {
        result.push_back(NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Destroy_)) {
        result.push_back(NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY);
    }

    return result;
}

TString TUnistatObjectHelper::GetNetworkSignalName(
    const TString& objectId
    , NStatusRepositoryTypes::EHookBackend backend
    , NStatusRepositoryTypes::ENetworkHookType hookType
    , TUnistatObjectHelper::ENetworkSignalType signalType
) const {
    return TStringBuilder()
        << NETWORK_SIGNAL_PREFIX
        << ToString(backend) << "_"
        << objectId << "_"
        << ToString(hookType) << "_"
        << ToString(signalType)
    ;
}

TVector<TString> TUnistatObjectHelper::GenerateAllNetworkSignals(
    const TString& objectId
    , const NStatusRepositoryTypes::EHookBackend backend
    , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
    , const TVector<ENetworkSignalType>& signalTypes
) const {
    TVector<TString> result;

    for (const auto& hookType : networkHookTypes) {
        for (const auto& signalType : signalTypes) {
            result.emplace_back(
                GetNetworkSignalName(
                    objectId
                    , backend
                    , hookType
                    , signalType
                )
            );
        }
    }
    return result;
}

void TUnistatObjectHelper::AddAllNetworkSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EHookBackend backend
    , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
    , const TVector<TUnistatObjectHelper::ENetworkSignalType>& signalTypes
) {
    AddFloatHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllNetworkSignals(
            objectId
            , backend
            , networkHookTypes
            , signalTypes
        )
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::USER_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
        , false
    );
}

void TUnistatObjectHelper::RemoveAllNetworkSignals(
    const TString& objectId
    , NStatusRepositoryTypes::EHookBackend backend
    , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
    , const TVector<ENetworkSignalType>& signalTypes
) {
    EraseHoles(
        TMultiUnistat::ESignalNamespace::USER
        , GenerateAllNetworkSignals(
             objectId
             , backend
             , networkHookTypes
             , signalTypes
        )
    );
}

TVector<NStatusRepositoryTypes::ENetworkHookType> TUnistatObjectHelper::ExtractWorkloadHttpHookTypes(
    const TWorkloadMeta& objectMeta
) {
    TVector<NStatusRepositoryTypes::ENetworkHookType> result;

    if (std::holds_alternative<TWorkloadMeta::THttpGetInfo>(objectMeta.Readiness_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::READINESS);
    }
    if (std::holds_alternative<TWorkloadMeta::THttpGetInfo>(objectMeta.Liveness_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::LIVENESS);
    }
    if (std::holds_alternative<TWorkloadMeta::THttpGetInfo>(objectMeta.Stop_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::STOP);
    }
    if (std::holds_alternative<TWorkloadMeta::THttpGetInfo>(objectMeta.Destroy_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::DESTROY);
    }

    return result;
}

TVector<NStatusRepositoryTypes::ENetworkHookType> TUnistatObjectHelper::ExtractWorkloadTcpHookTypes(
    const TWorkloadMeta& objectMeta
) {
    TVector<NStatusRepositoryTypes::ENetworkHookType> result;

    if (std::holds_alternative<TWorkloadMeta::TTcpCheckInfo>(objectMeta.Readiness_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::READINESS);
    }
    if (std::holds_alternative<TWorkloadMeta::TTcpCheckInfo>(objectMeta.Liveness_)) {
        result.push_back(NStatusRepositoryTypes::ENetworkHookType::LIVENESS);
    }

    return result;
}

} // namespace NInfra::NPodAgent
