#pragma once

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/types/status_repository_types.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/variant.h>

namespace NInfra::NPodAgent {

/*
    Thread-safe signal holder
*/
class TUnistatObjectHelper: public TAtomicRefCount<TUnistatObjectHelper> {
public:
    // Condition
    enum class EConditionSignalType {
        READY                   /* "ready" */,
        IN_PROGRESS             /* "in_progress" */,
        FAILED                  /* "failed" */,
    };
    enum class EConditionStatus {
        UNKNOWN = 0             /* "unknown" */,
        TRUE    = 1             /* "true" */,
        FALSE   = 2             /* "false" */,
    };

    // Container
    enum class EContainerSignalType {
        ZERO_RETURN_CODE        /* "zero_return_code" */,
        POSITIVE_RETURN_CODE    /* "positive_return_code" */,
        OOM                     /* "oom" */,
        TIMEOUT                 /* "timeout" */,
        KILLED_EXTERNALLY       /* "killed_externally" */,
        SYSTEM_FAILURE          /* "system_failure" */,
        EXITED                  /* "exited" */,
    };

    // Network
    enum class ENetworkSignalType {
        SUCCESS                 /* "success" */,
        ERROR                   /* "error" */,
        TIMEOUT                 /* "timeout" */,
        WRONG_ANSWER            /* "wrong_answer" */,
    };

    using TObjectMeta = std::variant<
        TBoxMeta
        , TLayerMeta
        , TStaticResourceMeta
        , TVolumeMeta
        , TWorkloadMeta
    >;

public:
    TUnistatObjectHelper()
    {
        AddAllPodConditionSignals();
    }

    static TUnistatObjectHelper& Instance() {
        return *Singleton<TUnistatObjectHelper>();
    };

    void AddCacheObject(
        const TObjectMeta& objectMeta
    );
    void RemoveCacheObject(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
    );

    void AddObject(
        const TObjectMeta& objectMeta
    );
    void RemoveObject(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
    );

    // Return true on success push
    bool PushPodConditionSignal(
        EConditionSignalType signalType
        , EConditionStatus status
    );
    bool PushCacheObjectConditionSignal(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
        , EConditionSignalType signalType
        , EConditionStatus status
    );
    bool PushObjectConditionSignal(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
        , EConditionSignalType signalType
        , EConditionStatus status
    );
    bool PushContainerSignal(
        const NStatusRepositoryTypes::TContainerDescription& container
        , EContainerSignalType signalType
        , ui32 value
    );

    // Network
    bool PushNetworkSignal(
        const TString& objectId
        , NStatusRepositoryTypes::EHookBackend backend
        , NStatusRepositoryTypes::ENetworkHookType hookType
        , ENetworkSignalType signalType
        , ui32 value
    );

private:
    void AddFloatHoles(
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TVector<TString>& names
        , const TString& suffix
        , NUnistat::TPriority priority
        , NUnistat::TStartValue startValue
        , EAggregationType type
        , bool alwaysVisible
    );

    void AddHistogramHoles(
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TVector<TString>& names
        , const TString& suffix
        , NUnistat::TPriority priority
        , const NUnistat::TIntervals& intervals
        , EAggregationType type
        , bool alwaysVisible
    );

    void EraseHoles(
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TVector<TString>& names
    );

    // Condition

    // Return true on success push
    bool PushConditionSignal(
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TString& signal
        , EConditionStatus status
    );

    void AddConditionSignals(
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TVector<TString>& names
        , NUnistat::TPriority priority
    );

    // Pod condition
    TString GetPodConditionSignalName(
        EConditionSignalType signalType
    ) const;
    TVector<TString> GenerateAllPodConditionSignals();
    void AddAllPodConditionSignals();

    // CacheObject condition
    TString GetCacheObjectConditionSignalName(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
        , EConditionSignalType signalType
    ) const;
    TVector<TString> GenerateAllCacheObjectConditionSignals(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
    ) const;
    void AddAllCacheObjectConditionSignals(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
    );
    void RemoveAllCacheObjectConditionSignals(
        const TString& objectId
        , ui32 revision
        , NStatusRepositoryTypes::EObjectType objectType
    );

    // Object condition
    TString GetObjectConditionSignalName(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
        , EConditionSignalType signalType
    ) const;
    TVector<TString> GenerateAllObjectConditionSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
    ) const;
    void AddAllObjectConditionSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
    );
    void RemoveAllObjectConditionSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
    );

    // Container
    TString GetContainerSignalName(
        const NStatusRepositoryTypes::TContainerDescription& container
        , EContainerSignalType signalType
    ) const;
    TVector<TString> GenerateAllContainerSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
        , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
        , ui32 initSize
    ) const;
    void AddAllContainerSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
        , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
        , ui32 initSize
    );
    void RemoveAllContainerSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EObjectType objectType
        , const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType>& containerTypes
    );
    TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> ExtractWorkloadContainerTypes(
        const TWorkloadMeta& objectMeta
    );

    // Network
    TString GetNetworkSignalName(
        const TString& objectId
        , NStatusRepositoryTypes::EHookBackend backend
        , NStatusRepositoryTypes::ENetworkHookType hookType
        , ENetworkSignalType signalType
    ) const;
    TVector<TString> GenerateAllNetworkSignals(
        const TString& objectId
        , const NStatusRepositoryTypes::EHookBackend backend
        , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
        , const TVector<ENetworkSignalType>& signalTypes
    ) const;
    void AddAllNetworkSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EHookBackend backend
        , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
        , const TVector<ENetworkSignalType>& signalTypes
    );
    void RemoveAllNetworkSignals(
        const TString& objectId
        , NStatusRepositoryTypes::EHookBackend backend
        , const TVector<NStatusRepositoryTypes::ENetworkHookType>& networkHookTypes
        , const TVector<ENetworkSignalType>& signalTypes
    );
    TVector<NStatusRepositoryTypes::ENetworkHookType> ExtractWorkloadHttpHookTypes(
        const TWorkloadMeta& objectMeta
    );
    TVector<NStatusRepositoryTypes::ENetworkHookType> ExtractWorkloadTcpHookTypes(
        const TWorkloadMeta& objectMeta
    );

private:
    static constexpr const char* CONDITION_POD_SIGNAL_PREFIX = "condition_pod_";
    static constexpr const char* CONDITION_CACHE_OBJECT_SIGNAL_PREFIX = "condition_cache_object_";
    static constexpr const char* CONDITION_OBJECT_SIGNAL_PREFIX = "condition_object_";

    // UNKNOWN = 0, TRUE = 1, FALSE = 2, 3 is a barrier value
    static inline const NUnistat::TIntervals CONDITION_SIGNAL_INTERVALS = {0, 1, 2, 3};

    static inline const TVector<EConditionSignalType> CONDITION_SIGNAL_TYPES = {
        EConditionSignalType::READY
        , EConditionSignalType::IN_PROGRESS
        , EConditionSignalType::FAILED
    };

    // Container
    static constexpr const char* CONTAINER_SIGNAL_PREFIX = "container_";

    static inline const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> BOX_CONTAINER_TYPES = {
        // META is special
        // INIT is special
    };

    static inline const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> WORKLOAD_CONTAINER_TYPES = {
        NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
        // INIT is special
    };

    static inline const TVector<EContainerSignalType> CONTAINER_SIGNAL_TYPES = {
        EContainerSignalType::ZERO_RETURN_CODE
        , EContainerSignalType::POSITIVE_RETURN_CODE
        , EContainerSignalType::OOM
        , EContainerSignalType::TIMEOUT
        , EContainerSignalType::KILLED_EXTERNALLY
        , EContainerSignalType::SYSTEM_FAILURE
        , EContainerSignalType::EXITED
    };

    TLightRWLock SignalLock_;

    // Network
    static constexpr const char* NETWORK_SIGNAL_PREFIX = "network_";

    static inline const TVector<NStatusRepositoryTypes::ENetworkHookType> HTTP_HOOK_TYPES = {
        NStatusRepositoryTypes::ENetworkHookType::READINESS
        , NStatusRepositoryTypes::ENetworkHookType::LIVENESS
        , NStatusRepositoryTypes::ENetworkHookType::STOP
        , NStatusRepositoryTypes::ENetworkHookType::DESTROY
    };

    static inline const TVector<NStatusRepositoryTypes::ENetworkHookType> TCP_HOOK_TYPES = {
        NStatusRepositoryTypes::ENetworkHookType::READINESS
        , NStatusRepositoryTypes::ENetworkHookType::LIVENESS
    };

    static inline const TVector<ENetworkSignalType> HTTP_SIGNAL_TYPES = {
        ENetworkSignalType::SUCCESS
        , ENetworkSignalType::ERROR
        , ENetworkSignalType::TIMEOUT
        , ENetworkSignalType::WRONG_ANSWER
    };

    static inline const TVector<ENetworkSignalType> TCP_SIGNAL_TYPES = {
        ENetworkSignalType::SUCCESS
        , ENetworkSignalType::ERROR
        , ENetworkSignalType::TIMEOUT
    };

};

} // namespace NInfra::NPodAgent
