#include "unistat_object_helper.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NUnistatObjectHelperTest {

// Not singleton because we want to call TUnistatObjectHelper constructor for every test
// It is important because AddAllPodConditionSignals is only called in the constructor
TUnistatObjectHelper GetEmptyUnistatObjectHelper() {
    TMultiUnistat::Instance().Reset(TMultiUnistat::ESignalNamespace::USER);
    TMultiUnistat::Instance().Reset(TMultiUnistat::ESignalNamespace::INFRA);
    return TUnistatObjectHelper();
}

TVector<TString> FilterAndNormalize(const TVector<TString>& arr, const TString& prefix = "") {
    TVector<TString> result;
    for (const TString& st : arr) {
        if (st.StartsWith(prefix)) {
            result.push_back(st);
        }
    }
    // Normalize by sorting
    Sort(result.begin(), result.end());

    return result;
}

Y_UNIT_TEST_SUITE(UnistatObjectHelperSuite) {

Y_UNIT_TEST(TestAddPodConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();
    Y_UNUSED(unistatObjectHelper);

    const TVector<TString> expected = {
        "condition_pod_failed"
        , "condition_pod_in_progress"
        , "condition_pod_ready"
    };

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::INFRA)
        , "condition_pod_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestAddRemoveCacheLayerConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";
    const ui32 revision1 = 3;
    const ui32 revision2 = 4;
    const TString downloadHash = "download_hash";

    unistatObjectHelper.AddCacheObject(
        NObjectMetaTestLib::CreateCacheLayerMetaSimple(
            id1
            , downloadHash
            , revision1
        )
    );
    unistatObjectHelper.AddCacheObject(
        NObjectMetaTestLib::CreateCacheLayerMetaSimple(
            id2
            , downloadHash
            , revision2
        )
    );

    {
        const TVector<TString> expected = {
            "condition_cache_object_layer_id1_3_failed"
            , "condition_cache_object_layer_id1_3_in_progress"
            , "condition_cache_object_layer_id1_3_ready"
            , "condition_cache_object_layer_id2_4_failed"
            , "condition_cache_object_layer_id2_4_in_progress"
            , "condition_cache_object_layer_id2_4_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_cache_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveCacheObject(
        id1
        , revision1
        , NStatusRepositoryTypes::EObjectType::LAYER
    );

    {
        const TVector<TString> expected = {
            "condition_cache_object_layer_id2_4_failed"
            , "condition_cache_object_layer_id2_4_in_progress"
            , "condition_cache_object_layer_id2_4_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_cache_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveCacheStaticResourceConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";
    const ui32 revision1 = 3;
    const ui32 revision2 = 4;
    const TString downloadHash = "download_hash";

    unistatObjectHelper.AddCacheObject(
        NObjectMetaTestLib::CreateCacheStaticResourceMetaSimple(
            id1
            , downloadHash
            , revision1
        )
    );
    unistatObjectHelper.AddCacheObject(
        NObjectMetaTestLib::CreateCacheStaticResourceMetaSimple(
            id2
            , downloadHash
            , revision2
        )
    );

    {
        const TVector<TString> expected = {
            "condition_cache_object_static_resource_id1_3_failed"
            , "condition_cache_object_static_resource_id1_3_in_progress"
            , "condition_cache_object_static_resource_id1_3_ready"
            , "condition_cache_object_static_resource_id2_4_failed"
            , "condition_cache_object_static_resource_id2_4_in_progress"
            , "condition_cache_object_static_resource_id2_4_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_cache_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveCacheObject(
        id1
        , revision1
        , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
    );

    {
        const TVector<TString> expected = {
            "condition_cache_object_static_resource_id2_4_failed"
            , "condition_cache_object_static_resource_id2_4_in_progress"
            , "condition_cache_object_static_resource_id2_4_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_cache_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveBoxConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateBoxMetaSimple(
            id1
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateBoxMetaSimple(
            id2
        )
    );

    {
        const TVector<TString> expected = {
            "condition_object_box_id1_failed"
            , "condition_object_box_id1_in_progress"
            , "condition_object_box_id1_ready"
            , "condition_object_box_id2_failed"
            , "condition_object_box_id2_in_progress"
            , "condition_object_box_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveObject(
        id1
        , NStatusRepositoryTypes::EObjectType::BOX
    );

    {
        const TVector<TString> expected = {
            "condition_object_box_id2_failed"
            , "condition_object_box_id2_in_progress"
            , "condition_object_box_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveLayerConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";
    const TString downloadHash = "download_hash";

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateLayerMetaSimple(
            id1
            , downloadHash
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateLayerMetaSimple(
            id2
            , downloadHash
        )
    );

    {
        const TVector<TString> expected = {
            "condition_object_layer_id1_failed"
            , "condition_object_layer_id1_in_progress"
            , "condition_object_layer_id1_ready"
            , "condition_object_layer_id2_failed"
            , "condition_object_layer_id2_in_progress"
            , "condition_object_layer_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveObject(
        id1
        , NStatusRepositoryTypes::EObjectType::LAYER
    );

    {
        const TVector<TString> expected = {
            "condition_object_layer_id2_failed"
            , "condition_object_layer_id2_in_progress"
            , "condition_object_layer_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveStaticResourceConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";
    const TString downloadHash = "download_hash";

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateStaticResourceMetaSimple(
            id1
            , downloadHash
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateStaticResourceMetaSimple(
            id2
            , downloadHash
        )
    );

    {
        const TVector<TString> expected = {
            "condition_object_static_resource_id1_failed"
            , "condition_object_static_resource_id1_in_progress"
            , "condition_object_static_resource_id1_ready"
            , "condition_object_static_resource_id2_failed"
            , "condition_object_static_resource_id2_in_progress"
            , "condition_object_static_resource_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveObject(
        id1
        , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
    );

    {
        const TVector<TString> expected = {
            "condition_object_static_resource_id2_failed"
            , "condition_object_static_resource_id2_in_progress"
            , "condition_object_static_resource_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveVolumeConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateVolumeMetaSimple(
            id1
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateVolumeMetaSimple(
            id2
        )
    );

    {
        const TVector<TString> expected = {
            "condition_object_volume_id1_failed"
            , "condition_object_volume_id1_in_progress"
            , "condition_object_volume_id1_ready"
            , "condition_object_volume_id2_failed"
            , "condition_object_volume_id2_in_progress"
            , "condition_object_volume_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveObject(
        id1
        , NStatusRepositoryTypes::EObjectType::VOLUME
    );

    {
        const TVector<TString> expected = {
            "condition_object_volume_id2_failed"
            , "condition_object_volume_id2_in_progress"
            , "condition_object_volume_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddRemoveWorkloadConditions) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString id1 = "id1";
    const TString id2 = "id2";

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            id1
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            id2
        )
    );

    {
        const TVector<TString> expected = {
            "condition_object_workload_id1_failed"
            , "condition_object_workload_id1_in_progress"
            , "condition_object_workload_id1_ready"
            , "condition_object_workload_id2_failed"
            , "condition_object_workload_id2_in_progress"
            , "condition_object_workload_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }

    unistatObjectHelper.RemoveObject(
        id1
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );

    {
        const TVector<TString> expected = {
            "condition_object_workload_id2_failed"
            , "condition_object_workload_id2_in_progress"
            , "condition_object_workload_id2_ready"
        };

        TVector<TString> holeNames = FilterAndNormalize(
            TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
            , "condition_object_"
        );
        UNIT_ASSERT_EQUAL(holeNames, expected);
    }
}

Y_UNIT_TEST(TestAddContainers) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            "workload_id"
            , 1
            , 1
            , 0 // seed = 0 => initSize = 2
            , true
        )
    );

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateBoxMetaSimple(
            "box_id"
            , 0
            , 1 // seed = 1 => initSize = 3
        )
    );

    const TVector<TString> expected = {
        "container_box_box_id_init_0_exited"
        , "container_box_box_id_init_0_killed_externally"
        , "container_box_box_id_init_0_oom"
        , "container_box_box_id_init_0_positive_return_code"
        , "container_box_box_id_init_0_system_failure"
        , "container_box_box_id_init_0_timeout"
        , "container_box_box_id_init_0_zero_return_code"
        , "container_box_box_id_init_1_exited"
        , "container_box_box_id_init_1_killed_externally"
        , "container_box_box_id_init_1_oom"
        , "container_box_box_id_init_1_positive_return_code"
        , "container_box_box_id_init_1_system_failure"
        , "container_box_box_id_init_1_timeout"
        , "container_box_box_id_init_1_zero_return_code"
        , "container_box_box_id_init_2_exited"
        , "container_box_box_id_init_2_killed_externally"
        , "container_box_box_id_init_2_oom"
        , "container_box_box_id_init_2_positive_return_code"
        , "container_box_box_id_init_2_system_failure"
        , "container_box_box_id_init_2_timeout"
        , "container_box_box_id_init_2_zero_return_code"
        , "container_workload_workload_id_destroy_exited"
        , "container_workload_workload_id_destroy_killed_externally"
        , "container_workload_workload_id_destroy_oom"
        , "container_workload_workload_id_destroy_positive_return_code"
        , "container_workload_workload_id_destroy_system_failure"
        , "container_workload_workload_id_destroy_timeout"
        , "container_workload_workload_id_destroy_zero_return_code"
        , "container_workload_workload_id_init_0_exited"
        , "container_workload_workload_id_init_0_killed_externally"
        , "container_workload_workload_id_init_0_oom"
        , "container_workload_workload_id_init_0_positive_return_code"
        , "container_workload_workload_id_init_0_system_failure"
        , "container_workload_workload_id_init_0_timeout"
        , "container_workload_workload_id_init_0_zero_return_code"
        , "container_workload_workload_id_init_1_exited"
        , "container_workload_workload_id_init_1_killed_externally"
        , "container_workload_workload_id_init_1_oom"
        , "container_workload_workload_id_init_1_positive_return_code"
        , "container_workload_workload_id_init_1_system_failure"
        , "container_workload_workload_id_init_1_timeout"
        , "container_workload_workload_id_init_1_zero_return_code"
        , "container_workload_workload_id_liveness_exited"
        , "container_workload_workload_id_liveness_killed_externally"
        , "container_workload_workload_id_liveness_oom"
        , "container_workload_workload_id_liveness_positive_return_code"
        , "container_workload_workload_id_liveness_system_failure"
        , "container_workload_workload_id_liveness_timeout"
        , "container_workload_workload_id_liveness_zero_return_code"
        , "container_workload_workload_id_readiness_exited"
        , "container_workload_workload_id_readiness_killed_externally"
        , "container_workload_workload_id_readiness_oom"
        , "container_workload_workload_id_readiness_positive_return_code"
        , "container_workload_workload_id_readiness_system_failure"
        , "container_workload_workload_id_readiness_timeout"
        , "container_workload_workload_id_readiness_zero_return_code"
        , "container_workload_workload_id_start_exited"
        , "container_workload_workload_id_start_killed_externally"
        , "container_workload_workload_id_start_oom"
        , "container_workload_workload_id_start_positive_return_code"
        , "container_workload_workload_id_start_system_failure"
        , "container_workload_workload_id_start_timeout"
        , "container_workload_workload_id_start_zero_return_code"
        , "container_workload_workload_id_stop_exited"
        , "container_workload_workload_id_stop_killed_externally"
        , "container_workload_workload_id_stop_oom"
        , "container_workload_workload_id_stop_positive_return_code"
        , "container_workload_workload_id_stop_system_failure"
        , "container_workload_workload_id_stop_timeout"
        , "container_workload_workload_id_stop_zero_return_code"
    };

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "container_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestRemoveContainer) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            "workload_id1"
            , 1
            , 1
            , 0 // seed = 0 => initSize = 2
            , true
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            "workload_id2"
            , 1
            , 1
            , 15 // seed = 15 => initSize = 17
            , true
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateBoxMetaSimple(
            "box_id1"
            , 0
            , 0 // seed = 0 => initSize = 2
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateBoxMetaSimple(
            "box_id2"
            , 0
            , 2 // seed = 2 => initSize = 4
        )
    );

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "container_"
    );

    // List is too big, just check size
    UNIT_ASSERT_EQUAL_C(
        holeNames.size()
        , 245
        , holeNames.size()
    );

    unistatObjectHelper.RemoveObject(
        "workload_id1"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );
    unistatObjectHelper.RemoveObject(
        "workload_id2"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );
    unistatObjectHelper.RemoveObject(
        "box_id2"
        , NStatusRepositoryTypes::EObjectType::BOX
    );

    // Check result after remove
    const TVector<TString> expected = {
        "container_box_box_id1_init_0_exited"
        , "container_box_box_id1_init_0_killed_externally"
        , "container_box_box_id1_init_0_oom"
        , "container_box_box_id1_init_0_positive_return_code"
        , "container_box_box_id1_init_0_system_failure"
        , "container_box_box_id1_init_0_timeout"
        , "container_box_box_id1_init_0_zero_return_code"
        , "container_box_box_id1_init_1_exited"
        , "container_box_box_id1_init_1_killed_externally"
        , "container_box_box_id1_init_1_oom"
        , "container_box_box_id1_init_1_positive_return_code"
        , "container_box_box_id1_init_1_system_failure"
        , "container_box_box_id1_init_1_timeout"
        , "container_box_box_id1_init_1_zero_return_code"
    };

    holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "container_"
    );
    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestPushConditionSignal) {
    auto checkConditionSignalStatus = [](
        const TMultiUnistat::ESignalNamespace signalNamespace
        , const TString& signalToCheck
        , TUnistatObjectHelper::EConditionStatus status
    ) {
        TInstanceStats signals;
        TMultiUnistat::Instance().ExportToProto(signalNamespace, signals, TMultiUnistat::ESignalPriority::USER_INFO);

        bool hasSignal = false;
        for (const auto& signal : signals.GetMetric()) {
            if (signal.GetName() == signalToCheck) {
                const auto& metric = signal.GetHgram();
                UNIT_ASSERT_EQUAL_C(metric.GetBucket().size(), 4, metric.GetBucket().size());
                for (ui32 i = 0; i < 4; ++i) {
                    if (i == ui32(status)) {
                        UNIT_ASSERT_EQUAL_C(metric.GetBucket(i).GetWeight(), 1, metric.GetBucket(i).GetWeight());
                    } else {
                        UNIT_ASSERT_EQUAL_C(metric.GetBucket(i).GetWeight(), 0, metric.GetBucket(i).GetWeight());
                    }

                    UNIT_ASSERT_DOUBLES_EQUAL_C(metric.GetBucket(i).GetBoundary(), i, 1e-9, metric.GetBucket(i).GetBoundary());
                }
                hasSignal = true;
            }
        }

        UNIT_ASSERT_C(hasSignal, "signal not found");
    };

    auto test = [&checkConditionSignalStatus](
        std::function<void(TUnistatObjectHelper& unistatObjectHelper)> addObject
        , std::function<void(TUnistatObjectHelper& unistatObjectHelper, TUnistatObjectHelper::EConditionStatus status)> pushConditionSignal
        , const TMultiUnistat::ESignalNamespace signalNamespace
        , const TString& signalToCheck
    ) {
        auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

        addObject(unistatObjectHelper);
        // Check default value
        checkConditionSignalStatus(signalNamespace, signalToCheck, TUnistatObjectHelper::EConditionStatus::UNKNOWN);

        const ui32 iter = 2;
        const TVector<TUnistatObjectHelper::EConditionStatus> conditionStatuses = {
            TUnistatObjectHelper::EConditionStatus::UNKNOWN
            , TUnistatObjectHelper::EConditionStatus::TRUE
            , TUnistatObjectHelper::EConditionStatus::FALSE
        };

        for (ui32 i = 0; i < iter; ++i) {
            for (const auto& status : conditionStatuses) {
                pushConditionSignal(unistatObjectHelper, status);
                checkConditionSignalStatus(signalNamespace, signalToCheck, status);
            }
        }
    };

    // Pod
    test(
        [](
            TUnistatObjectHelper& /* unistatObjectHelper */
        ) {
        }
        , [](
            TUnistatObjectHelper& unistatObjectHelper
            , TUnistatObjectHelper::EConditionStatus status
        ) {
            UNIT_ASSERT(
                unistatObjectHelper.PushPodConditionSignal(
                    TUnistatObjectHelper::EConditionSignalType::READY
                    , status
                )
            );
        }
        , TMultiUnistat::ESignalNamespace::INFRA
        , "condition_pod_ready_ahhh"
    );

    // CacheObject
    test(
        [](
            TUnistatObjectHelper& unistatObjectHelper
        ) {
            unistatObjectHelper.AddCacheObject(
                NObjectMetaTestLib::CreateCacheStaticResourceMetaSimple(
                    "id"
                    , "downloadHash"
                    , 1
                )
            );
        }
        , [](
            TUnistatObjectHelper& unistatObjectHelper
            , TUnistatObjectHelper::EConditionStatus status
        ) {
            UNIT_ASSERT(
                unistatObjectHelper.PushCacheObjectConditionSignal(
                    "id"
                    , 1
                    , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
                    , TUnistatObjectHelper::EConditionSignalType::IN_PROGRESS
                    , status
                )
            );
        }
        , TMultiUnistat::ESignalNamespace::USER
        , "condition_cache_object_static_resource_id_1_in_progress_ahhh"
    );

    // Object
    test(
        [](
            TUnistatObjectHelper& unistatObjectHelper
        ) {
            unistatObjectHelper.AddObject(
                NObjectMetaTestLib::CreateVolumeMetaSimple(
                    "id"
                )
            );
        }
        , [](
            TUnistatObjectHelper& unistatObjectHelper
            , TUnistatObjectHelper::EConditionStatus status
        ) {
            UNIT_ASSERT(
                unistatObjectHelper.PushObjectConditionSignal(
                    "id"
                    , NStatusRepositoryTypes::EObjectType::VOLUME
                    , TUnistatObjectHelper::EConditionSignalType::FAILED
                    , status
                )
            );
        }
        , TMultiUnistat::ESignalNamespace::USER
        , "condition_object_volume_id_failed_ahhh"
    );
}

Y_UNIT_TEST(TestPushContainerSignal) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    const TString workloadId = "workload_id";
    const NStatusRepositoryTypes::TContainerDescription container(
        workloadId
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    const ui32 failValue = 3;
    const ui32 successValue = 4;

    UNIT_ASSERT(!unistatObjectHelper.PushContainerSignal(container, TUnistatObjectHelper::EContainerSignalType::TIMEOUT, failValue));
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaSimple(
            "workload_id"
            , 1
            , 1
            , 0 // seed = 0 => initSize = 2
            , true
        )
    );
    UNIT_ASSERT(unistatObjectHelper.PushContainerSignal(container, TUnistatObjectHelper::EContainerSignalType::TIMEOUT, successValue));

    TInstanceStats signals;
    TMultiUnistat::Instance().ExportToProto(TMultiUnistat::ESignalNamespace::USER, signals, TMultiUnistat::ESignalPriority::USER_INFO);

    bool hasTimeout = false;
    for (const auto& signal : signals.GetMetric()) {
        if (signal.GetName().StartsWith("container_")) {
            if (signal.GetName() == "container_workload_workload_id_start_timeout_deee") {
                UNIT_ASSERT_DOUBLES_EQUAL_C(signal.GetNumber(), successValue, 1e-9, signal.GetNumber());
                hasTimeout = true;
            } else {
                UNIT_ASSERT_DOUBLES_EQUAL_C(signal.GetNumber(), 0, 1e-9, signal.GetNumber());
            }
        }
    }

    UNIT_ASSERT_C(hasTimeout, "timeout signal not found");
}

Y_UNIT_TEST(TestAddHttp) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id"
            , 0
            , 0  // seed = 0 => initSize = 2
            , 2
            , 2
            , 2
            , 2
        )
    );

    const TVector<TString> expected = {
        "network_http_workload_id_destroy_error"
        , "network_http_workload_id_destroy_success"
        , "network_http_workload_id_destroy_timeout"
        , "network_http_workload_id_destroy_wrong_answer"
        , "network_http_workload_id_liveness_error"
        , "network_http_workload_id_liveness_success"
        , "network_http_workload_id_liveness_timeout"
        , "network_http_workload_id_liveness_wrong_answer"
        , "network_http_workload_id_readiness_error"
        , "network_http_workload_id_readiness_success"
        , "network_http_workload_id_readiness_timeout"
        , "network_http_workload_id_readiness_wrong_answer"
        , "network_http_workload_id_stop_error"
        , "network_http_workload_id_stop_success"
        , "network_http_workload_id_stop_timeout"
        , "network_http_workload_id_stop_wrong_answer"
    };

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestAddTcp) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id"
            , 0
            , 0  // seed = 0 => initSize = 2
            , 3
            , 3
            , 3
            , 3
        )
    );

    const TVector<TString> expected = {
        "network_tcp_workload_id_liveness_error"
        , "network_tcp_workload_id_liveness_success"
        , "network_tcp_workload_id_liveness_timeout"
        , "network_tcp_workload_id_readiness_error"
        , "network_tcp_workload_id_readiness_success"
        , "network_tcp_workload_id_readiness_timeout"
    };

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestRemoveHttp) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id1"
            , 0
            , 0 // seed = 0 => initSize = 2
            , 2
            , 2
            , 2
            , 2
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id2"
            , 0
            , 15 // seed = 15 => initSize = 17
            , 2
            , 2
            , 2
            , 2
        )
    );

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    // List is too big, just check size
    UNIT_ASSERT_EQUAL_C(
        holeNames.size()
        , 32
        , holeNames.size()
    );

    unistatObjectHelper.RemoveObject(
        "workload_id1"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );

    // Check result after remove
    const TVector<TString> expected = {
        "network_http_workload_id2_destroy_error"
        , "network_http_workload_id2_destroy_success"
        , "network_http_workload_id2_destroy_timeout"
        , "network_http_workload_id2_destroy_wrong_answer"
        , "network_http_workload_id2_liveness_error"
        , "network_http_workload_id2_liveness_success"
        , "network_http_workload_id2_liveness_timeout"
        , "network_http_workload_id2_liveness_wrong_answer"
        , "network_http_workload_id2_readiness_error"
        , "network_http_workload_id2_readiness_success"
        , "network_http_workload_id2_readiness_timeout"
        , "network_http_workload_id2_readiness_wrong_answer"
        , "network_http_workload_id2_stop_error"
        , "network_http_workload_id2_stop_success"
        , "network_http_workload_id2_stop_timeout"
        , "network_http_workload_id2_stop_wrong_answer"
    };

    holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestRemoveTcp) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id1"
            , 0
            , 0 // seed = 0 => initSize = 2
            , 3
            , 3
            , 3
            , 3
        )
    );
    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id2"
            , 0
            , 15 // seed = 15 => initSize = 17
            , 3
            , 3
            , 3
            , 3
        )
    );

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    // List is too big, just check size
    UNIT_ASSERT_EQUAL_C(
        holeNames.size()
        , 12
        , holeNames.size()
    );

    unistatObjectHelper.RemoveObject(
        "workload_id1"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );

    // Check result after remove
    const TVector<TString> expected = {
        "network_tcp_workload_id2_liveness_error"
        , "network_tcp_workload_id2_liveness_success"
        , "network_tcp_workload_id2_liveness_timeout"
        , "network_tcp_workload_id2_readiness_error"
        , "network_tcp_workload_id2_readiness_success"
        , "network_tcp_workload_id2_readiness_timeout"
    };

    holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expected);
}

Y_UNIT_TEST(TestPushNetworkSignal) {
    auto test = [](NStatusRepositoryTypes::EHookBackend hookType, const TString& signalName, ui32 hookTypeForCreateWorkloadMeta) {
        auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

        const TString workloadId = "workload_id";
        const ui32 failValue = 3;
        const ui32 successValue = 4;

        UNIT_ASSERT(
            !unistatObjectHelper.PushNetworkSignal(
                workloadId
                , hookType
                , NStatusRepositoryTypes::ENetworkHookType::READINESS
                , TUnistatObjectHelper::ENetworkSignalType::SUCCESS
                , failValue
            )
        );
        unistatObjectHelper.AddObject(
            NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
                "workload_id"
                , 0
                , 0 // seed = 0 => initSize = 2
                , hookTypeForCreateWorkloadMeta
                , hookTypeForCreateWorkloadMeta
                , hookTypeForCreateWorkloadMeta
                , hookTypeForCreateWorkloadMeta
            )
        );
        UNIT_ASSERT(
            unistatObjectHelper.PushNetworkSignal(
                workloadId
                , hookType
                , NStatusRepositoryTypes::ENetworkHookType::READINESS
                , TUnistatObjectHelper::ENetworkSignalType::SUCCESS
                , successValue
            )
        );

        TInstanceStats signals;
        TMultiUnistat::Instance().ExportToProto(TMultiUnistat::ESignalNamespace::USER, signals, TMultiUnistat::ESignalPriority::USER_INFO);

        bool hasSuccess = false;
        for (const auto& signal : signals.GetMetric()) {
            if (signal.GetName().StartsWith("network_")) {
                if (signal.GetName() == signalName) {
                    UNIT_ASSERT_DOUBLES_EQUAL_C(signal.GetNumber(), successValue, 1e-9, signal.GetNumber());
                    hasSuccess = true;
                } else {
                    UNIT_ASSERT_DOUBLES_EQUAL_C(signal.GetNumber(), 0, 1e-9, signal.GetNumber());
                }
            }
        }

        UNIT_ASSERT_C(hasSuccess, "success signal not found");
    };

    test(NStatusRepositoryTypes::EHookBackend::HTTP, "network_http_workload_id_readiness_success_deee", 2);
    test(NStatusRepositoryTypes::EHookBackend::TCP, "network_tcp_workload_id_readiness_success_deee", 3);
}

Y_UNIT_TEST(TestAddSignalsWithSpecificHookTypes) {
    auto unistatObjectHelper = GetEmptyUnistatObjectHelper();

    unistatObjectHelper.AddObject(
        NObjectMetaTestLib::CreateWorkloadMetaWithSpecificHookTypes(
            "workload_id"
            , 0
            , 0  // seed = 0 => initSize = 2
            , 3
            , 1
            , 2
            , 3
        )
    );

    TVector<TString> expectedUser = {
        "network_http_workload_id_stop_error"
        , "network_http_workload_id_stop_success"
        , "network_http_workload_id_stop_timeout"
        , "network_http_workload_id_stop_wrong_answer"
        , "network_tcp_workload_id_readiness_error"
        , "network_tcp_workload_id_readiness_success"
        , "network_tcp_workload_id_readiness_timeout"
    };

    TVector<TString> holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "network_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expectedUser);

    expectedUser = {
        "container_workload_workload_id_init_0_exited"
        , "container_workload_workload_id_init_0_killed_externally"
        , "container_workload_workload_id_init_0_oom"
        , "container_workload_workload_id_init_0_positive_return_code"
        , "container_workload_workload_id_init_0_system_failure"
        , "container_workload_workload_id_init_0_timeout"
        , "container_workload_workload_id_init_0_zero_return_code"
        , "container_workload_workload_id_init_1_exited"
        , "container_workload_workload_id_init_1_killed_externally"
        , "container_workload_workload_id_init_1_oom"
        , "container_workload_workload_id_init_1_positive_return_code"
        , "container_workload_workload_id_init_1_system_failure"
        , "container_workload_workload_id_init_1_timeout"
        , "container_workload_workload_id_init_1_zero_return_code"
        , "container_workload_workload_id_liveness_exited"
        , "container_workload_workload_id_liveness_killed_externally"
        , "container_workload_workload_id_liveness_oom"
        , "container_workload_workload_id_liveness_positive_return_code"
        , "container_workload_workload_id_liveness_system_failure"
        , "container_workload_workload_id_liveness_timeout"
        , "container_workload_workload_id_liveness_zero_return_code"
        , "container_workload_workload_id_start_exited"
        , "container_workload_workload_id_start_killed_externally"
        , "container_workload_workload_id_start_oom"
        , "container_workload_workload_id_start_positive_return_code"
        , "container_workload_workload_id_start_system_failure"
        , "container_workload_workload_id_start_timeout"
        , "container_workload_workload_id_start_zero_return_code"
    };

    holeNames = FilterAndNormalize(
        TMultiUnistat::Instance().GetHolenames(TMultiUnistat::ESignalNamespace::USER)
        , "container_"
    );

    UNIT_ASSERT_EQUAL(holeNames, expectedUser);
}

}

} // namespace NInfra::NPodAgent::NUnistatObjectHelperTest
