#include "update_holder.h"

namespace NInfra::NPodAgent {

bool TUpdateHolderTarget::LayerHasTarget(const TString& layerId) const {
    return ObjectHasTarget(TObject(layerId, EObjectType::LAYER));
}

bool TUpdateHolderTarget::LayerHasTargetRemove(const TString& layerId) const {
    return ObjectHasTargetRemove(TObject(layerId, EObjectType::LAYER));
}

bool TUpdateHolderTarget::StaticResourceHasTarget(const TString& staticResourceId) const {
    return ObjectHasTarget(TObject(staticResourceId, EObjectType::STATIC_RESOURCE));
}

bool TUpdateHolderTarget::StaticResourceHasTargetRemove(const TString& staticResourceId) const {
    return ObjectHasTargetRemove(TObject(staticResourceId, EObjectType::STATIC_RESOURCE));
}

bool TUpdateHolderTarget::VolumeHasTarget(const TString& volumeId) const {
    return ObjectHasTarget(TObject(volumeId, EObjectType::VOLUME));
}

bool TUpdateHolderTarget::VolumeHasTargetRemove(const TString& volumeId) const {
    return ObjectHasTargetRemove(TObject(volumeId, EObjectType::VOLUME));
}

bool TUpdateHolderTarget::BoxHasTarget(const TString& boxId) const {
    return ObjectHasTarget(TObject(boxId, EObjectType::BOX));
}

bool TUpdateHolderTarget::BoxHasTargetRemove(const TString& boxId) const {
    return ObjectHasTargetRemove(TObject(boxId, EObjectType::BOX));
}

bool TUpdateHolderTarget::WorkloadHasTarget(const TString& workloadId) const {
    return ObjectHasTarget(TObject(workloadId, EObjectType::WORKLOAD));
}

bool TUpdateHolderTarget::WorkloadHasTargetRemove(const TString& workloadId) const {
    return ObjectHasTargetRemove(TObject(workloadId, EObjectType::WORKLOAD));
}

void TUpdateHolderTarget::SetObjectTarget(const TObject& object, ETargetType targetType) {
    TWriteGuardBase<TLightRWLock> guard(TargetsLock_);
    Targets_[object] = targetType;
}

void TUpdateHolderTarget::RemoveObjectTarget(const TObject& object) {
    TWriteGuardBase<TLightRWLock> guard(TargetsLock_);
    Targets_.erase(object);
}

bool TUpdateHolderTarget::ObjectHasTarget(const TObject& object) const {
    TReadGuardBase<TLightRWLock> guard(TargetsLock_);

    return Targets_.FindPtr(object);
}

bool TUpdateHolderTarget::ObjectHasTargetRemove(const TObject& object) const {
    TReadGuardBase<TLightRWLock> guard(TargetsLock_);

    auto ptr = Targets_.find(object);

    if (ptr != Targets_.end()) {
        return ptr->second == ETargetType::TT_REMOVE;
    }

    return false;
}

TUpdateHolderTargetPtr TUpdateHolder::GetUpdateHolderTarget() const {
    return UpdateHolderTarget_;
}

void TUpdateHolder::SetBoxTarget(const TBoxTarget& target) {
    // All checks must be in StatusNTickerHolder
    TGuard<TMutex> gMutex(BoxMutex_);
    UpdateHolderTarget_->SetObjectTarget(TUpdateHolderTarget::TObject(target.Meta_.Id_, TUpdateHolderTarget::EObjectType::BOX), TUpdateHolderTarget::GetTargetType(target));
    if (auto ptr = BoxTargets_.find(target.Meta_.Id_); ptr != BoxTargets_.end()) {
        BoxTargets_.erase(ptr);
    }
    BoxTargets_.insert({target.Meta_.Id_, target});
}

TUpdateHolder::TBoxTarget TUpdateHolder::GetAndRemoveBoxTarget(const TString& boxId) {
    TGuard<TMutex> gMutex(BoxMutex_);
    auto ptr = BoxTargets_.find(boxId);

    if (ptr != BoxTargets_.end()) {
        TBoxTarget copy(ptr->second);
        BoxTargets_.erase(ptr);
        UpdateHolderTarget_->RemoveObjectTarget(TUpdateHolderTarget::TObject(boxId, TUpdateHolderTarget::EObjectType::BOX));
        return copy;
    }

    return TBoxTarget::GetEmptyTarget();
}

void TUpdateHolder::SetLayerTarget(const TLayerTarget& target) {
    // All checks must be in StatusNTickerHolder
    TGuard<TMutex> gMutex(LayerMutex_);
    UpdateHolderTarget_->SetObjectTarget(TUpdateHolderTarget::TObject(target.Meta_.Id_, TUpdateHolderTarget::EObjectType::LAYER), TUpdateHolderTarget::GetTargetType(target));
    if (auto ptr = LayerTargets_.find(target.Meta_.Id_); ptr != LayerTargets_.end()) {
        LayerTargets_.erase(ptr);
    }
    LayerTargets_.insert({target.Meta_.Id_, target});
}

TUpdateHolder::TLayerTarget TUpdateHolder::GetAndRemoveLayerTarget(const TString& layerId) {
    TGuard<TMutex> gMutex(LayerMutex_);
    auto ptr = LayerTargets_.find(layerId);

    if (ptr != LayerTargets_.end()) {
        TLayerTarget copy(ptr->second);
        LayerTargets_.erase(ptr);
        UpdateHolderTarget_->RemoveObjectTarget(TUpdateHolderTarget::TObject(layerId, TUpdateHolderTarget::EObjectType::LAYER));
        return copy;
    }

    return TLayerTarget::GetEmptyTarget();
}

void TUpdateHolder::SetStaticResourceTarget(const TStaticResourceTarget& target) {
    // All checks must be in StatusNTickerHolder
    TGuard<TMutex> gMutex(StaticResourceMutex_);
    UpdateHolderTarget_->SetObjectTarget(TUpdateHolderTarget::TObject(target.Meta_.Id_, TUpdateHolderTarget::EObjectType::STATIC_RESOURCE), TUpdateHolderTarget::GetTargetType(target));
    if (auto ptr = StaticResourceTargets_.find(target.Meta_.Id_); ptr != StaticResourceTargets_.end()) {
        StaticResourceTargets_.erase(ptr);
    }
    StaticResourceTargets_.insert({target.Meta_.Id_, target});
}

TUpdateHolder::TStaticResourceTarget TUpdateHolder::GetAndRemoveStaticResourceTarget(const TString& staticResourceId) {
    TGuard<TMutex> gMutex(StaticResourceMutex_);
    auto ptr = StaticResourceTargets_.find(staticResourceId);

    if (ptr != StaticResourceTargets_.end()) {
        TStaticResourceTarget copy(ptr->second);
        StaticResourceTargets_.erase(ptr);
        UpdateHolderTarget_->RemoveObjectTarget(TUpdateHolderTarget::TObject(staticResourceId, TUpdateHolderTarget::EObjectType::STATIC_RESOURCE));
        return copy;
    }

    return TStaticResourceTarget::GetEmptyTarget();
}

void TUpdateHolder::SetVolumeTarget(const TVolumeTarget& target) {
    // All checks must be in StatusNTickerHolder
    TGuard<TMutex> gMutex(VolumeMutex_);
    UpdateHolderTarget_->SetObjectTarget(TUpdateHolderTarget::TObject(target.Meta_.Id_, TUpdateHolderTarget::EObjectType::VOLUME), TUpdateHolderTarget::GetTargetType(target));
    if (auto ptr = VolumeTargets_.find(target.Meta_.Id_); ptr != VolumeTargets_.end()) {
        VolumeTargets_.erase(ptr);
    }
    VolumeTargets_.insert({target.Meta_.Id_, target});
}

TUpdateHolder::TVolumeTarget TUpdateHolder::GetAndRemoveVolumeTarget(const TString& volumeId) {
    TGuard<TMutex> gMutex(VolumeMutex_);
    auto ptr = VolumeTargets_.find(volumeId);

    if (ptr != VolumeTargets_.end()) {
        TVolumeTarget copy(ptr->second);
        VolumeTargets_.erase(ptr);
        UpdateHolderTarget_->RemoveObjectTarget(TUpdateHolderTarget::TObject(volumeId, TUpdateHolderTarget::EObjectType::VOLUME));
        return copy;
    }

    return TVolumeTarget::GetEmptyTarget();
}

void TUpdateHolder::SetWorkloadTarget(const TWorkloadTarget& target) {
    // All checks must in at StatusNTickerHolder
    TGuard<TMutex> gMutex(WorkloadMutex_);
    UpdateHolderTarget_->SetObjectTarget(TUpdateHolderTarget::TObject(target.Meta_.Id_, TUpdateHolderTarget::EObjectType::WORKLOAD), TUpdateHolderTarget::GetTargetType(target));
    if (auto ptr = WorkloadTargets_.find(target.Meta_.Id_); ptr != WorkloadTargets_.end()) {
        WorkloadTargets_.erase(ptr);
    }
    WorkloadTargets_.insert({target.Meta_.Id_, target});
}

TUpdateHolder::TWorkloadTarget TUpdateHolder::GetAndRemoveWorkloadTarget(const TString& workloadId) {
    TGuard<TMutex> gMutex(WorkloadMutex_);
    auto ptr = WorkloadTargets_.find(workloadId);

    if (ptr != WorkloadTargets_.end()) {
        TWorkloadTarget copy(ptr->second);
        WorkloadTargets_.erase(ptr);
        UpdateHolderTarget_->RemoveObjectTarget(TUpdateHolderTarget::TObject(workloadId, TUpdateHolderTarget::EObjectType::WORKLOAD));
        return copy;
    }

    return TWorkloadTarget::GetEmptyTarget();
}

} // namespace NInfra::NPodAgent
