#pragma once

#include <infra/pod_agent/libs/behaviour/bt/core/tree.h>

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/system/mutex.h>

namespace NInfra::NPodAgent {

/*
    Thread-safe UpdateHolder
*/
class TUpdateHolder;

// This class used to avoid loops between pointers to TTree and to TUpdateHolder
class TUpdateHolderTarget: public TAtomicRefCount<TUpdateHolderTarget> {
public:
    friend TUpdateHolder;

    // Box
    bool BoxHasTarget(const TString& boxId) const;
    bool BoxHasTargetRemove(const TString& boxId) const;

    // Layer
    bool LayerHasTarget(const TString& layerId) const;
    bool LayerHasTargetRemove(const TString& layerId) const;

    // StaticResource
    bool StaticResourceHasTarget(const TString& staticResourceId) const;
    bool StaticResourceHasTargetRemove(const TString& staticResourceId) const;

    // Volume
    bool VolumeHasTarget(const TString& volumeId) const;
    bool VolumeHasTargetRemove(const TString& volumeId) const;

    // Workload
    bool WorkloadHasTarget(const TString& workloadId) const;
    bool WorkloadHasTargetRemove(const TString& workloadId) const;

protected:
    enum EObjectType {
        BOX,
        STATIC_RESOURCE,
        LAYER,
        VOLUME,
        WORKLOAD
    };

    enum ETargetType {
        TT_REVISION,
        TT_REMOVE
    };

    struct TObject {
        TObject(const TString id, EObjectType objectType)
            : Id_(id)
            , ObjectType_(objectType)
        {}

        bool operator<(const TObject& other) const {
            return (ObjectType_ < other.ObjectType_) || (ObjectType_ == other.ObjectType_ && Id_ < other.Id_);
        }

        TString Id_;
        EObjectType ObjectType_;
    };

protected:
    TUpdateHolderTarget() = default;

    void SetObjectTarget(const TObject& object, ETargetType targetType);
    void RemoveObjectTarget(const TObject& object);
    bool ObjectHasTarget(const TObject& object) const;
    bool ObjectHasTargetRemove(const TObject& object) const;

    template<typename T>
    static ETargetType GetTargetType(const T& target) {
        return target.TargetRemove_ ? TUpdateHolderTarget::ETargetType::TT_REMOVE : TUpdateHolderTarget::ETargetType::TT_REVISION;
    }

private:
    TLightRWLock TargetsLock_;
    TMap<TObject, ETargetType> Targets_;
};

using TUpdateHolderTargetPtr = TIntrusivePtr<TUpdateHolderTarget>;

class TUpdateHolder: public TAtomicRefCount<TUpdateHolder> {
public:
    struct TBoxTarget {
        TBoxTarget(
            const TBoxMeta& meta
            , const TTreePtr tree
            , const TString& hash
            , bool targetRemove = false
        )
            : Meta_(meta)
            , Tree_(tree)
            , Hash_(hash)
            , TargetRemove_(targetRemove)
        {}

        bool operator==(const TBoxTarget& other) const {
            return Meta_ == other.Meta_
                && Tree_ == other.Tree_
                && Hash_ == other.Hash_
                && TargetRemove_ == other.TargetRemove_
            ;
        }

        static TBoxTarget GetEmptyTarget(const TString& id = "", bool targetRemove = false) {
            return TBoxTarget(
                TBoxMeta(
                    id
                    , 0
                    , 0
                    , {}
                    , {}
                    , {}
                    , ""
                    , {}
                    , ""
                )
                , nullptr
                , ""
                , targetRemove
            );
        }

        static TBoxTarget GetTargetRemove(const TString& id) {
            return GetEmptyTarget(id, true);
        }

        const TBoxMeta Meta_;
        const TTreePtr Tree_;
        const TString Hash_;
        const bool TargetRemove_;
    };

    struct TLayerTarget {
        TLayerTarget(
            const TLayerMeta& meta
            , const TTreePtr tree
            , bool targetRemove = false
        )
            : Meta_(meta)
            , Tree_(tree)
            , TargetRemove_(targetRemove)
        {}

        bool operator==(const TLayerTarget& other) const {
            return Meta_ == other.Meta_
                && Tree_ == other.Tree_
                && TargetRemove_ == other.TargetRemove_
            ;
        }

        static TLayerTarget GetEmptyTarget(const TString& id = "", bool targetRemove = false) {
            return TLayerTarget(
                TLayerMeta(
                    id
                    , 0
                    , 0
                    , ""
                    , false
                )
                , nullptr
                , targetRemove
            );
        }

        static TLayerTarget GetTargetRemove(const TString& id) {
            return GetEmptyTarget(id, true);
        }

        const TLayerMeta Meta_;
        TTreePtr Tree_;
        const bool TargetRemove_;
    };

    struct TStaticResourceTarget {
        TStaticResourceTarget(
            const TStaticResourceMeta& meta
            , TTreePtr tree
            , bool targetRemove = false
        )
            : Meta_(meta)
            , Tree_(tree)
            , TargetRemove_(targetRemove)
        {}

        bool operator==(const TStaticResourceTarget& other) const {
            return Meta_ == other.Meta_
                && Tree_ == other.Tree_
                && TargetRemove_ == other.TargetRemove_
            ;
        }

        static TStaticResourceTarget GetEmptyTarget(const TString& id = "", bool targetRemove = false) {
            return TStaticResourceTarget(
                TStaticResourceMeta(
                    id
                    , 0
                    , 0
                    , ""
                    , 0
                )
                , nullptr
                , targetRemove
            );
        }

        static TStaticResourceTarget GetTargetRemove(const TString& id) {
            return GetEmptyTarget(id, true);
        }

        const TStaticResourceMeta Meta_;
        const TTreePtr Tree_;
        const bool TargetRemove_;
    };

    struct TVolumeTarget {
        TVolumeTarget(
            const TVolumeMeta& meta
            , const TTreePtr tree
            , const TString& hash
            , bool isPersistent = true
            , bool targetRemove = false
        )
            : Meta_(meta)
            , Tree_(tree)
            , Hash_(hash)
            , IsPersistent_(isPersistent)
            , TargetRemove_(targetRemove)
        {}

        bool operator==(const TVolumeTarget& other) const {
            return Meta_ == other.Meta_
                && Tree_ == other.Tree_
                && Hash_ == other.Hash_
                && IsPersistent_ == other.IsPersistent_
                && TargetRemove_ == other.TargetRemove_
            ;
        }

        static TVolumeTarget GetEmptyTarget(const TString& id = "", bool targetRemove = false) {
            return TVolumeTarget(
                TVolumeMeta(
                    id
                    , 0
                    , 0
                    , {}
                    , {}
                )
                , nullptr
                , ""
                , true
                , targetRemove
            );
        }

        static TVolumeTarget GetTargetRemove(const TString& id) {
            return GetEmptyTarget(id, true);
        }

        const TVolumeMeta Meta_;
        const TTreePtr Tree_;
        const TString Hash_;
        const bool IsPersistent_;
        const bool TargetRemove_;
    };

    struct TWorkloadTarget {
        TWorkloadTarget(
            const TWorkloadMeta& meta
            , const TTreePtr tree
            , const TString& hash
            , bool targetRemove = false
        )
            : Meta_(meta)
            , Tree_(tree)
            , Hash_(hash)
            , TargetRemove_(targetRemove)
        {}

        bool operator==(const TWorkloadTarget& other) const {
            return Meta_ == other.Meta_
                && Tree_ == other.Tree_
                && Hash_ == other.Hash_
                && TargetRemove_ == other.TargetRemove_
            ;
        }

        static TWorkloadTarget GetEmptyTarget(const TString& id = "", bool targetRemove = false) {
            return TWorkloadTarget(
                TWorkloadMeta(
                    id
                    , 0
                    , 0
                    , ""

                    , TWorkloadMeta::TContainerInfo("")
                    , {}

                    , TWorkloadMeta::TEmptyInfo()
                    , TWorkloadMeta::TEmptyInfo()
                    , TWorkloadMeta::TEmptyInfo()
                    , TWorkloadMeta::TEmptyInfo()
                )
                , nullptr
                , ""
                , targetRemove
            );
        }

        static TWorkloadTarget GetTargetRemove(const TString& id) {
            return GetEmptyTarget(id, true);
        }

        const TWorkloadMeta Meta_;
        const TTreePtr Tree_;
        const TString Hash_;
        const bool TargetRemove_;
    };

public:
    TUpdateHolder()
        : UpdateHolderTarget_(new TUpdateHolderTarget())
    {}

    TUpdateHolderTargetPtr GetUpdateHolderTarget() const;

    // Box
    void SetBoxTarget(const TBoxTarget& target);
    // Return target with empty id if no target
    TBoxTarget GetAndRemoveBoxTarget(const TString& boxId);

    // Layer
    void SetLayerTarget(const TLayerTarget& target);
    // Return target with empty id if no target
    TLayerTarget GetAndRemoveLayerTarget(const TString& layerId);

    // StaticResource
    void SetStaticResourceTarget(const TStaticResourceTarget& target);
    // Return target with empty id if no target
    TStaticResourceTarget GetAndRemoveStaticResourceTarget(const TString& staticResourceId);

    // Volume
    void SetVolumeTarget(const TVolumeTarget& target);
    // Return target with empty id if no target
    TVolumeTarget GetAndRemoveVolumeTarget(const TString& volumeId);

    // Workload
    void SetWorkloadTarget(const TWorkloadTarget& target);
    // Return target with empty id if no target
    TWorkloadTarget GetAndRemoveWorkloadTarget(const TString& workloadId);

private:
    TUpdateHolderTargetPtr UpdateHolderTarget_;

    // Box
    TMutex BoxMutex_;
    TMap<TString, TBoxTarget> BoxTargets_;

    // Layer
    TMutex LayerMutex_;
    TMap<TString, TLayerTarget> LayerTargets_;

    // StaticResource
    TMutex StaticResourceMutex_;
    TMap<TString, TStaticResourceTarget> StaticResourceTargets_;

    // Volume
    TMutex VolumeMutex_;
    TMap<TString, TVolumeTarget> VolumeTargets_;

    // Workload
    TMutex WorkloadMutex_;
    TMap<TString, TWorkloadTarget> WorkloadTargets_;
};

using TUpdateHolderPtr = TIntrusivePtr<TUpdateHolder>;

} // namespace NInfra::NPodAgent
