#pragma once

#include "client.h"

namespace NInfra::NPodAgent {

using NThreading::TFuture;

class TAsyncPortoClient;
using TAsyncPortoClientPtr = TIntrusivePtr<TAsyncPortoClient>;

/*
    Runs calls to porto in a given MtpQueue
*/
class TAsyncPortoClient: public TAtomicRefCount<TAsyncPortoClient> {
public:
    TAsyncPortoClient(TPortoClientPtr client, TAtomicSharedPtr<IThreadPool> mtpQueue)
        : Client_(client)
        , MtpQueuePtr_(mtpQueue)
    {
        Y_ENSURE(MtpQueuePtr_, "MtpQueue not defined for TAsyncPortoClient");
    }

    TAsyncPortoClientPtr SwitchLogFrame(TLogFramePtr logFrame) {
        return new TAsyncPortoClient(Client_->SwitchLogFrame(logFrame), MtpQueuePtr_);
    }

    TFuture<TExpected<void, TPortoError>> Create(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Create(name);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<void, TPortoError>> CreateRecursive(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->CreateRecursive(name);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<void, TPortoError>> Destroy(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Destroy(name);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<int, TPortoError>> IsContainerExists(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->IsContainerExists(name);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<void, TPortoError>> Start(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Start(name);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> Stop(const TPortoContainerName& name, TDuration timeout = TDuration::Seconds(30)) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Stop(name, timeout);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> Kill(const TPortoContainerName& name, int sig) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Kill(name, sig);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> Pause(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Pause(name);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> Resume(const TPortoContainerName& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Resume(name);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TString, TPortoError>> WaitContainers(const TVector<TPortoContainerName>& containers, TDuration timeout) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->WaitContainers(containers, timeout);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TVector<TPortoContainerName>, TPortoError>> List(const TString& mask = "") {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->List(mask);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError>> Get(const TVector<TPortoContainerName>& name, const TVector<EPortoContainerProperty>& variable) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->Get(name, variable);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TString, TPortoError>> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int flags = 0) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->GetProperty(name, property, flags);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& value) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->SetProperty(name, property, value);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<void, TPortoError>> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->SetProperties(name, properties);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TString, TPortoError>> GetStdout(const TPortoContainerName& name, int offset, int length, int flags = 0) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->GetStdout(name, offset, length, flags);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TString, TPortoError>> GetStderr(const TPortoContainerName& name, int offset, int length, int flags = 0) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->GetStderr(name, offset, length, flags);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture <TExpected<TString, TPortoError>> CreateVolume(
        const TString& path
        , const TString& storage
        , const TString& place
        , const TVector<TString>& layers
        , unsigned long long quotaBytes
        , const TString& privateValue
        , const EPortoVolumeBackend backend
        , const TPortoContainerName& containerName
        , const TVector<TPortoVolumeShare>& staticResources
        , bool readOnly
    ) {
        auto client = Client_;
        return NThreading::Async(
             [=] () {
                return client->CreateVolume(
                    path
                    , storage
                    , place
                    , layers
                    , quotaBytes
                    , privateValue
                    , backend
                    , containerName
                    , staticResources
                    , readOnly
                );
             }
             , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->LinkVolume(path, container, target, readOnly, required);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict = false) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->UnlinkVolume(path, container, target, strict);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<TVector<TPortoVolume>, TPortoError>> ListVolumes(const TString& path = "", const TPortoContainerName& container = TPortoContainerName("")) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ListVolumes(path, container);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<TVector<TString>, TPortoError>> ListVolumesPaths(const TString& path = "", const TPortoContainerName& container = TPortoContainerName("")) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ListVolumesPaths(path, container);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<int, TPortoError>> IsVolumeExists(const TString& path) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->IsVolumeExists(path);
            }
            , *MtpQueuePtr_
        );
    };

    TFuture<TExpected<void, TPortoError>> ImportLayer(const TString& layer, const TString& tarball, bool merge = false, const TString& place = "", const TString& privateValue = "") {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ImportLayer(layer, tarball, merge, place, privateValue);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> RemoveLayer(const TString& layer, const TString& place) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->RemoveLayer(layer, place);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<TVector<TPortoLayer>, TPortoError>> ListLayers(const TString& place, const TString& mask = "") {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ListLayers(place, mask);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TString, TPortoError>> GetLayerPrivate(const TString& layer, const TString& place) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->GetLayerPrivate(layer, place);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->SetLayerPrivate(privateValue, layer, place);
            }
            , *MtpQueuePtr_
        );
    }

    TFuture<TExpected<TVector<TPortoStorage>, TPortoError>> ListStorages(const TString& place, const TString& mask) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ListStorages(place, mask);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<TString, TPortoError>> GetStoragePrivate(const TString& place, const TString& name) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->GetStoragePrivate(place, name);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> RemoveStorage(const TString& name, const TString& place) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->RemoveStorage(name, place);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<void, TPortoError>> ImportStorage(const TString& name, const TString& archive, const TString& place, const TString& compression, const TString& privateValue) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->ImportStorage(name, archive, place, compression, privateValue);
            }
            , *MtpQueuePtr_
        );
    }
    TFuture<TExpected<bool, TPortoError>> IsStorageExists(const TString& place, const TString& mask) {
        auto client = Client_;
        return NThreading::Async(
            [=] () {
                return client->IsStorageExists(place, mask);
            }
            , *MtpQueuePtr_
        );
    }
private:
    TPortoClientPtr Client_;
    TAtomicSharedPtr<IThreadPool> MtpQueuePtr_;
};

} // namespace NInfra::NPodAgent
