#pragma once

#include "porto_types.h"
#include "property.h"

#include <infra/libs/outcome/result.h>

#include <infra/libs/logger/logger.h>

#include <infra/porto/api/libporto.hpp>
#include <infra/porto/proto/rpc.pb.h>

#include <util/generic/ptr.h>
#include <util/stream/printf.h>
#include <util/string/builder.h>
#include <util/thread/lfstack.h>

namespace NInfra::NPodAgent {

const int PORTO_CLIENT_DEFAULT_TIMEOUT = 60; // seconds

class IPortoClient;
using TPortoClientPtr = TIntrusivePtr<IPortoClient>;

class IPortoClient: public TAtomicRefCount<IPortoClient> {
public:
    virtual TPortoClientPtr SwitchLogFrame(TLogFramePtr logFrame) = 0;

    virtual TExpected<void, TPortoError> Create(const TPortoContainerName& name) = 0;
    virtual TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) = 0;
    virtual TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) = 0;
    virtual TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& name) = 0;

    virtual TExpected<void, TPortoError> Start(const TPortoContainerName& name) = 0;
    virtual TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration timeout = TDuration::Seconds(30)) = 0;
    virtual TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int sig) = 0;
    virtual TExpected<void, TPortoError> Pause(const TPortoContainerName& name) = 0;
    virtual TExpected<void, TPortoError> Resume(const TPortoContainerName& name) = 0;

    virtual TExpected<TString, TPortoError> WaitContainers(const TVector<TPortoContainerName>& containers, TDuration timeout = TDuration::Seconds(30)) = 0;

    virtual TExpected<TVector<TPortoContainerName>, TPortoError> List(const TString& mask = "") = 0;

    virtual TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(const TVector<TPortoContainerName>& name, const TVector<EPortoContainerProperty>& variable) = 0;

    virtual TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int flags = 0) = 0;
    virtual TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& value) = 0;
    virtual TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) = 0;

    virtual TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset = -1, int length = -1, int flags = 0) = 0;
    virtual TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int offset = -1, int length = -1, int flags = 0) = 0;
    virtual TExpected<TString, TPortoError> GetStream(const TPortoContainerName& name, const TString& stream, int offset = -1, int length = -1, int flags = 0) = 0;

    virtual TExpected <TString, TPortoError> CreateVolume(
        const TString& path
        , const TString& storage
        , const TString& place
        , const TVector<TString>& layers
        , unsigned long long quotaBytes
        , const TString& privateValue
        , const EPortoVolumeBackend backend
        , const TPortoContainerName& containerName
        , const TVector<TPortoVolumeShare>& staticResources
        , bool readOnly
    ) = 0;

    virtual TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container = TPortoContainerName(""), const TString& target = "", bool readOnly = false, bool required = false) = 0;
    virtual TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container = TPortoContainerName(""), const TString& target = "***", bool strict = false) = 0;
    virtual TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path = "", const TPortoContainerName& container = TPortoContainerName("")) = 0;
    virtual TExpected<TVector<TString>, TPortoError> ListVolumesPaths(const TString& path = "", const TPortoContainerName& container = TPortoContainerName("")) = 0;
    virtual TExpected<int, TPortoError> IsVolumeExists(const TString& path) = 0;

    virtual TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool merge = false, const TString& place = "", const TString& privateValue = "") = 0;
    virtual TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place = "") = 0;
    virtual TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place = "", const TString& mask = "") = 0;

    virtual TExpected<TString, TPortoError> GetLayerPrivate(const TString& layer, const TString& place = "") = 0;
    virtual TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place = "") = 0;

    virtual TExpected<TVector<TPortoStorage>, TPortoError> ListStorages(const TString& place, const TString& mask) = 0;
    virtual TExpected<TString, TPortoError> GetStoragePrivate(const TString& place, const TString& name) = 0;
    virtual TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place = "") = 0;
    virtual TExpected<void, TPortoError> ImportStorage(const TString& name, const TString& archive, const TString& place = "", const TString& compression = "", const TString& privateValue = "") = 0;
    virtual TExpected<bool, TPortoError> IsStorageExists(const TString& place, const TString& name) = 0;

    virtual ~IPortoClient() = default;
};

// auto-destroyable container - useful for tests as a root container
class TLocalContainer;
using TLocalContainerPtr = TIntrusivePtr<TLocalContainer>;

class TLocalContainer: TNonCopyable, public TAtomicRefCount<TLocalContainer> {
public:
    TLocalContainer(TPortoClientPtr porto, const TPortoContainerName& name)
        : Porto_(porto)
        , Name_(name)
    {
        Porto_->Create(Name_).Success();
    }

    ~TLocalContainer() {
        auto result = Porto_->Destroy(Name_);
        if (!result) {
            Cerr << "Failed to destroy porto container '" << TString(Name_) << "': " << result.Error().Message << Endl;
        }
    }

    operator TPortoContainerName() {
        return Name_;
    }

private:
    TPortoClientPtr Porto_;
    TPortoContainerName Name_;
};

// auto-destroyable volume - useful for tests
class TLocalVolume: TNonCopyable {
public:
    TLocalVolume(TPortoClientPtr porto, size_t quotaBytes = 0)
        : Porto_(porto)
    {
        VolumePath_ = Porto_->CreateVolume("", "", "", {}, quotaBytes, "", EPortoVolumeBackend::Auto, {""}, {}, false).Success();
    }

    ~TLocalVolume() {
        auto result = Porto_->UnlinkVolume(VolumePath_);
        if (!result) {
            Cerr << "Failed to destroy porto volume '" << VolumePath_ << "': " << result.Error().Message << Endl;
        }
    }

    const TString& GetPath() const {
        return VolumePath_;
    }

private:
    TPortoClientPtr Porto_;
    TString VolumePath_;
};

} // namespace NInfra::NPodAgent
