#pragma once

#include "client.h"

namespace NInfra::NPodAgent {

/*
    Using only in tests

    Adds given prefix to every container operation
    Useful for resource cleaning - just destroy your 'root' container
    So:
      if prefix is set to "root"
      calling 'Create("name")' will create container "root/name"
*/
class TNestedPortoClient: public IPortoClient {
public:
    TNestedPortoClient(TPortoClientPtr client, const TPortoContainerName& prefix)
        : Client_(client)
        , Prefix_(prefix)
    {
    }

    TPortoClientPtr SwitchLogFrame(TLogFramePtr logFrame) override {
        return new TNestedPortoClient(Client_->SwitchLogFrame(logFrame), Prefix_);
    }

    TExpected<void, TPortoError> Create(const TPortoContainerName& name) override {
        return Client_->Create(GetRealContainerName(name));
    }
    TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
        return Client_->CreateRecursive(GetRealContainerName(name));
    }
    TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
        return Client_->Destroy(GetRealContainerName(name));
    }

    TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& name) override {
        return Client_->IsContainerExists(GetRealContainerName(name));
    }

    TExpected<void, TPortoError> Start(const TPortoContainerName& name) override {
        return Client_->Start(GetRealContainerName(name));
    }
    TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration timeout) override {
        return Client_->Stop(GetRealContainerName(name), timeout);
    }
    TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int sig) override {
        return Client_->Kill(GetRealContainerName(name), sig);
    }
    TExpected<void, TPortoError> Pause(const TPortoContainerName& name) override {
        return Client_->Pause(GetRealContainerName(name));
    }
    TExpected<void, TPortoError> Resume(const TPortoContainerName& name) override {
        return Client_->Resume(GetRealContainerName(name));
    }

    TExpected<TString, TPortoError> WaitContainers(const TVector<TPortoContainerName>& containers, TDuration timeout) override {
        TVector<TPortoContainerName> trueContainers;
        for (auto& it : containers) {
            trueContainers.push_back(GetRealContainerName(it));
        }
        return Client_->WaitContainers(trueContainers, timeout);
    }

    TExpected<TVector<TPortoContainerName>, TPortoError> List(const TString& mask) override {
        Y_ENSURE(mask.empty(), "mask is not supported for TNestedPortoClient.List()");

        TVector<TPortoContainerName> result;
        auto list = OUTCOME_TRYX(Client_->List(mask));
        TString needPathPrefix = TString(Prefix_) + '/';
        for (auto& c : list) {
            TString path = (TString)c;
            if (path.StartsWith(needPathPrefix)) {
                result.push_back(TPortoContainerName::NoEscape(path.substr(needPathPrefix.length())));
            }
        }
        return result;
    }

    TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(const TVector<TPortoContainerName>& name, const TVector<EPortoContainerProperty>& variable) override {
        TVector<TPortoContainerName> trueName;
        for (auto& it : name) {
            trueName.push_back(GetRealContainerName(it));
        }

        TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
        TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> it = OUTCOME_TRYX(Client_->Get(trueName, variable));
        for (auto& container : it) {
            result[container.first.CutPrefix(Prefix_)] = container.second;
        }
        return result;
    }

    TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int flags) override {
        return Client_->GetProperty(GetRealContainerName(name), property, flags);
    }

    TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& value) override {
        return Client_->SetProperty(GetRealContainerName(name), property, value);
    }

    TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override {
        return Client_->SetProperties(GetRealContainerName(name), properties);
    }

    TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int flags) override {
        return Client_->GetStdout(GetRealContainerName(name), offset, length, flags);
    }

    TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int offset, int length, int flags) override {
        return Client_->GetStderr(GetRealContainerName(name), offset, length, flags);
    }

    TExpected<TString, TPortoError> GetStream(const TPortoContainerName& name, const TString& stream, int offset, int length, int flags) override {
        return Client_->GetStream(GetRealContainerName(name), stream, offset, length, flags);
    }

    TExpected <TString, TPortoError> CreateVolume(
        const TString& path
        , const TString& storage
        , const TString& place
        , const TVector<TString>& layers
        , unsigned long long quotaBytes
        , const TString& privateValue
        , const EPortoVolumeBackend backend
        , const TPortoContainerName& containerName
        , const TVector<TPortoVolumeShare>& staticResources
        , bool readOnly
    ) override {
        return Client_->CreateVolume(
            path
            , storage
            , place
            , layers
            , quotaBytes
            , privateValue
            , backend
            , GetRealContainerName(containerName)
            , staticResources
            , readOnly
        );
    }
    TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) override {
        return Client_->LinkVolume(path, GetRealContainerName(container), target, readOnly, required);
    }
    TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict) override {
        return Client_->UnlinkVolume(path, GetRealContainerName(container), target, strict);
    }
    TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override {
        auto result = OUTCOME_TRYX(Client_->ListVolumes(path, GetRealContainerName(container)));

        TString rootContainer = TString(Prefix_);
        TString needPathPrefix = TString(Prefix_) + '/';
        for (auto& volume : result) {
            volume.mutable_links()->erase(
                std::remove_if(
                    volume.mutable_links()->begin()
                    , volume.mutable_links()->end()
                    , [rootContainer, needPathPrefix](TPortoVolumeLink link) { return !(link.container() == rootContainer || link.container().StartsWith(needPathPrefix)); }
                )
                , volume.mutable_links()->end()
            );

            for (auto& link : *volume.mutable_links()) {
                if (link.container() == rootContainer) {
                    link.set_container("");
                } else {
                    link.set_container(link.container().substr(needPathPrefix.size()));
                }
            }
        }

        return result;
    }

    TExpected<TVector<TString>, TPortoError> ListVolumesPaths(const TString& path, const TPortoContainerName& container) override {
        auto result = OUTCOME_TRYX(Client_->ListVolumesPaths(path, GetRealContainerName(container)));
        return result;
    }
    TExpected<int, TPortoError> IsVolumeExists(const TString& path) override {
        return Client_->IsVolumeExists(path);
    };

    TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool merge, const TString& place, const TString& privateValue) override {
        return Client_->ImportLayer(layer, tarball, merge, place, privateValue);
    }
    TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override {
        return Client_->RemoveLayer(layer, place);
    }
    TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
        return Client_->ListLayers(place, mask);
    }

    TExpected<TString, TPortoError> GetLayerPrivate(const TString& layer, const TString& place) override {
        return Client_->GetLayerPrivate(layer, place);
    }
    TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) override {
        return Client_->SetLayerPrivate(privateValue, layer, place);
    }

    TExpected<TVector<TPortoStorage>, TPortoError> ListStorages(const TString& place, const TString& mask) override {
        return Client_->ListStorages(place, mask);
    };

    TExpected<TString, TPortoError> GetStoragePrivate(const TString& place, const TString& name) override {
        return Client_->GetStoragePrivate(place, name);
    }
    TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place) override {
        return Client_->RemoveStorage(name, place);
    }
    TExpected<void, TPortoError> ImportStorage(const TString& name, const TString& archive, const TString& place, const TString& compression, const TString& privateValue) override {
        return Client_->ImportStorage(name, archive, place, compression, privateValue);
    }

    TExpected<bool, TPortoError> IsStorageExists(const TString& place, const TString& mask) override {
        return Client_->IsStorageExists(place, mask);
    };

private:
    TPortoContainerName GetRealContainerName(const TPortoContainerName& name) const {
        if (name == TPortoContainerName::NoEscape("***") || name == TPortoContainerName::NoEscape("/")) {
            return name;
        } else if (name == TPortoContainerName::NoEscape(".") || name == TPortoContainerName::NoEscape("") || name == TPortoContainerName::NoEscape("///")) {
            return Prefix_;
        } else {
            return {Prefix_, name};
        }
    }

private:
    TPortoClientPtr Client_;
    TPortoContainerName Prefix_;
};

} // namespace NInfra::NPodAgent
