#include "nested_client.h"
#include "simple_client.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/thread.h>

namespace NInfra::NPodAgent::NPortoTest {

Y_UNIT_TEST_SUITE(NestedPortoClient) {

Y_UNIT_TEST(NestedPortoClientCreateContainer) {
    TPortoClientPtr rootClient = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer rootContainer(rootClient, GetTestPrefix() + "NestedPortoClient");
    TPortoContainerName container("test");

    {
        TPortoClientPtr nestedClient = new TNestedPortoClient(rootClient, rootContainer);
        nestedClient->Create(container).Success();

        TVector<TPortoContainerName> list = nestedClient->List().Success();
        UNIT_ASSERT_EQUAL_C(list.size(), 1, "expected one container at list, got " << list.size());
        UNIT_ASSERT_C(list[0] == container, "container not found");

        auto res = nestedClient->Get({container}, {EPortoContainerProperty::State}).Success();
        auto* ptr = res.FindPtr(container);
        UNIT_ASSERT_C(ptr, "container " << TString(container) << " not found");
        auto* valPtr = ptr->FindPtr(EPortoContainerProperty::State);
        UNIT_ASSERT_C(valPtr, "property " << EPortoContainerProperty::State << " not found");
        UNIT_ASSERT_EQUAL_C( valPtr->value(), ToString(EPortoContainerState::Stopped), "unexpected container state");
    }

    {
        TVector<TPortoContainerName> list = rootClient->List().Success();
        bool found = false;
        for (auto& it : list) {
            found |= (it == TPortoContainerName(rootContainer, container));
        }
        UNIT_ASSERT_C(found, "container not found");
    }
}

Y_UNIT_TEST(NestedPortoClientLinkVolumeWithTargetContainer) {
    TPortoClientPtr rootClient = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer rootContainer(rootClient, GetTestPrefix() + "NestedPortoClient");
    TPortoClientPtr nestedClient = new TNestedPortoClient(rootClient, rootContainer);

    TLocalVolume volume(nestedClient);
    TLocalContainer container(nestedClient, {"some_container"});
    {
        auto result = nestedClient->LinkVolume(volume.GetPath(), container);
        UNIT_ASSERT_C(result, result.Error().Message);
    }
    {
        auto result = nestedClient->ListVolumes(volume.GetPath(), container);
        UNIT_ASSERT_C(result, result.Error().Message);
        auto volumeVector = result.Success();
        UNIT_ASSERT_EQUAL(1, volumeVector.size());
        TPortoVolume& sameVolume = volumeVector[0];
        UNIT_ASSERT_EQUAL(volume.GetPath(), sameVolume.path());
        bool found = false;
        for (auto& link : sameVolume.links()) {
            found |= (link.container() == TString(TPortoContainerName(container)));
        }
        UNIT_ASSERT_C(found, "container not found in links");
    }
}

Y_UNIT_TEST(NestedPortoClientIsContainerExists) {
    const TPortoContainerName rootContainerName(GetTestPrefix() + "NestedPortoClientIsContainerExists");
    const TPortoContainerName nestedContainerName("NestedContainer");
    TPortoClientPtr rootClient = NPortoTestLib::GetSimplePortoClient();
    TPortoClientPtr nestedClient = new TNestedPortoClient(rootClient, rootContainerName);
    TLocalContainer rootContainer(rootClient, rootContainerName);

    auto result = nestedClient->List();
    UNIT_ASSERT(result);
    TVector<TPortoContainerName>& list = result.Success();
    UNIT_ASSERT(Find(list.begin(), list.end(), nestedContainerName) == list.end());

    auto existResult = nestedClient->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(existResult);
    UNIT_ASSERT(!existResult.Success());

    TLocalContainer nestedContainer(nestedClient, nestedContainerName);

    result = nestedClient->List();
    UNIT_ASSERT(result);
    list = result.Success();
    UNIT_ASSERT(Find(list.begin(), list.end(), nestedContainerName) != list.end());

    existResult = nestedClient->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(existResult);
    UNIT_ASSERT(existResult.Success());
}

Y_UNIT_TEST(NestedPortoClientCreateRecursive) {
    const TPortoContainerName rootContainerName(GetTestPrefix() + "NestedPortoClientCreateRecursive");
    const TPortoContainerName nestedContainerName("NestedContainer");
    TPortoClientPtr rootClient = NPortoTestLib::GetSimplePortoClient();
    TPortoClientPtr nestedClient = new TNestedPortoClient(rootClient, rootContainerName);

    auto result = nestedClient->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(result);
    UNIT_ASSERT(!(result.Success()));

    UNIT_ASSERT(nestedClient->CreateRecursive(nestedContainerName));

    result = nestedClient->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(result);
    UNIT_ASSERT(result.Success());

    UNIT_ASSERT(rootClient->Destroy(rootContainerName));
}

}

} // namespace NInfra::NPodAgent::NPortoTest
