#include "porto_request_response_converter.h"

#include <infra/pod_agent/libs/util/string_utils.h>
#include <infra/porto/proto/rpc.pb.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NInfra::NPodAgent::NPortoRequestResponseConverter  {

Porto::TVolumeSpec ConvertVolumeCreationRequest(
    const TString& path
    , const TString& storage
    , const TString& place
    , const TVector<TString>& layers
    , unsigned long long quotaBytes
    , const TString& privateValue
    , const EPortoVolumeBackend backend
    , const TPortoContainerName& containerName
    , const TVector<TPortoVolumeShare>& staticResources
    , bool readOnly
) {
    Porto::TVolumeSpec volumeSpec;

    if (!path.empty()) {
        volumeSpec.set_path(path.c_str());
    }

    if (!storage.empty()) {
        volumeSpec.set_storage(storage.c_str());
    }

    if (!place.empty()) {
        volumeSpec.set_place(place.c_str());
    }

    for (const auto& staticResource: staticResources) {
        auto share = volumeSpec.add_shares();
        share->set_path(staticResource.Path);
        share->set_origin_path(staticResource.OriginPath);
        share->set_cow(staticResource.CopyOnWrite);
    }

    if (!privateValue.empty()) {
        volumeSpec.set_private_value(privateValue);
    }

    if (!layers.empty()) {
        for(const auto& layer: layers) {
            volumeSpec.add_layers(layer);
        }
    }

    if (quotaBytes) {
        Porto::TVolumeResource *spaceQuota = volumeSpec.mutable_space();
        spaceQuota->set_limit(quotaBytes);
        spaceQuota->set_guarantee(quotaBytes);
    }

    volumeSpec.set_backend(ToString(backend));

    if (!TString(containerName).empty()) {
        auto linkToAdd = volumeSpec.add_links();
        linkToAdd->set_container(TString(containerName));
    }

    if (readOnly) {
        volumeSpec.set_read_only(readOnly);
    }

    return volumeSpec;
}

Porto::TGetVolumeRequest ConvertGetVolumeRequest(
    const TVector<TString>& paths
    , const TPortoContainerName& containerName
) {
    Porto::TGetVolumeRequest getVolumeRequest;

    for (const auto& path : paths) {
        if (!path.empty()) {
            getVolumeRequest.add_path(path);
        }
    }

    if (!TString(containerName).empty()) {
        getVolumeRequest.set_container(TString(containerName));
    }

    return getVolumeRequest;
}

TString ConvertVolumeCreationResponse(const Porto::TVolumeSpec& volumeSpec) {
    if (volumeSpec.has_path()) {
        return volumeSpec.path();
    }
    return "";
}

TExpected<void, TPortoError> ConvertContainerEnv(const TString& envStr, Porto::TContainerEnv* containerEnv, bool secret) {
    TVector<TString> envs = SplitEscaped(envStr, ';');
    for (const auto& env : envs) {
        size_t sep = env.find('=');

        if (sep == TString::npos) {
            TString message;
            if (secret) {
                message = "Secret env properties have unexpected format";
            } else {
                message = "Env properties have unexpected format: " + envStr;
            }
            return TPortoError{.Code=EPortoError::InvalidValue, .Message=message};
        }
        Porto::TContainerEnvVar* envVar = containerEnv->add_var();
        envVar->set_name(Trim(env.substr(0, sep)));
        envVar->set_value(Trim(env.substr(sep + 1)));
    }
    return TExpected<void, TPortoError>::DefaultSuccess();
}

TExpected<Porto::TContainerSpec, TPortoError> ConvertPropertiesToProtoContainerSpec(const TPortoContainerName& containerName, const TMap<EPortoContainerProperty, TString>& properties) {
    Porto::TContainerSpec spec;
    spec.set_name(ToString(containerName));
    for (const auto& [property, value] : properties) {
        switch (property) {
            case EPortoContainerProperty::Command:
                spec.set_command(value);
                break;
            case EPortoContainerProperty::CoreCommand:
                spec.set_core_command(value);
                break;
            case EPortoContainerProperty::EnvSecret:
                OUTCOME_TRYV(ConvertContainerEnv(value, spec.mutable_env_secret(), /* secret = */ true));
                break;
            case EPortoContainerProperty::Cgroupfs:
                spec.set_cgroupfs(value);
                break;
            // TODO(DEPLOY-3231): Implement all properties from TreeCheckAndCreateContainer
            default:
                return TPortoError{
                    .Code = EPortoError::InvalidValue,
                    .Message="Property " + ToString(property) + " has not been supported for TContainerUpdateSpec request",
                };
        }
    }
    return spec;
}

} // namespace NInfra::NPodAgent::NPortoRequestResponseConverter
