#include "porto_request_response_converter.h"

#include <infra/porto/proto/rpc.pb.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestPortoRequestResponseConverter {

Y_UNIT_TEST_SUITE(PortoRequestResponseConverter) {

Y_UNIT_TEST(TestConvertVolumeCreationRequestWithNotEmptyFields) {
    const TString path = "path";
    const TString storage = "my_storage";
    const TString place = "my_place";
    const TVector<TString> layers = {"layer1", "layer2"};
    const unsigned long long quotaBytes = 1 << 30;
    const TString privateValue = "private_value";
    const EPortoVolumeBackend backend = EPortoVolumeBackend::Bind;
    const TPortoContainerName containerName = {"container_name"};
    const TVector<TPortoVolumeShare> staticResources = {
        {"resource_path1", "origin_path1", true}
        , {"resource_path2", "origin_path2", false}
    };
    const bool readOnly = true;

    Porto::TVolumeSpec volumeSpec = NPortoRequestResponseConverter::ConvertVolumeCreationRequest(
        path
        , storage
        , place
        , layers
        , quotaBytes
        , privateValue
        , backend
        , containerName
        , staticResources
        , readOnly
    );

    UNIT_ASSERT_EQUAL(TString(volumeSpec.path()), path);
    UNIT_ASSERT_EQUAL(TString(volumeSpec.storage()), storage);
    UNIT_ASSERT_EQUAL(TString(volumeSpec.place()), place);
    UNIT_ASSERT_EQUAL(TString(*volumeSpec.mutable_layers(0)), layers.at(0));
    UNIT_ASSERT_EQUAL(TString(*volumeSpec.mutable_layers(1)), layers.at(1));
    UNIT_ASSERT_EQUAL(TString(volumeSpec.private_value()), privateValue);
    UNIT_ASSERT_EQUAL(TString(volumeSpec.backend()), ToString(backend));
    UNIT_ASSERT_EQUAL(TString(volumeSpec.links(0).container()), TString(containerName));
    UNIT_ASSERT_EQUAL(volumeSpec.space().guarantee(), quotaBytes);
    UNIT_ASSERT_EQUAL(volumeSpec.space().limit(), quotaBytes);
    UNIT_ASSERT_EQUAL(TString(volumeSpec.shares(0).path()), "resource_path1");
    UNIT_ASSERT_EQUAL(TString(volumeSpec.shares(0).origin_path()), "origin_path1");
    UNIT_ASSERT_EQUAL(volumeSpec.shares(0).cow(), true);
    UNIT_ASSERT_EQUAL(TString(volumeSpec.shares(1).path()), "resource_path2");
    UNIT_ASSERT_EQUAL(TString(volumeSpec.shares(1).origin_path()), "origin_path2");
    UNIT_ASSERT_EQUAL(volumeSpec.shares(1).cow(), false);
    UNIT_ASSERT_EQUAL(volumeSpec.read_only(), readOnly);
}

Y_UNIT_TEST(TestConvertVolumeCreationRequestWithEmptyFields) {
    const TString path = "";
    const TString storage = "";
    const TString place = "";
    const TVector<TString> layers;
    const unsigned long long quotaBytes = 0;
    const TString privateValue = "";
    const EPortoVolumeBackend backend = EPortoVolumeBackend::Auto;
    const TPortoContainerName containerName = {""};
    const TVector<TPortoVolumeShare> staticResources;
    const bool readOnly = false;

    Porto::TVolumeSpec volumeSpec = NPortoRequestResponseConverter::ConvertVolumeCreationRequest(
        path
        , storage
        , place
        , layers
        , quotaBytes
        , privateValue
        , backend
        , containerName
        , staticResources
        , readOnly
    );

    UNIT_ASSERT(!volumeSpec.has_path());
    UNIT_ASSERT(!volumeSpec.has_storage());
    UNIT_ASSERT(!volumeSpec.has_place());
    UNIT_ASSERT(!volumeSpec.layers_size());
    UNIT_ASSERT(!volumeSpec.has_space());
    UNIT_ASSERT(!volumeSpec.has_private_value());
    UNIT_ASSERT(volumeSpec.has_backend());
    UNIT_ASSERT(!volumeSpec.links_size());
    UNIT_ASSERT(!volumeSpec.shares_size());
    UNIT_ASSERT(!volumeSpec.has_read_only());
}

Y_UNIT_TEST(TestConvertGetVolumeRequest) {
    const TVector<TString> paths = {"", "path0", "", "", "path1", "path2"};
    const TString containerName = "container_name";

    Porto::TGetVolumeRequest getVolumeRequest = NPortoRequestResponseConverter::ConvertGetVolumeRequest(
        paths
        , containerName
    );

    UNIT_ASSERT_EQUAL(getVolumeRequest.path_size(), 3);
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(0), "path0");
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(1), "path1");
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(2), "path2");
    UNIT_ASSERT_EQUAL(getVolumeRequest.container(), "container_name");
    UNIT_ASSERT(!getVolumeRequest.has_changed_since());
    UNIT_ASSERT_EQUAL(getVolumeRequest.label_size(), 0);

    getVolumeRequest = NPortoRequestResponseConverter::ConvertGetVolumeRequest(
        paths
    );

    UNIT_ASSERT_EQUAL(getVolumeRequest.path_size(), 3);
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(0), "path0");
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(1), "path1");
    UNIT_ASSERT_EQUAL(getVolumeRequest.path(2), "path2");
    UNIT_ASSERT_EQUAL(getVolumeRequest.container(), "");
    UNIT_ASSERT(!getVolumeRequest.has_changed_since());
    UNIT_ASSERT_EQUAL(getVolumeRequest.label_size(), 0);
}

Y_UNIT_TEST(TestConvertVolumeCreationResponse) {
    const TString path = "path";

    Porto::TVolumeSpec volumeSpec;
    volumeSpec.set_path(path.c_str());
    TString pathFromResponse = NPortoRequestResponseConverter::ConvertVolumeCreationResponse(volumeSpec);

    UNIT_ASSERT_EQUAL(pathFromResponse, path);
}

Y_UNIT_TEST(TestConvertPropertiesToProtoContainerSpec) {
    const TPortoContainerName container("my_container");

    auto containerSpecRes = NPortoRequestResponseConverter::ConvertPropertiesToProtoContainerSpec(container, {});
    UNIT_ASSERT(containerSpecRes);
    UNIT_ASSERT_EQUAL(containerSpecRes.Success().name(), "my_container");

    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "/command"}
        , {EPortoContainerProperty::CoreCommand, "/core_command"}
        , {EPortoContainerProperty::EnvSecret, "Key1=value1;Key2=value2"}
    };

    containerSpecRes = NPortoRequestResponseConverter::ConvertPropertiesToProtoContainerSpec(container, properties);
    UNIT_ASSERT(containerSpecRes);
    auto containerSpec = containerSpecRes.Success();
    UNIT_ASSERT_EQUAL(containerSpec.name(), "my_container");
    UNIT_ASSERT_EQUAL(containerSpec.command(), "/command");
    UNIT_ASSERT_EQUAL(containerSpec.core_command(), "/core_command");
    UNIT_ASSERT(containerSpec.has_env_secret());
    UNIT_ASSERT(!containerSpec.env_secret().has_merge());
    UNIT_ASSERT_EQUAL(containerSpec.env_secret().var_size(), 2);
    UNIT_ASSERT_EQUAL(containerSpec.env_secret().var(0).name(), "Key1");
    UNIT_ASSERT_EQUAL(containerSpec.env_secret().var(0).value(), "value1");
    UNIT_ASSERT_EQUAL(containerSpec.env_secret().var(1).name(), "Key2");
    UNIT_ASSERT_EQUAL(containerSpec.env_secret().var(1).value(), "value2");
    for (int i = 0; i < 2; ++i) {
        UNIT_ASSERT(!containerSpec.env_secret().var(i).has_unset());
        UNIT_ASSERT(!containerSpec.env_secret().var(i).has_salt());
        UNIT_ASSERT(!containerSpec.env_secret().var(i).has_hash());
    }
}

Y_UNIT_TEST(TestConvertPropertiesToProtoContainerSpecWithUnknownProperty) {
    const TPortoContainerName container("my_container");

    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::IoLimit, "123"}
    };
    auto containerSpecRes = NPortoRequestResponseConverter::ConvertPropertiesToProtoContainerSpec(container, properties);
    UNIT_ASSERT(!containerSpecRes);
    UNIT_ASSERT_EQUAL(containerSpecRes.Error().Code, EPortoError::InvalidValue);
    UNIT_ASSERT_EQUAL_C(containerSpecRes.Error().Message, "Property io_limit has not been supported for TContainerUpdateSpec request", containerSpecRes.Error().Message);
}

Y_UNIT_TEST(TestConvertContainerEnv) {
    Porto::TContainerEnv containerEnv;
    auto result = NPortoRequestResponseConverter::ConvertContainerEnv("Key1= value1 ; TestEnv = TestEnv\\;\\\\Value", &containerEnv, true);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT(!containerEnv.has_merge());
    UNIT_ASSERT_EQUAL(containerEnv.var_size(), 2);
    UNIT_ASSERT_EQUAL_C(containerEnv.var(0).name(), "Key1", containerEnv.var(0).name());
    UNIT_ASSERT_EQUAL_C(containerEnv.var(0).value(), "value1", containerEnv.var(0).value());
    UNIT_ASSERT_EQUAL_C(containerEnv.var(1).name(), "TestEnv", containerEnv.var(1).name());
    UNIT_ASSERT_EQUAL_C(containerEnv.var(1).value(), "TestEnv;\\Value", containerEnv.var(1).value());
    for (int i = 0; i < 2; ++i) {
        UNIT_ASSERT(!containerEnv.var(i).has_unset());
        UNIT_ASSERT(!containerEnv.var(i).has_salt());
        UNIT_ASSERT(!containerEnv.var(i).has_hash());
    }
}

Y_UNIT_TEST(TestConvertContainerEnvWithWrongFormat) {
    Porto::TContainerEnv containerEnv;
    auto result = NPortoRequestResponseConverter::ConvertContainerEnv("Key1 value1", &containerEnv, true);
    UNIT_ASSERT(!result);
    UNIT_ASSERT_EQUAL(result.Error().Code, EPortoError::InvalidValue);
    UNIT_ASSERT_EQUAL_C(result.Error().Message, "Secret env properties have unexpected format", result.Error().Message);

    result = NPortoRequestResponseConverter::ConvertContainerEnv("Key1=value1;Key2 Value2", &containerEnv, false);
    UNIT_ASSERT(!result);
    UNIT_ASSERT_EQUAL(result.Error().Code, EPortoError::InvalidValue);
    UNIT_ASSERT_EQUAL_C(result.Error().Message, "Env properties have unexpected format: Key1=value1;Key2 Value2", result.Error().Message);
}

}

} // namespace NInfra::NPodAgent::NTestPortoRequestResponseConverter
