#pragma once

#include <infra/libs/outcome/result.h>
#include <infra/porto/api/libporto.hpp>
#include <infra/porto/proto/rpc.pb.h>

#include <util/stream/printf.h>
#include <util/string/builder.h>

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

namespace NInfra::NPodAgent {

using EPortoError = Porto::EError;
struct TPortoError;

} // namespace NInfra::NPodAgent

inline TString ToString(NInfra::NPodAgent::EPortoError code);
inline TString ToString(const NInfra::NPodAgent::TPortoError& error);

namespace NInfra::NPodAgent {

using TPortoLayer = Porto::TLayer;
using TPortoStorage = Porto::TStorage;
using TPortoVolumeLink = Porto::TVolumeLink;
using TPortoVolume = Porto::TVolumeSpec;
using TPortoGetResponse = Porto::TGetResponse::TContainerGetValueResponse;
using TPortoGet = Porto::TGetResponse;
using TPortoListProperties = Porto::TListPropertiesResponse;
using TPortoVersion = Porto::TVersionResponse;
using TPortoProperty = Porto::TListPropertiesResponse::TContainerPropertyListEntry;

struct TPortoError {
    EPortoError Code = EPortoError::Success;
    TString Action = "";
    TString Args = "";
    TString Message = "";

    bool operator==(const TPortoError& other) const {
        return Code == other.Code
            && Action == other.Action
            && Args == other.Args
            && Message == other.Message;
    }
};

/* Escapes non-porto characters (including '/') */
class TPortoContainerName {
public:
    TPortoContainerName(const TString& name)
        : Name_(Escape(name))
    {
    }

    TPortoContainerName(const TPortoContainerName& otherA, const TPortoContainerName& otherB)
        : Name_(TString(otherA) + '/' + otherB)
    {
    }

    TPortoContainerName(const TPortoContainerName& otherA, const TString& name)
        : TPortoContainerName(otherA, TPortoContainerName(name))
    {
    }

    operator TString() const {
        return Name_;
    }

    bool empty() const {
        return Name_.empty();
    }

    bool operator==(const TPortoContainerName& other) const {
        return Name_ == other.Name_;
    }

    bool operator!=(const TPortoContainerName& other) const {
        return Name_ != other.Name_;
    }

    bool operator!=(const TString& stringName) const {
        return Name_ != stringName;
    }

    bool operator<(const TPortoContainerName& other) const {
        return Name_ < other.Name_;
    }

    friend IOutputStream& operator<<(IOutputStream& stream, const TPortoContainerName& container) {
        stream << TString(container);
        return stream;
    }

    bool IsChildOf(const TPortoContainerName& container) const {
        return Name_.StartsWith(container.Name_ + '/') && Name_.find('/', container.Name_.size() + 1) == TString::npos;
    }

    bool IsSimple() const {
        return (Name_.find('/') == TString::npos);
    }

    TPortoContainerName GetChild() const {
        const size_t pos = Name_.find('/');
        Y_ENSURE(pos != TString::npos, "Can not get child of simple container '" << Name_ << "'");
        return TPortoContainerName::NoEscape(Name_.substr(pos + 1));
    }

    TPortoContainerName CutPrefix(const TPortoContainerName& prefix) const {
        Y_ENSURE(StartsWith(prefix), "Container name '" << Name_ << "' doesn't start with expected prefix '" << prefix << "'");
        return TPortoContainerName::NoEscape(Name_.substr(prefix.Name_.length() + 1));
    }

    bool StartsWith(const TString& prefix) const {
        return Name_.StartsWith(prefix);
    }

    /* use with care */
    static TPortoContainerName NoEscape(const TString& name) {
        TPortoContainerName result("");
        result.Name_ = name;
        return result;
    }

private:
    TString Escape(const TString& str) {
        TStringBuilder result;
        for (char c : str) {
            if (
                (c >= 'a' && c <= 'z')
                || (c >= 'A' && c <= 'Z')
                || (c >= '0' && c <= '9')
                || (c == '_' || c == '-' || c == '@' || c == ':' || c == '.')
            ) {
                result << c;
            } else {
                Printf(result.Out, "0x%02x", c);
            }
        }
        return result;
    }

    TString Name_;
};

struct TPortoVolumeShare {
    TString Path = "";
    TString OriginPath = "";
    bool CopyOnWrite = false;
};

} // namespace NPodAgent

namespace NPrivate {

template<>
class SuccessCastErrorMessageGetter<NInfra::NPodAgent::TPortoError> {
public:
    static TString GetMessage(const NInfra::NPodAgent::TPortoError* const error) {
        Y_ASSERT(error);
        return " Error message: " + ToString(*error);
    }
};

} // namespace NPrivate

template <>
struct THash<NInfra::NPodAgent::TPortoContainerName> {
    size_t operator()(const NInfra::NPodAgent::TPortoContainerName& portoContainerName) const {
        size_t result = ComputeHash((TString)portoContainerName);
        return result;
    }
};

inline TString ToString(NInfra::NPodAgent::EPortoError code) {
    return Porto::EError_Name(code);
}

inline TString ToString(const NInfra::NPodAgent::TPortoError& error) {
    return TStringBuilder()
        << ToString(error.Code) << "(" << i64(error.Code) << ")"
        << ':' << error.Action
        << ':' << error.Args
        << ':' << error.Message
    ;
}
