#pragma once

#include <util/generic/strbuf.h>
#include <util/system/types.h>

namespace NInfra::NPodAgent {

enum class EPortoContainerState: ui8 {
    Stopped    = 0 /* "stopped" */,    // initial state
    Starting   = 1 /* "starting" */,   // start in progress
    Running    = 2 /* "running" */,    // execution in progress
    Paused     = 3 /* "paused" */,     // frozen, consumes memory but no cpu
    Dead       = 4 /* "dead" */,       // execution complete
    Meta       = 5 /* "meta" */,       // running container without command
    Stopping   = 6 /* "stopping" */,   // stop in progress
    Respawning = 7 /* "respawning" */, // dead and will be started again
};

enum EEnablePorto {
    False       = 0 /* "false" */,          // no access
    None        = 1 /* "none" */,           // no access (same as False)
    ReadIsolate = 2 /* "read-isolate" */,   // read-only access, show only sub-containers
    ReadOnly    = 3 /* "read-only" */,      // read-only access
    Isolate     = 4 /* "isolate" */,        // read-write access, show only sub-containers
    ChildOnly   = 5 /* "child-only" */,     // write-access to sub-containers
    True        = 6 /* "true" */,           // full access (default)
    Full        = 7 /* "full" */,           // full access (default) (same as True)
};

enum class EPortoContainerProperty: ui8 {
    Command                    = 0  /* "command" */,                             // container command string
    State                      = 1  /* "state" */,                               // current state, see NPortoContainerState
    ExitCode                   = 2  /* "exit_code" */,                           // 0: success, 1..255: error code, -64..-1: termination signal, -99: OOM
    ExitStatus                 = 3  /* "exit_status" */,                         // container exit status, see wait(2)
    Cwd                        = 4  /* "cwd" */,                                 // working directory
    MemoryGuarantee            = 5  /* "memory_guarantee" */,                    // guarantee for memory_usage, default: 0
    MemoryLimit                = 6  /* "memory_limit" */,                        // limit for memory_usage, default: 0 (unlimited)
    MemoryGuaranteeTotal       = 7  /* "memory_guarantee_total" */,              // hierarchical memory guarantee
    MemoryLimitTotal           = 8  /* "memory_limit_total" */,                  // hierarchical memory limit
    AnonLimit                  = 9  /* "anon_limit" */,                          // limit for anon_usage (offstream kernel feature)
    RechPgfault                = 10 /* "recharge_on_pgfault" */,                 // recharge on pgfault
    StdErr                     = 11 /* "stderr" */,                              // stderr
    StdOut                     = 12 /* "stdout" */,                              // stdout
    CreationTimeRaw            = 13 /* "creation_time[raw]" */,                  // creation_time in seconds
    StartTimeRaw               = 14 /* "start_time[raw]" */,                     // start_time in seconds
    Time                       = 15 /* "time" */,                                // time
    Net                        = 16 /* "net" */,                                 // network namespace configuration
    Root                       = 17 /* "root" */,                                // container root path, default: /
    CpuGuarantee               = 18 /* "cpu_guarantee" */,                       // desired CPU power, <cores>c (in cores), default: 0
    CpuLimit                   = 19 /* "cpu_limit" */,                           // lCPU usage limit, <cores>c (in cores), default: 0 (unlimited)
    CpuUsage                   = 20 /* "cpu_usage" */,                           // CPU time used in nanoseconds (1 / 1000_000_000s)
    Private                    = 21 /* "private" */,                             // user-defined property
    Env                        = 22 /* "env" */,                                 // container environment variables: <name>=<value>;...
    MaxRss                     = 23 /* "max_rss" */,                             // peak anon_usage (offstream kernel feature)
    Isolate                    = 24 /* "isolate" */,                             // create new pid/ipc/utc/env namespace: true - create new namespaces (default), false - use parent namespaces
    StdErr1MB                  = 25 /* "stderr[:1048576]" */,                    // last MB of stderr
    StdOut1MB                  = 26 /* "stdout[:1048576]" */,                    // last MB of stdout
    CoreCommand                = 27 /* "core_command" */,                        // coredump command
    Ulimit                     = 28 /* "ulimit" */,                              // ulimit
    StdoutOffset               = 29 /* "stdout_offset"*/,                        // stdout_offset
    StderrOffset               = 30 /* "stderr_offset"*/,                        // stderr_offset
    Hostname                   = 31 /* "hostname"*/,                             // hostname
    User                       = 32 /* "user" */,                                // uid of container processes, default owner_user
    Group                      = 33 /* "group" */,                               // gid of container processes, default: owner_group
    Place                      = 34 /* "place" */,                               // places allowed for volumes and layers, syntax: [default][;[alias=]path;...]
    CapabilitiesAmbient        = 35 /* "capabilities_ambient" */,                // raise ambient capabilities, syntax: CAP;...
    AbsoluteName               = 36 /* "absolute_name" */,                       // full container name including porto namespaces
    StdErrPath                 = 37 /* "stderr_path" */,                         // stderr_path
    StdOutPath                 = 38 /* "stdout_path" */,                         // stdout_path
    StdOutAndStdErrLimit       = 39 /* "stdout_limit" */,                        // stdout and stderr logs file size limit (WARNING: It is not a bug, this property really puts a limit on both stdout and stderr)
    EnablePorto                = 40 /* "enable_porto" */,                        // access to porto. Containers with restricted view sees truncated container names excluding isolated part
    CachedStdoutPortoOffset    = 41 /* "PODAGENT.cached_stdout_porto_offset" */, // cached stdout porto offset for recovery after pod_agent shutdown
    CachedStderrPortoOffset    = 42 /* "PODAGENT.cached_stderr_porto_offset" */, // cached stderr porto offset for recovery after pod_agent shutdown
    ResolvConf                 = 43 /* "resolv_conf" */,                         // DNS resolver configuration, syntax: default|keep|<resolv.conf option>;...
    DeathTimeRaw               = 44 /* "death_time[raw]" */,                     // death_time in seconds
    ThreadLimit                = 45 /* "thread_limit" */,                        // limit for thread count
    AgingTime                  = 46 /* "aging_time" */,                          // time in seconds before auto-destroying dead container
    CpuPolicy                  = 47 /* "cpu_policy" */,                          // CPU scheduler policy, see sched(7)
    CpuWeight                  = 48 /* "cpu_weight" */,                          // CPU weight, multiplies cpu.shares and +10% cpu_weight is -1 nice
    IoLimit                    = 49 /* "io_limit" */,                            // IO bandwidth limit, syntax: fs|<path>|<disk> [r|w]: <bytes/s>;...
    IoOpsLimit                 = 50 /* "io_ops_limit" */,                        // IOPS limit: fs|<path>|<disk> [r|w]: <iops>;...
    IoPolicy                   = 51 /* "io_policy" */,                           // IO scheduler policy, see ioprio_set(2)
    IoWeight                   = 52 /* "io_weight" */,                           // IO weight, additional multiplier for blkio.weight
    EnvSecret                  = 53 /* "env_secret" */,                          // container secret environment variables: <name>=<value>;...
    Cgroupfs                   = 54 /* "cgroupfs" */,                            // mode for mounting cgroup fs
};

enum class EPortoVolumeProperty: ui8 {
    SpaceLimit      = 0  /* "space_limit" */,      // disk space limit, default: 0 - unlimited
    INodeLimit      = 1  /* "inode_limit" */,      // disk inode limit, default: 0 - unlimited
    SpaceGuarantee  = 2  /* "space_guarantee" */,  // disk space guarantee, default: 0
    INodeGuarantee  = 3  /* "inode_guarantee" */,  // disk inode guarantee, default: 0
    SpaceUsed       = 4  /* "space_used" */,       // current disk space usage
    INodeUsed       = 5  /* "inode_used" */,       // current disk inode used
    SpaceAvailable  = 6  /* "space_available" */,  // available disk space
    INodeAvailable  = 7  /* "inode_available" */,  // available disk inodes
    Containers      = 8  /* "containers" */,       // initial links, syntax: container [target] [ro] [!];... default: "self". Target defines path inside container, flag "ro" makes link read-only, "!" - required.
    TargetContainer = 9  /* "target_container" */, // define root path, default: creator. Volume will be created inside root path of this container.
    Layers          = 10 /* "layers" */,           // layers, syntax: top-layer;...;bottom-layer
    Private         = 11 /* "private" */,          // user-defined property
    Storage         = 12 /* "storage" */,          // persistent volume storage path
    Backend         = 13 /* "backend" */,          // backend volume, default: auto
    Place           = 14 /* "place" */,            // place for layers and default storage
    State           = 15 /* "state" */,            // state (unready/building/ready/unlinked/tuning/deleting/deleted)
    ReadOnly        = 16 /* "read_only" */,        // is volume read_only (true or false)
};

enum class EPortoVolumeBackend: ui8 {
    Auto     = 0 /* "" */,
    Bind     = 1 /* "bind" */,
    RBind    = 2 /* "rbind" */,
};

} // namespace NInfra::NPodAgent
