#include "simple_client.h"
#include "porto_request_response_converter.h"

#include <infra/pod_agent/libs/util/string_utils.h>
#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>

#include <util/string/cast.h>
#include <util/string/vector.h>
#include <util/string/split.h>

namespace NInfra::NPodAgent {

bool TSimplePortoClient::IsInitialized_ = false;
TRWMutex TSimplePortoClient::SignalsMutex_;

TSimplePortoClient::TSimplePortoClient(TLogFramePtr logFrame, TPortoConnectionPoolPtr pool, int timeoutSeconds)
    : LogFrame_(logFrame)
    , Pool_(pool)
    , TimeoutSeconds_(timeoutSeconds)
{
    InitSignals();
}

TPortoClientPtr TSimplePortoClient::SwitchLogFrame(TLogFramePtr logFrame) {
    return new TSimplePortoClient(logFrame, Pool_, TimeoutSeconds_);
}

TExpected<void, TPortoError> TSimplePortoClient::Create(const TPortoContainerName& name) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Create(name);
        }, "Create", name
    );
}

TExpected<void, TPortoError> TSimplePortoClient::CreateRecursive(const TPortoContainerName& name) {
    TVector<TString> components;
    StringSplitter(TString(name)).Split('/').SkipEmpty().Collect(&components);

    TPortoContainerName currentName("");
    for (const auto& item : components) {
        if (TString(currentName)) {
            currentName = TPortoContainerName(currentName, item);
        } else {
            currentName = TPortoContainerName(item);
        }

        int existResult = OUTCOME_TRYX(IsContainerExists(currentName));
        if (!existResult) {
            auto result = Create(currentName);
            if (!result && result.Error().Code != EPortoError::ContainerAlreadyExists) {
                return result.Error();
            }
        }
    }

    return TExpected<void, TPortoError>::DefaultSuccess();
}

TExpected<void, TPortoError> TSimplePortoClient::Destroy(const TPortoContainerName& name) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Destroy(name);
        }, "Destroy", name
    );
}

TExpected<int, TPortoError> TSimplePortoClient::IsContainerExists(const TPortoContainerName& name) {
    auto action = [=](Porto::TPortoApi& c, Porto::TPortoResponse& result) {
        Porto::TPortoRequest req;
        req.mutable_list()->set_mask(name);
        return c.Call(req, result);
    };
    auto rsp = OUTCOME_TRYX(RunWithResult<Porto::TPortoResponse>(action, "IsContainerExists"));
    return rsp.list().name_size() > 0;
}

TExpected<void, TPortoError> TSimplePortoClient::Start(const TPortoContainerName& name) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Start(name);
        }, "Start", name
    );
}

TExpected<void, TPortoError> TSimplePortoClient::Stop(const TPortoContainerName& name, TDuration timeout) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Stop(name, static_cast<int>(timeout.MilliSeconds()));
        }, "Stop", name
    );
}

TExpected<void, TPortoError> TSimplePortoClient::Kill(const TPortoContainerName& name, int sig) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Kill(name, sig);
        }, "Kill", name
    );
}

TExpected<void, TPortoError> TSimplePortoClient::Pause(const TPortoContainerName& name) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Pause(name);
        }, "Pause", name
    );
}

TExpected<void, TPortoError> TSimplePortoClient::Resume(const TPortoContainerName& name) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.Resume(name);
        }, "Resume", name
    );
}

TExpected<TString, TPortoError>
TSimplePortoClient::WaitContainers(const TVector<TPortoContainerName>& containers, TDuration timeout) {
    auto action = [=](Porto::TPortoApi& c, TString& result) {
        TVector<TString> names;
        for (const auto& container : containers) {
            names.push_back(ToString(container));
        }
        TString result_state;
        return c.WaitContainers(names, result, result_state, timeout.MilliSeconds());
    };
    return RunWithResult<TString>(action, "WaitContainers");
}

TExpected<TVector<TPortoContainerName>, TPortoError> TSimplePortoClient::List(const TString& mask) {
    auto action = [=](Porto::TPortoApi& c, TVector<TPortoContainerName>& result) {
        auto ret = c.List(mask);
        if (ret) {
            result.reserve(ret->name_size());
            for (const auto& it : ret->name()) {
                result.push_back(TPortoContainerName::NoEscape(it));
            }
        }
        return c.Error();
    };
    return RunWithResult<TVector<TPortoContainerName>>(action, "List");
}

TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError>
TSimplePortoClient::Get(const TVector<TPortoContainerName>& name, const TVector<EPortoContainerProperty>& variable) {
    auto action = [=](Porto::TPortoApi& c, TPortoGet& result) {
        TVector<TString> new_name,
                         new_variable;

        new_name.reserve(name.size());
        new_variable.reserve(variable.size());
        for (const auto& it : name) {
            new_name.push_back(ToString(it));
        }

        for (const auto& it : variable) {
            new_variable.push_back(ToString(it));
        }

        auto ret = c.Get(new_name, new_variable);
        if (ret)
            result = *ret;

        return c.Error();
    };

    auto ret = OUTCOME_TRYX(RunWithResult<TPortoGet>(action, "Get"));

    TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>> result;
    for (const auto& entry : ret.list()) {
        for (const auto& keyval : entry.keyval()) {
            TPortoGetResponse resp;
            resp.set_error(EPortoError(0));
            if (keyval.has_error())
                resp.set_error(keyval.error());
            if (keyval.has_errormsg())
                resp.set_errormsg(keyval.errormsg());
            if (keyval.has_value())
                resp.set_value(keyval.value());

            result[TPortoContainerName::NoEscape(entry.name())][FromString(keyval.variable())] = resp;
        }
    }
    return result;
}

TExpected<TString, TPortoError>
TSimplePortoClient::GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int flags) {
    auto action = [=](Porto::TPortoApi& c, TString& result) {
        return c.GetProperty(name, ToString(property), result, flags);
    };
    return RunWithResult<TString>(
        action, "GetProperty", TStringBuilder() << TString(name) << "'.'" << ToString(property) << "'"
    );
}

TExpected<void, TPortoError> TSimplePortoClient::SetProperty(
    const TPortoContainerName& name
    , EPortoContainerProperty property
    , const TString& value
) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.SetProperty(name, ToString(property), value);
        }
        , "SetProperty"
        , TStringBuilder() << TString(name) << "'.'" << ToString(property) << "' -> '" << value << "'"
    );
}

TExpected<void, TPortoError> TSimplePortoClient::SetProperties(
    const TPortoContainerName& name,
    const TMap<EPortoContainerProperty, TString>& properties
) {
    if (properties.empty()) {
        // TODO(DEPLOY-3231): Remove when empty map will be impossible in this call
        return TExpected<void, TPortoError>::DefaultSuccess();
    }

    using namespace NInfra::NPodAgent;
    TVector<TString> propertyArgs;
    for (const auto& [property, value] : properties) {
        if (property == EPortoContainerProperty::EnvSecret) {
            propertyArgs.push_back(Quote(ToString(property)) + " -> <hidden>");
        } else {
            propertyArgs.push_back(Quote(ToString(property)) + " -> " + Quote(value));
        }
    }
    TString args = TStringBuilder() << Quote(TString(name)) << ": " << JoinStrings(propertyArgs, ", ");

    auto convertResult = NPortoRequestResponseConverter::ConvertPropertiesToProtoContainerSpec(name, properties);
    if (!convertResult) {
        auto error = convertResult.Error();
        error.Action = "SetProperties";
        error.Args = args;
        return error;
    }
    Porto::TContainerSpec containerSpec = convertResult.Success();
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.UpdateFromSpec(containerSpec);
        }
        , "SetProperties"
        , args
    );
}

TExpected<TString, TPortoError>
TSimplePortoClient::GetStdout(const TPortoContainerName& name, int offset, int length, int flags) {
    return GetStream(name, "stdout", offset, length, flags);
}

TExpected<TString, TPortoError>
TSimplePortoClient::GetStderr(const TPortoContainerName& name, int offset, int length, int flags) {
    return GetStream(name, "stderr", offset, length, flags);
}

TExpected<TString, TPortoError> TSimplePortoClient::GetStream(
    const TPortoContainerName& name
    , const TString& stream
    , int offset
    , int length
    , int flags

) {
    auto action = [=](Porto::TPortoApi& c, TString& result) {
        TStringBuilder property;
        property << stream;
        if (offset != -1 || length != -1) {
            property << '[';
            if (offset != -1) {
                property << offset;
            }
            property << ':';
            if (length != -1) {
                property << length;
            }
            property << ']';
        }

        Porto::TPortoRequest request;
        Porto::TPortoResponse resp;

        auto* get_property = request.mutable_getproperty();
        get_property->set_name(name);
        get_property->set_property(property);
        if (flags & Porto::GET_SYNC)
            get_property->set_sync(true);
        if (flags & Porto::GET_REAL)
            get_property->set_real(true);

        int ret = c.Call(request, resp);
        if (!ret)
            result.assign(resp.getproperty().value());
        return ret;
    };
    return RunWithResult<TString>(action, "GetStream" + stream);
}

TExpected<TString, TPortoError> TSimplePortoClient::CreateVolume(
    const TString& path
    , const TString& storage
    , const TString& place
    , const TVector<TString>& layers
    , unsigned long long quotaBytes
    , const TString& privateValue
    , const EPortoVolumeBackend backend
    , const TPortoContainerName& containerName
    , const TVector<TPortoVolumeShare>& staticResources
    , bool readOnly
) {
    Porto::TVolumeSpec request = NPortoRequestResponseConverter::ConvertVolumeCreationRequest(
        path
        , storage
        , place
        , layers
        , quotaBytes
        , privateValue
        , backend
        , containerName
        , staticResources
        , readOnly
    );

    auto action = [=](Porto::TPortoApi& c, TString& volumePath) {
        Porto::TVolumeSpec response;
        int ret = c.CreateVolumeFromSpec(request, response);

        if (!ret)
            volumePath = NPortoRequestResponseConverter::ConvertVolumeCreationResponse(response);

        return ret;
    };
    return RunWithResult<TString>(action, "CreateVolume", path);
}

TExpected<void, TPortoError> TSimplePortoClient::LinkVolume(
    const TString& path
    , const TPortoContainerName& container
    , const TString& target
    , bool readOnly
    , bool required
) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.LinkVolume(path, container, target, readOnly, required);
        }
        , "LinkVolume"
        , TStringBuilder() << path << '(' << TString(container) << '.' << target << '.' << (readOnly ? "RO" : "RW") << ')'
    );
}

TExpected<void, TPortoError> TSimplePortoClient::UnlinkVolume(
    const TString& path
    , const TPortoContainerName& container
    , const TString& target
    , bool strict
) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.UnlinkVolume(path, container, target, strict);
        }
        , "UnlinkVolume"
        , TStringBuilder() << path << '(' << TString(container) << '.' << target << ')'
    );
}

TExpected<TVector<TPortoVolume>, TPortoError>
TSimplePortoClient::ListVolumes(const TString& path, const TPortoContainerName& container) {
    auto action = [=](Porto::TPortoApi& c, TVector<TPortoVolume>& result) {
        Porto::TGetVolumeRequest request = NPortoRequestResponseConverter::ConvertGetVolumeRequest({path}, container);

        return c.ListVolumesBy(request, result);
    };
    return RunWithResult<TVector<TPortoVolume>>(action, "ListVolumes");
}

TExpected<TVector<TString>, TPortoError>
TSimplePortoClient::ListVolumesPaths(const TString& path, const TPortoContainerName& container) {
    auto action = [=](Porto::TPortoApi& c, TVector<TString>& result) {
        Porto::TGetVolumeRequest request = NPortoRequestResponseConverter::ConvertGetVolumeRequest({path}, container);

        TVector<TPortoVolume> volumes;
        auto ret = c.ListVolumesBy(request, volumes);
        result.reserve(volumes.size());
        for (const auto& volume : volumes) {
            result.emplace_back(volume.path());
        }

        return ret;
    };
    return RunWithResult<TVector<TString>>(action, "ListVolumesPaths");
}

TExpected<int, TPortoError> TSimplePortoClient::IsVolumeExists(const TString& path) {
    auto action = [=](Porto::TPortoApi& c, int& result) {
        Porto::TGetVolumeRequest request = NPortoRequestResponseConverter::ConvertGetVolumeRequest({path});

        TVector<TPortoVolume> volumes;
        auto ret = c.ListVolumesBy(request, volumes);
        result = volumes.size() > 0;

        return ret;
    };
    auto result = RunWithResult<int>(action, "IsVolumeExists");
    if (result) {
        return 1;
    } else {
        if (result.Error().Code == EPortoError::VolumeNotFound) {
            return 0;
        }
        return result.Error();
    }
    return result;
}

TExpected<void, TPortoError> TSimplePortoClient::ImportLayer(
    const TString& layer
    , const TString& tarball
    , bool merge
    , const TString& place
    , const TString& privateValue
) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.ImportLayer(layer, tarball, merge, place, privateValue);
        }, "ImportLayer", tarball
    );
}

TExpected<void, TPortoError> TSimplePortoClient::RemoveLayer(const TString& layer, const TString& place) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.RemoveLayer(layer, place);
        }, "RemoveLayer"
    );
}

TExpected<TVector<TPortoLayer>, TPortoError> TSimplePortoClient::ListLayers(const TString& place, const TString& mask) {
    auto action = [=](Porto::TPortoApi& c, TVector<TPortoLayer>& result) {
        auto ret = c.ListLayers(place, mask);

        int i = 0;
        if (ret) {
            if (ret->layers().size()) {
                result.resize(ret->layers_size());
                for (const auto& layer: ret->layers()) {
                    result[i++] = layer;
                }
            } else {
                result.resize(ret->layer_size());
                for (const auto& layer: ret->layer()) {
                    TPortoLayer l;
                    l.set_name(layer);
                    result[i++] = l;
                }
            }
        }

        return c.Error();
    };
    return RunWithResult<TVector<TPortoLayer>>(action, "ListLayers");
}

TExpected<TString, TPortoError> TSimplePortoClient::GetLayerPrivate(const TString& layer, const TString& place) {
    auto action = [=](Porto::TPortoApi& c, TString& result) {
        return c.GetLayerPrivate(result, layer, place);
    };
    return RunWithResult<TString>(action, "GetLayerPrivate");
}

TExpected<void, TPortoError>
TSimplePortoClient::SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.SetLayerPrivate(privateValue, layer, place);
        }, "SetLayerPrivate"
    );
}

TExpected<TVector<TPortoStorage>, TPortoError>
TSimplePortoClient::ListStorages(const TString& place, const TString& mask) {
    return RunWithResult<TVector<TPortoStorage>>(
        [=](Porto::TPortoApi& c, TVector<TPortoStorage>& result) {
            auto storagelist = c.ListStorages(place, mask);

            if (storagelist) {
                result.resize(storagelist->storages_size());
                int i = 0;
                for (const auto& storage: storagelist->storages()) {
                    result[i++] = storage;
                }
            }

            return c.Error();
        }
        , "ListStorages"
    );
}

TExpected<TString, TPortoError> TSimplePortoClient::GetStoragePrivate(const TString& place, const TString& name) {
    auto action = [=](Porto::TPortoApi& c, TString& result) {
        Porto::TPortoRequest req;
        auto listReq = req.mutable_liststorages();
        if (place.size())
            listReq->set_place(place);
        if (name.size())
            listReq->set_mask(name);
        Porto::TPortoResponse resp;
        auto code = c.Call(req, resp);
        if (code)
            return static_cast<int>(code);
        const Porto::TListStoragesResponse& storageList = resp.liststorages();
        for (const auto& storage: storageList.storages()) {
            if (storage.name() == name) {
                result = storage.private_value();
                return 0;
            }
        }
        result = "";
        return static_cast<int>(EPortoError::VolumeNotFound);
    };
    return RunWithResult<TString>(
        action
        , "GetStoragePrivate"
    );
}

TExpected<void, TPortoError> TSimplePortoClient::RemoveStorage(const TString& name, const TString& place) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.RemoveStorage(name, place);
        }, "RemoveStorage"
    );
}

TExpected<void, TPortoError> TSimplePortoClient::ImportStorage(
    const TString& name
    , const TString& archive
    , const TString& place
    , const TString& compression
    , const TString& privateValue
) {
    return Run(
        [=](Porto::TPortoApi& c) {
            return c.ImportStorage(name, archive, place, compression, privateValue);
        }, "ImportStorage"
    );
}

TExpected<bool, TPortoError> TSimplePortoClient::IsStorageExists(const TString& place, const TString& name) {
    return RunWithResult<bool>(
        [=](Porto::TPortoApi& c, bool& result) {
            auto storagelist = c.ListStorages(place, name);

            if (storagelist) {
                if (storagelist->storages_size()) {
                    result = true;
                    return c.Error();
                }
            }
            result = false;
            return c.Error();
        }
        , "IsStorageExists"
    );
}

TExpected<void, TPortoError> TSimplePortoClient::Run(
    std::function<int(Porto::TPortoApi& connection)> action
    , const TString& name
    , const TString& args
) {
    Porto::TPortoApi* connection = OUTCOME_TRYX(Pool_->Dequeue());
    connection->SetTimeout(TimeoutSeconds_);
    int err = action(*connection);
    TString textError = connection->GetLastError();
    Pool_->Enqueue(connection);

    TAtomic id = SpawnCallId(name);
    return HandleResponseCode(textError, err, id, name, args);
}

template<class T>
TExpected<T, TPortoError> TSimplePortoClient::RunWithResult(
    std::function<int(Porto::TPortoApi& connection, T& result)> action
    , const TString& name
    , const TString& args
) {
    Porto::TPortoApi* connection = OUTCOME_TRYX(Pool_->Dequeue());
    connection->SetTimeout(TimeoutSeconds_);
    T result;
    int err = action(*connection, result);
    TString textError = connection->GetLastError();
    Pool_->Enqueue(connection);

    TAtomic id = SpawnCallId(name);
    OUTCOME_TRYV(HandleResponseCode(textError, err, id, name, args));
    return result;
}

TAtomic TSimplePortoClient::SpawnCallId(const TString& name) {
    TAtomic id = AtomicIncrement(Counter_);
    IncCall(name, id);
    return id;
}

TExpected<void, TPortoError> TSimplePortoClient::HandleResponseCode(
    const TString& textError
    , int code
    , TAtomic id
    , const TString& action
    , const TString& args
) {
    IncResponse(action, code, id);
    if (!code) {
        return TExpected<void, TPortoError>::DefaultSuccess();
    } else {
        return TPortoError({EPortoError(code), action, args, textError});
    }
}

void TSimplePortoClient::InitSignals() {
    {
        TReadGuard guard(SignalsMutex_);
        if (IsInitialized_) {
            return;
        }
    }

    TWriteGuard guard(SignalsMutex_);
    if (IsInitialized_) {
        return;
    }
    IsInitialized_ = true;

    const TVector<TString> signalNames = {
        "Create"
        , "Destroy"
        , "Start"
        , "Stop"
        , "Kill"
        , "Pause"
        , "Resume"
        , "ImportLayer"
        , "RemoveLayer"
        , "SetLayerPrivate"
        , "RemoveStorage"
        , "ImportStorage"
        , "AttachProcess"
        , "IsContainerExists"
        , "WaitContainers"
        , "List"
        , "Get"
        , "GetProperty"
        , "CreateVolume"
        , "ListVolumes"
        , "ListVolumesPaths"
        , "IsVolumeExists"
        , "ListLayers"
        , "GetLayerPrivate"
        , "ConvertPath"
        , "LocateProcess"
        , "SetProperty"
        , "SetProperties"
        , "GetStreamstdout"
        , "GetStreamstderr"
        , "LinkVolume"
        , "UnlinkVolume"
        , "GetStoragePrivate"
        , "ListStorages"
        , "IsStorageExists"
    };

    for (const auto& name : signalNames) {
        for (const auto& suffix : TVector<TString>{"", COUNTER_ZERO_CODE_SUFFIX, COUNTER_NON_ZERO_CODE_SUFFIX}) {
            TString curName = COUNTER_PREFIX + name + suffix;

            TMultiUnistat::Instance().DrillFloatHole(
                TMultiUnistat::ESignalNamespace::INFRA
                , curName
                , "deee"
                , NUnistat::TPriority(TMultiUnistat::ESignalPriority::DEBUG)
                , NUnistat::TStartValue(0)
                , EAggregationType::Sum
            );
        }
    }

    for (const auto& suffix : TVector<TString>{"", COUNTER_ZERO_CODE_SUFFIX, COUNTER_NON_ZERO_CODE_SUFFIX}) {
        TString curName = COUNTER_PREFIX + COUNTER_REQUESTS_TOTAL + suffix;

        TMultiUnistat::Instance().DrillFloatHole(
            TMultiUnistat::ESignalNamespace::INFRA
            , curName
            , "deee"
            // Infra info priority for requests_total
            , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
            , NUnistat::TStartValue(0)
            , EAggregationType::Sum
        );
    }
}

void TSimplePortoClient::IncCounter(const TString& name, const TString& suffix, TAtomic id) {
    // Update debug name sensor and info requests_total sensor
    for (const TString& currentName : {COUNTER_PREFIX + name + suffix, COUNTER_PREFIX + COUNTER_REQUESTS_TOTAL + suffix}) {
        if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, currentName, 1)) {
            LogFrame_->LogEvent(
                ELogPriority::TLOG_ERR
                , NLogEvent::TPortoSignalError(
                    id
                    , name
                    , "PushSingalUnsafe(" + currentName + ", 1) returned false."
                )
            );
        }
    }
}

void TSimplePortoClient::IncCall(const TString& name, TAtomic id) {
    IncCounter(name, "" /* suffix */, id);
}

void TSimplePortoClient::IncResponse(const TString& name, int code, TAtomic id) {
    if (code) {
        IncCounter(name, COUNTER_NON_ZERO_CODE_SUFFIX, id);
    } else {
        IncCounter(name, COUNTER_ZERO_CODE_SUFFIX, id);
    }
}

} // namespace NInfra::NPodAgent

