#pragma once

#include "client.h"

#include <util/system/rwlock.h>
#include <util/string/cast.h>

namespace NInfra::NPodAgent {

/**
    Thread pool + Porto::TPortoApi pool
    Each Porto::TPortoApi itself is sync (and holds opened unix socket to porto-daemon)
*/
class TPortoConnectionPool: public TAtomicRefCount<TPortoConnectionPool> {
public:
    TPortoConnectionPool(size_t poolSize) {
        for (size_t i = 0; i < poolSize; ++i) {
            ConnectionsHolders_.push_back(MakeHolder<Porto::TPortoApi>());
            Connections_.Enqueue(ConnectionsHolders_.back().Get());
        }
    }

    TExpected<Porto::TPortoApi*, TPortoError> Dequeue() {
        Porto::TPortoApi* connection;
        bool res = Connections_.Dequeue(&connection);
        if (res) {
            return connection;
        } else {
            return TPortoError{EPortoError::Unknown, "GetConnection", "No porto connections available at pool"};
        }
    }

    void Enqueue(Porto::TPortoApi* connection) {
        Connections_.Enqueue(connection);
    }

private:
    TLockFreeStack<Porto::TPortoApi*> Connections_;
    TVector<THolder<Porto::TPortoApi>> ConnectionsHolders_;
};

using TPortoConnectionPoolPtr = TIntrusivePtr<TPortoConnectionPool>;

/**
    Lightweight porto client

    Calls to Porto daemon
    Takes connections from connection pool (@see TPortoConnectionPool)

    Pushes counters to sensors
    Logs events to given logframe
*/
class TSimplePortoClient: public IPortoClient {
public:
    TSimplePortoClient(TLogFramePtr logFrame, TPortoConnectionPoolPtr pool, int timeoutSeconds);

    TPortoClientPtr SwitchLogFrame(TLogFramePtr logFrame) override;

    TExpected<void, TPortoError> Create(const TPortoContainerName& name) override;

    TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override;

    TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override;

    TExpected<int, TPortoError> IsContainerExists(const TPortoContainerName& name) override;

    TExpected<void, TPortoError> Start(const TPortoContainerName& name) override;

    TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration timeout) override;

    TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int sig) override;

    TExpected<void, TPortoError> Pause(const TPortoContainerName& name) override;

    TExpected<void, TPortoError> Resume(const TPortoContainerName& name) override;

    TExpected<TString, TPortoError> WaitContainers(const TVector<TPortoContainerName>& containers, TDuration timeout) override;

    TExpected<TVector<TPortoContainerName>, TPortoError> List(const TString& mask) override;

    TExpected<TMap<TPortoContainerName, TMap<EPortoContainerProperty, TPortoGetResponse>>, TPortoError> Get(const TVector<TPortoContainerName>& name, const TVector<EPortoContainerProperty>& variable) override;

    TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int flags) override;

    TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& value) override;

    TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override;

    TExpected<TString, TPortoError> GetStdout(const TPortoContainerName& name, int offset, int length, int flags) override;

    TExpected<TString, TPortoError> GetStderr(const TPortoContainerName& name, int offset, int length, int flags) override;

    TExpected<TString, TPortoError> GetStream(const TPortoContainerName& name, const TString& stream, int offset, int length, int flags) override;

    TExpected <TString, TPortoError> CreateVolume(
        const TString& path
        , const TString& storage
        , const TString& place
        , const TVector<TString>& layers
        , unsigned long long quotaBytes
        , const TString& privateValue
        , const EPortoVolumeBackend backend
        , const TPortoContainerName& containerName
        , const TVector<TPortoVolumeShare>& staticResources
        , bool readOnly
    ) override;

    TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) override;

    TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict) override;

    TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override;

    TExpected<TVector<TString>, TPortoError> ListVolumesPaths(const TString& path, const TPortoContainerName& container) override;

    TExpected<int, TPortoError> IsVolumeExists(const TString& path) override;

    TExpected<void, TPortoError> ImportLayer(const TString& layer, const TString& tarball, bool merge, const TString& place, const TString& privateValue) override;

    TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override;

    TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override;

    TExpected<TString, TPortoError> GetLayerPrivate(const TString& layer, const TString& place) override;

    TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) override;

    TExpected<TVector<TPortoStorage>, TPortoError> ListStorages(const TString& place, const TString& mask) override;

    TExpected<TString, TPortoError> GetStoragePrivate(const TString& place, const TString& name) override;

    TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place) override;

    TExpected<void, TPortoError> ImportStorage(const TString& name, const TString& archive, const TString& place, const TString& compression, const TString& privateValue) override;

    TExpected<bool, TPortoError> IsStorageExists(const TString& place, const TString& name) override;

private:
    TExpected<void, TPortoError> Run(std::function<int(Porto::TPortoApi& connection)> action, const TString& name, const TString& args = "");

    template <class T>
    TExpected<T, TPortoError> RunWithResult(std::function<int(Porto::TPortoApi& connection, T& result)> action, const TString& name, const TString& args = "");

    TAtomic SpawnCallId(const TString& name);

    TExpected<void, TPortoError> HandleResponseCode(const TString& textError, int code, TAtomic id, const TString& action, const TString& args);

    void InitSignals();
    void IncCounter(const TString& name, const TString& suffix, TAtomic id);

    void IncCall(const TString& name, TAtomic id);
    void IncResponse(const TString& name, int code, TAtomic id);

private:
    TAtomic Counter_ = 0;
    mutable TLogFramePtr LogFrame_;
    TPortoConnectionPoolPtr Pool_;
    const int TimeoutSeconds_;

    static TRWMutex SignalsMutex_;
    static bool IsInitialized_;

    static inline const TString COUNTER_PREFIX = "pod_agent_porto_call_";
    static inline const TString COUNTER_ZERO_CODE_SUFFIX = "_zero_exit_code";
    static inline const TString COUNTER_NON_ZERO_CODE_SUFFIX = "_non_zero_exit_code";
    static inline const TString COUNTER_REQUESTS_TOTAL = "requests_total";
};

} // namespace NInfra::NPodAgent
