#include "simple_client.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/util/string_utils.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/system/thread.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NPortoTest {

Y_UNIT_TEST_SUITE(PortoClient) {

Y_UNIT_TEST(CreateClient) {
    UNIT_ASSERT_NO_EXCEPTION(NPortoTestLib::GetSimplePortoClient());
}

Y_UNIT_TEST(ContainerCreateListDestroy) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerCreateListDestroy");

    TVector<TPortoContainerName> list = client->List().Success();
    bool found = false;
    for (auto& it : list) {
        found |= (it == TPortoContainerName(container));
    }

    UNIT_ASSERT_C(found, "container not found");
}

Y_UNIT_TEST(ContainerRunHelloWorld) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerRunHelloWorld");

    TString hello = "Hello World";
    client->SetProperty(container, EPortoContainerProperty::Command, "echo -n " + hello).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString stdoutResult = client->GetStdout(container).Success();
    ui64 stdoutOffset = FromString(client->GetProperty(container, EPortoContainerProperty::StdoutOffset).Success());

    UNIT_ASSERT_EQUAL_C(stdoutOffset, 0, "unexpected stdout offset");
    UNIT_ASSERT_EQUAL_C(stdoutResult, hello, "unexpected stdout");
}

Y_UNIT_TEST(ContainerRunHelloWorldPrintingToStderr) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerRunHelloWorldPrintingToStderr");

    TString hello = "Hello World";
    client->SetProperty(container, EPortoContainerProperty::Command, "bash -c \"echo -n " + hello + " >&2\"").Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString stderrResult = client->GetStderr(container).Success();
    ui64 stderrOffset = FromString(client->GetProperty(container, EPortoContainerProperty::StderrOffset).Success());

    UNIT_ASSERT_EQUAL_C(stderrOffset, 0, "unexpected stderr offset");
    UNIT_ASSERT_EQUAL_C(stderrResult, hello, "unexpected stderr");
}

Y_UNIT_TEST(ContainerRunHelloWorldPrintingWithGetStream) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerRunHelloWorld");

    TString hello = "Hello World";
    client->SetProperty(container, EPortoContainerProperty::Command, "echo -n " + hello).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString stdoutResult = client->GetStream(container, "stdout").Success();
    ui64 stdoutOffset = FromString(client->GetProperty(container, EPortoContainerProperty::StdoutOffset).Success());

    UNIT_ASSERT_EQUAL_C(stdoutOffset, 0, "unexpected stdout offset");
    UNIT_ASSERT_EQUAL_C(stdoutResult, hello, "unexpected stdout");
}

Y_UNIT_TEST(LayerLastUsage) {
    TStringBuilder commandBuilder;
    commandBuilder << "bash -c \"echo \\\" data \\\" > tmp.txt; "
                   << "tar -czvf " << GetWorkPath() << "/layer.tar.xz tmp.txt\"";
    TShellCommand shellCommand(commandBuilder);
    shellCommand.Run();
    shellCommand.Wait();
    UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());
    TString layerName = GetTestPrefix() + "usage_layer";
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    client->RemoveLayer(layerName);
    client->ImportLayer(layerName, GetWorkPath() + "/layer.tar.xz").Success();
    auto layers = client->ListLayers().Success();
    bool found = false;
    for (auto& layer: layers) {
        if (layer.name() == layerName) {
            UNIT_ASSERT(layer.last_usage() < 3);
            found = true;
        }
    }
    UNIT_ASSERT_C(found, "layer " << layerName << " not found");
    client->RemoveLayer(layerName).Success();
}

Y_UNIT_TEST(StorageLastUsage) {
    TStringBuilder commandBuilder;
    commandBuilder << "bash -c \"echo \\\" data \\\" > tmp.txt; "
                   << "tar -czvf " << GetWorkPath() << "/storage.tar.xz tmp.txt\"";
    TShellCommand shellCommand(commandBuilder);
    shellCommand.Run();
    shellCommand.Wait();
    UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());
    TString storageName = GetTestPrefix() + "usage_storage";
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    client->RemoveStorage(storageName);
    client->ImportStorage(storageName, GetWorkPath() + "/storage.tar.xz").Success();
    auto storages = client->ListStorages("", "").Success();
    bool found = false;
    for (auto& storage: storages) {
        if (storage.name() == storageName) {
            UNIT_ASSERT(storage.last_usage() < 3);
            found = true;
        }
    }
    UNIT_ASSERT_C(found, "storage " << storageName << " not found");
    client->RemoveStorage(storageName).Success();
}

Y_UNIT_TEST(VolumeCreate) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalVolume volume(client);
}

Y_UNIT_TEST(CreateVolumeWithStorage) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TString storage = GetTestPrefix() + "my_storage";
    TString path;
    client->RemoveStorage(storage);
    UNIT_ASSERT_NO_EXCEPTION(path = client->CreateVolume("", storage, "", {}, 0, "my_value", EPortoVolumeBackend::Auto, {""}, {}, false).Success());
    UNIT_ASSERT(client->GetStoragePrivate("", storage).Success() == "my_value");
    UNIT_ASSERT(client->IsStorageExists("", storage).Success());
    UNIT_ASSERT_NO_EXCEPTION(client->UnlinkVolume(path).Success());
    UNIT_ASSERT_NO_EXCEPTION(client->RemoveStorage(storage).Success());
}

Y_UNIT_TEST(VolumeSpaced) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    size_t quotaBytes = 1 << 20;
    TLocalVolume volume(client, quotaBytes);

    TLocalContainer container(client, GetTestPrefix() + "VolumeSpaced");
    TString command = "dd if=/dev/zero bs=2M count=1 of=" + volume.GetPath() + "/sample.txt";

    client->SetProperty(container, EPortoContainerProperty::Command, command).Success();
    client->LinkVolume(volume.GetPath(), container);

    auto portoVolume = client->ListVolumes(volume.GetPath()).Success()[0];
    UNIT_ASSERT_EQUAL_C(portoVolume.space().limit(), quotaBytes, "unexpected space limit: " << portoVolume.space().limit());
    UNIT_ASSERT_EQUAL_C(portoVolume.space().guarantee(), quotaBytes, "unexpected space guarantee: " << portoVolume.space().limit());
    if (NPortoTestLib::IsInsideSandboxPortoIsolation()) {
        // space limit will not work correctly into isolation because project quota over project quota doesn't work
        return;
    }
    client->Start(container).Success();
    client->WaitContainers({container}).Success();

    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    TString status = client->GetProperty(container, EPortoContainerProperty::ExitStatus).Success();

    UNIT_ASSERT_C(FromString<i32>(code), "error code expected, got " << code);
    UNIT_ASSERT_EQUAL_C(status, "256", "error status expected, got " << status);
}

Y_UNIT_TEST(LinkVolumeToVolume) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    client->LinkVolume(volumeSmall.GetPath(), {""}, target);
    auto list = client->ListVolumes(target).Success();
    UNIT_ASSERT_C(list.size(), "link volume not found");
    UNIT_ASSERT_EQUAL_C(list[0].links_size(), 2, "there must be exactly two links");
    UNIT_ASSERT(target == list[0].links(0).target() || target == list[0].links(1).target());

    TLocalContainer container(client, GetTestPrefix() + "LinkVolumeToVolume");
    TString command = TStringBuilder() << "bash -c \""
        << "echo -n some_data > " << volumeSmall.GetPath() << "/some_file"
        << ";cat " << target << "/some_file" << "\"";
    client->SetProperty(container, EPortoContainerProperty::Command, command).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString stdout = client->GetProperty(container, EPortoContainerProperty::StdOut).Success();
    UNIT_ASSERT_EQUAL_C("some_data", stdout, "unexpected stdout: '" << stdout << "'");
}

Y_UNIT_TEST(LinkVolumeToVolumeDefaultReadAndWrite) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    client->LinkVolume(volumeSmall.GetPath(), {""}, target);
    auto list = client->ListVolumes(target).Success();
    UNIT_ASSERT_C(list.size(), "link volume not found");
    UNIT_ASSERT_EQUAL_C(list[0].links_size(), 2, "there must be exactly two links");
    UNIT_ASSERT(target == list[0].links(0).target() || target == list[0].links(1).target());

    TLocalContainer container(client, GetTestPrefix() + "LinkVolumeToVolumeDefaultReadAndWrite");
    TString command = TStringBuilder() << "bash -c \""
        << "echo -n some_data > " << target << "/some_file"
        << ";cat " << target << "/some_file" << "\"";
    client->SetProperty(container, EPortoContainerProperty::Command, command).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    TString stdout = client->GetProperty(container, EPortoContainerProperty::StdOut).Success();
    UNIT_ASSERT_EQUAL_C("0", code, "Command expected to succeed");
    UNIT_ASSERT_EQUAL_C("some_data", stdout, "unexpected stdout: '" << stdout << "'");
}

Y_UNIT_TEST(LinkVolumeToVolumeReadOnlyWrite) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    bool readOnly = true;
    client->LinkVolume(volumeSmall.GetPath(), {""}, target, readOnly);
    auto list = client->ListVolumes(target).Success();
    UNIT_ASSERT_C(list.size(), "link volume not found");
    UNIT_ASSERT_EQUAL_C(list[0].links_size(), 2, "there must be exactly two links");
    UNIT_ASSERT(target == list[0].links(0).target() || target == list[0].links(1).target());

    TLocalContainer container(client, GetTestPrefix() + "LinkVolumeToVolumeReadOnlyWrite");
    TString command = TStringBuilder() << "touch " << target << "/some_file";
    client->SetProperty(container, EPortoContainerProperty::Command, command).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    UNIT_ASSERT_UNEQUAL_C("0", code, "Command expected to fail because of read-only, but succeed");
}

Y_UNIT_TEST(LinkVolumeToVolumeReadOnlyRead) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    bool readOnly = true;
    client->LinkVolume(volumeSmall.GetPath(), {""}, target, readOnly);
    auto list = client->ListVolumes(target).Success();
    UNIT_ASSERT_C(list.size(), "link volume not found");
    UNIT_ASSERT_EQUAL_C(list[0].links_size(), 2, "there must be exactly two links");
    UNIT_ASSERT(target  == list[0].links(0).target() || target  == list[0].links(1).target());

    TLocalContainer container(client, GetTestPrefix() + "LinkVolumeToVolumeReadOnlyRead");
    TString command = TStringBuilder() << "bash -c \""
        << "echo -n some_data > " << volumeSmall.GetPath() << "/some_file"
        << ";cat " << target << "/some_file" << "\"";
    client->SetProperty(container, EPortoContainerProperty::Command, command).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();
    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    TString stdout = client->GetProperty(container, EPortoContainerProperty::StdOut).Success();
    UNIT_ASSERT_EQUAL_C("0", code, "Command expected to succeed");
    UNIT_ASSERT_EQUAL_C("some_data", stdout, "unexpected stdout: '" << stdout << "'");
}

Y_UNIT_TEST(LinkVolumeToVolumeRequired) {
    const TPortoContainerName containerName(GetTestPrefix() + "LinkVolumeToVolumeRequired");
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();

    TLocalContainer container(client, containerName);
    TString command = "echo \"None\"";
    client->SetProperty(containerName, EPortoContainerProperty::Command, command).Success();

    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    bool readOnly = false;
    bool required = true;
    client->LinkVolume(volumeSmall.GetPath(), containerName, target, readOnly, required).Success();
    client->UnlinkVolume(volumeSmall.GetPath(), containerName).Success();

    client->Start(containerName).Error();
}

Y_UNIT_TEST(LinkVolumeToVolumeNotRequired) {
    const TPortoContainerName containerName(GetTestPrefix() + "LinkVolumeToVolumeNotRequired");
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();

    TLocalContainer container(client, containerName);
    TString command = "echo \"None\"";
    client->SetProperty(containerName, EPortoContainerProperty::Command, command).Success();

    TLocalVolume volumeSmall(client);
    TLocalVolume volumeBig(client);
    const TString target = volumeBig.GetPath() + "/link";
    bool readOnly = false;
    bool required = false;
    client->LinkVolume(volumeSmall.GetPath(), containerName, target, readOnly, required).Success();
    client->UnlinkVolume(volumeSmall.GetPath(), containerName).Success();

    client->Start(containerName).Success();
    client->WaitContainers({containerName}).Success();
}

Y_UNIT_TEST(ContainerMemoryLimit) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerMemoryLimit");

    client->SetProperty(container, EPortoContainerProperty::Command, "dd if=/dev/zero bs=64M count=1 of=/dev/null").Success();
    client->SetProperty(container, EPortoContainerProperty::MemoryLimit, "32M").Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();

    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    TString status = client->GetProperty(container, EPortoContainerProperty::ExitStatus).Success();

    UNIT_ASSERT_EQUAL_C(code, "-99", "error code expected, got " << code);
    UNIT_ASSERT_UNEQUAL_C(status, "0", "error status expected, got " << status);
}

Y_UNIT_TEST(IsContainerExists) {
    const TPortoContainerName containerName(GetTestPrefix() + "IsContainerExists");
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();

    auto result = client->List();
    UNIT_ASSERT(result);
    TVector<TPortoContainerName>& list = result.Success();
    UNIT_ASSERT(Find(list.begin(), list.end(), containerName) == list.end());

    auto existResult = client->IsContainerExists(containerName);
    UNIT_ASSERT(existResult);
    UNIT_ASSERT(!existResult.Success());

    TLocalContainer container(client, containerName);

    result = client->List();
    UNIT_ASSERT(result);
    list = result.Success();
    UNIT_ASSERT(Find(list.begin(), list.end(), containerName) != list.end());

    existResult = client->IsContainerExists(containerName);
    UNIT_ASSERT(existResult);
    UNIT_ASSERT(existResult.Success());
}

Y_UNIT_TEST(EmptyMaskList) {
    const TPortoContainerName containerName(GetTestPrefix() + "EmptyMaskList");
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();

    TLocalContainer container(client, containerName);

    auto result = client->List();
    UNIT_ASSERT(result);
    auto list = result.Success();
    UNIT_ASSERT(Find(list.begin(), list.end(), containerName) != list.end());

    UNIT_ASSERT(client->IsContainerExists(containerName));
}

Y_UNIT_TEST(CreateRecursive) {
    const TPortoContainerName rootContainerName(GetTestPrefix() + "CreateRecursive");
    const TPortoContainerName nestedContainerName(GetTestPrefix() + "CreateRecursive", "NestedContainer");
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();

    auto result = client->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(result);
    UNIT_ASSERT(!(result.Success()));

    UNIT_ASSERT(client->CreateRecursive(nestedContainerName));

    result = client->IsContainerExists(nestedContainerName);
    UNIT_ASSERT(result);
    UNIT_ASSERT(result.Success());

    UNIT_ASSERT(client->Destroy(rootContainerName));
}

Y_UNIT_TEST(SetProperties) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerSetProperties");

    TString hello = "Hello World";
    TString coreCommandExpected = "echo -n CoreCommand";
    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "echo -n " + hello},
        {EPortoContainerProperty::CoreCommand, coreCommandExpected},
    };

    client->SetProperties(container, properties).Success();
    client->Start(container).Success();
    client->WaitContainers({container}).Success();

    TString code = client->GetProperty(container, EPortoContainerProperty::ExitCode).Success();
    TString stdout = client->GetProperty(container, EPortoContainerProperty::StdOut).Success();
    TString coreCommand = client->GetProperty(container, EPortoContainerProperty::CoreCommand).Success();

    UNIT_ASSERT_EQUAL_C(code, "0", "successful code expected, got " << code);
    UNIT_ASSERT_EQUAL_C(stdout, hello, "stdout " << Quote(hello) << "expected, got " << Quote(stdout));
    UNIT_ASSERT_EQUAL_C(coreCommand, coreCommandExpected, "core command " << Quote(coreCommandExpected) <<  " expected, got " << Quote(coreCommand));

    client->Stop(container).Success();

    // Set properties with empty map does nothing
    UNIT_ASSERT_NO_EXCEPTION(client->SetProperties(container, {}).Success());
    client->Start(container).Success();
    client->WaitContainers({container}).Success();

    UNIT_ASSERT_EQUAL_C(code, "0", "successful code expected, got " << code);
    UNIT_ASSERT_EQUAL_C(stdout, hello, "stdout " << Quote(hello) << "expected, got " << Quote(stdout));
    UNIT_ASSERT_EQUAL_C(coreCommand, coreCommandExpected, "core command " << Quote(coreCommandExpected) <<  " expected, got " << Quote(coreCommand));
}

Y_UNIT_TEST(SetPropertiesConvertError) {
    TPortoClientPtr client = NPortoTestLib::GetSimplePortoClient();
    TLocalContainer container(client, GetTestPrefix() + "ContainerSetProperties");

    TString command = "echo -n Hello World";
    TString envSecret = "Key Value";

    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, command},
        {EPortoContainerProperty::EnvSecret, "Key Value"}
    };

    // Convert error
    auto result = client->SetProperties(container, properties);
    UNIT_ASSERT(!result);
    UNIT_ASSERT_EQUAL_C(result.Error().Action, "SetProperties", result.Error().Action);
    UNIT_ASSERT_EQUAL_C(result.Error().Message, "Secret env properties have unexpected format", result.Error().Message);
    UNIT_ASSERT_STRING_CONTAINS_C(result.Error().Args, "'command' -> " + Quote(command), result.Error().Args);
    UNIT_ASSERT_STRING_CONTAINS_C(result.Error().Args, "'env_secret' -> <hidden>", result.Error().Args);
}

}

} // namespace NInfra::NPodAgent::NPortoTest
