#include "posix_worker.h"

#include <util/folder/filelist.h>
#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/system/fstat.h>
#include <util/thread/factory.h>

namespace NInfra::NPodAgent {

NThreading::TFuture<TPosixResult> TPosixWorker::RenameAsync(const TString& oldPath, const TString& newPath) {
    return NThreading::Async(
        [oldPath = oldPath, newPath = newPath]() {
            return Rename(oldPath, newPath);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixCheckResult> TPosixWorker::ExistsAsync(const TString& path) {
    return NThreading::Async(
        [path = path]() {
            return TPosixCheckResult { NFs::Exists(path) };
        }, *Pool_
    );
}

NThreading::TFuture<TPosixCheckResult> TPosixWorker::CheckSameINodeAsync(const TString& path1, const TString& path2) {
    return NThreading::Async(
        [path1 = path1, path2 = path2]() {
            return TPosixCheckResult { CheckSameINode(path1, path2) };
        }, *Pool_
    );
}

NThreading::TFuture<TPosixTimeResult> TPosixWorker::GetFileModificationTimeAsync(const TString &path) {
    return NThreading::Async(
        [path](){
            return TPosixTimeResult { GetFileModificationTime(path) };
        }, *Pool_
    );
}

NThreading::TFuture<TPosixTimeResultRecursive> TPosixWorker::GetFileModificationTimeRecursiveAsync(const TString& path) {
    return NThreading::Async(
        [path = path]() {
            return TPosixTimeResultRecursive { GetFileModificationTimeRecursive(path) };
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::MakeDirectoryAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return MakeDirectory(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::MakeFileAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return MakeFile(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::MakeDirectoryRecursiveAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return MakeDirectoryRecursive(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixCheckResult> TPosixWorker::IsDirectoryAsync(const TString& sourcePath) {
    return NThreading::Async(
        [sourcePath]() {
            TFsPath path(sourcePath);
            return TPosixCheckResult { path.IsDirectory() };
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::MakeHardLinkAsync(const TString &existingPath, const TString& linkPath) {
    return NThreading::Async(
        [existingPath = existingPath, linkPath = linkPath]() {
            return MakeHardLink(existingPath, linkPath);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::MakeSymLinkAsync(const TString &existingPath, const TString& linkPath) {
    return NThreading::Async(
        [existingPath = existingPath, linkPath = linkPath]() {
            return MakeSymLink(existingPath, linkPath);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixStringVectorResult> TPosixWorker::ListAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return List(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::RemoveRecursiveAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return RemoveRecursive(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::RemoveFileAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return RemoveFile(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixStringResult> TPosixWorker::ReadFileDataAsync(const TString& path) {
    return NThreading::Async(
        [path]() {
            return ReadFileData(path);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::WriteFileDataAsync(const TString& path, const TString& data) {
    return NThreading::Async(
        [path, data]() {
            return WriteFileData(path, data);
        }, *Pool_
    );
}

NThreading::TFuture<TPosixResult> TPosixWorker::SetFileModeRecursiveAsync(const TString &path, EFileAccessMode mode) {
    return NThreading::Async(
        [path, mode]() {
            return SetFileModeRecursive(path, mode);
        }, *Pool_
    );
}

TPosixStringResult TPosixWorker::ReadFileData(const TString& path) {
    if (!NFs::Exists(path)){
        return ConstructPosixError();
    }

    TString data;
    try {
        TUnbufferedFileInput fileInput(path);
        data = fileInput.ReadAll();
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return data;
}

TPosixResult TPosixWorker::WriteFileData(const TString& path, const TString& data) {
    try {
        TUnbufferedFileOutput fileOutput(path);
        fileOutput.Write(data.c_str(), data.size());
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return TPosixResult::DefaultSuccess();
}

TPosixCheckResult TPosixWorker::CheckSameINode(const TString& path1, const TString& path2) {
    if (!NFs::Exists(path1)) {
        return ConstructPosixError();
    }
    if (!NFs::Exists(path2)) {
        return ConstructPosixError();
    }

    return TFileStat(path1).INode == TFileStat(path2).INode;
}

TPosixTimeResult TPosixWorker::GetFileModificationTime(const TString& path) {
    if (!NFs::Exists(path)){
        return ConstructPosixError();
    }

    time_t modificationTime = 0;
    try {
        modificationTime = TFileStat(path).MTime;
        if (!modificationTime) {
            return ConstructPosixError();
        }
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return modificationTime;
}

TPosixTimeResultRecursive TPosixWorker::GetFileModificationTimeRecursive(const TString& sourcePath) {
    TFsPath path(sourcePath);

    TList<time_t> modificationTimes;

    // For symlinks path.Exists() returnsт false ans modificationTime == 0
    // because of this, we do not look at them
    if (path.IsSymlink()) {
        return modificationTimes;
    }

    if (!path.Exists()){
        return ConstructPosixError();
    }

    try {
        time_t  modificationTime = TFileStat(path.GetPath()).MTime;
        if (!modificationTime){
            return ConstructPosixError();
        }

        modificationTimes.push_back(modificationTime);

        if (path.IsDirectory()){
            TVector<TFsPath> childrenPath;
            path.List(childrenPath);

            for (const auto& childPath: childrenPath){
                auto childResult = OUTCOME_TRYX(GetFileModificationTimeRecursive(childPath.GetPath()));
                modificationTimes.merge(childResult);
            }
        }
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return modificationTimes;
}

TPosixResult TPosixWorker::MakeHardLink(const TString& existingPath, const TString& linkPath) {
    if (!NFs::HardLink(existingPath, linkPath)){
        return ConstructPosixError();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::MakeSymLink(const TString& existingPath, const TString& linkPath) {
    if (!NFs::SymLink(existingPath, linkPath)){
        return ConstructPosixError();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::Rename(const TString& oldPath, const TString& newPath) {
    if (!NFs::Rename(oldPath, newPath)){
        return ConstructPosixError();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::MakeDirectory(const TString& path) {
    if (!NFs::MakeDirectory(path)){
        return ConstructPosixError();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::MakeDirectoryRecursive(const TString& path) {
    if (!NFs::MakeDirectoryRecursive(path)){
        return ConstructPosixError();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::MakeFile(const TString& path) {
    if (NFs::Exists(path)){
        return ConstructPosixError();
    }

    try {
        TFile file(path, CreateNew);
    } catch(...) {
        return ConstructPosixErrorFromException();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixStringVectorResult TPosixWorker::List(const TString& path) {
    TDirsList dirList;
    try {
        dirList.Fill(path);
    } catch (...) {
        return ConstructPosixErrorFromException();
    }

    TVector<TString> res;
    for (const char* curDirName = dirList.Next(); curDirName; curDirName = dirList.Next()) {
        res.emplace_back(curDirName);
    }

    return res;
}

TPosixResult TPosixWorker::RemoveRecursive(const TString& path) {
    try {
        NFs::RemoveRecursive(path);
    } catch (...) {
        return ConstructPosixErrorFromException();
    }

    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::RemoveFile(const TString& path) {
    try {
        NFs::Remove(path);
    } catch (...) {
        return ConstructPosixErrorFromException();
    }

    return TPosixResult::DefaultSuccess();
}

// got from yt sources
void TPosixWorker::Chmod(const TString& path, int mode)
{
#ifdef _linux_
     int result = ::Chmod(path.data(), mode);
     if (result < 0) {
         ythrow TSystemError() << "Failed to change mode of " << path
             << " to mode: " << mode << " with system error: " << LastSystemError()
             << " " << strerror(LastSystemError());
     }
#else
    Y_UNUSED(path, mode);
    ThrowNotSupported();
#endif
}

int TPosixWorker::FileAccessModeToInteger(EFileAccessMode mode) {
    switch (mode) {
        case EFileAccessMode::Mode_660:
            return 0660;
        case EFileAccessMode::Mode_600:
            return 0600;
        default:
            ythrow TBadCastException() << "Failed to convert EFileAccessMode " << mode << " to integer value ";
    }
}

TPosixResult TPosixWorker::SetFileMode(const TString& path, EFileAccessMode mode) {
    if (!NFs::Exists(path)){
        return ConstructPosixError();
    }

    if (EFileAccessMode::Mode_UNMODIFIED == mode) {
        return TPosixResult::DefaultSuccess();
    }

    try {
        int intMode = FileAccessModeToInteger(mode);
        Chmod(path, intMode);
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return TPosixResult::DefaultSuccess();
}

TPosixResult TPosixWorker::SetFileModeRecursive(const TString& sourcePath, EFileAccessMode mode) {
    TFsPath path(sourcePath);

    // For symlinks path.Exists() returns false
    // also we dont want to change permissions for files under symlinks
    // because they can be in another dir where we don't want to change permissions
    // because of this, we do not look at them
    if (path.IsSymlink()) {
        return TPosixResult::DefaultSuccess();
    }

    if (!path.Exists()){
        return ConstructPosixError();
    }

    try {
        if (path.IsFile()) {
            OUTCOME_TRYX(SetFileMode(path.GetPath(), mode));
        }

        if (path.IsDirectory()){
            TVector<TFsPath> childrenPath;
            path.List(childrenPath);

            for (const auto& childPath: childrenPath){
                OUTCOME_TRYX(SetFileModeRecursive(childPath.GetPath(), mode));
            }
        }
    } catch (...) {
        return ConstructPosixErrorFromException();
    }
    return TPosixResult::DefaultSuccess();
}

TPosixError TPosixWorker::ConstructPosixError() {
    return TPosixError{LastSystemError(), TString(LastSystemErrorText())};
}

TPosixError TPosixWorker::ConstructPosixErrorFromException() {
    return TPosixError{1, CurrentExceptionMessage()};
}

} // namespace NInfra::NPodAgent
