#pragma once

#include "posix_types.h"

#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/future/async.h>

#include <util/generic/list.h>
#include <util/generic/ptr.h>

namespace NInfra::NPodAgent {

class TPosixWorker;
using TPosixWorkerPtr = TAtomicSharedPtr<TPosixWorker>;

class TPosixWorker : private TNonCopyable {
public:
    explicit TPosixWorker(TAtomicSharedPtr<IThreadPool> pool)
        : Pool_(pool)
    {}
    virtual ~TPosixWorker() = default;

    virtual NThreading::TFuture<TPosixResult> RenameAsync(const TString& oldPath, const TString& newPath);

    virtual NThreading::TFuture<TPosixCheckResult> ExistsAsync(const TString& path);

    virtual NThreading::TFuture<TPosixCheckResult> CheckSameINodeAsync(const TString& path1, const TString& path2);

    virtual NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString &path);
    virtual NThreading::TFuture<TPosixTimeResultRecursive> GetFileModificationTimeRecursiveAsync(const TString& path);


    virtual NThreading::TFuture<TPosixCheckResult> IsDirectoryAsync(const TString& path);
    virtual NThreading::TFuture<TPosixResult> MakeDirectoryAsync(const TString& path);
    virtual NThreading::TFuture<TPosixResult> MakeFileAsync(const TString& path);
    virtual NThreading::TFuture<TPosixResult> MakeDirectoryRecursiveAsync(const TString& path);

    virtual NThreading::TFuture<TPosixResult> MakeHardLinkAsync(const TString &existingPath, const TString& linkPath);
    virtual NThreading::TFuture<TPosixResult> MakeSymLinkAsync(const TString &existingPath, const TString& linkPath);

    virtual NThreading::TFuture<TPosixStringVectorResult> ListAsync(const TString& path);

    virtual NThreading::TFuture<TPosixResult> RemoveRecursiveAsync(const TString& path);
    virtual NThreading::TFuture<TPosixResult> RemoveFileAsync(const TString& path);

    virtual NThreading::TFuture<TPosixStringResult> ReadFileDataAsync(const TString& path);
    virtual NThreading::TFuture<TPosixResult> WriteFileDataAsync(const TString& path, const TString& data);

    virtual NThreading::TFuture<TPosixResult> SetFileModeRecursiveAsync(const TString& path, EFileAccessMode mode);

private:
    static TPosixStringResult ReadFileData(const TString& path);
    static TPosixResult WriteFileData(const TString& path, const TString& data);

    static TPosixCheckResult CheckSameINode(const TString& path1, const TString& path2);

    static TPosixTimeResult GetFileModificationTime(const TString& path);
    static TPosixTimeResultRecursive GetFileModificationTimeRecursive(const TString& sourcePath);

    static TPosixResult MakeHardLink(const TString& existingPath, const TString& linkPath);
    static TPosixResult MakeSymLink(const TString& existingPath, const TString& linkPath);

    static TPosixResult Rename(const TString& oldPath, const TString& newPath);

    static TPosixResult MakeDirectory(const TString& path);
    static TPosixResult MakeDirectoryRecursive(const TString& path);
    static TPosixResult MakeFile(const TString& path);

    static TPosixStringVectorResult List(const TString& path);

    static TPosixResult RemoveRecursive(const TString& path);
    static TPosixResult RemoveFile(const TString& path);

    static void Chmod(const TString& path, int mode);
    static int FileAccessModeToInteger(EFileAccessMode mode);
    static TPosixResult SetFileMode(const TString& path, EFileAccessMode mode);
    static TPosixResult SetFileModeRecursive(const TString& path, EFileAccessMode mode);

    static TPosixError ConstructPosixError();
    static TPosixError ConstructPosixErrorFromException();

private:
    TAtomicSharedPtr<IThreadPool> Pool_;
};

} // namespace NInfra::NPodAgent
