#include <library/cpp/testing/unittest/registar.h>

#include "posix_worker.h"

#include <util/folder/path.h>
#include <util/generic/algorithm.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestPosixWorker {

Y_UNIT_TEST_SUITE(PosixWorkerSuite) {

void CreateFile(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

TPosixWorkerPtr CreatePosixWorker() {
    TAtomicSharedPtr<TFakeThreadPool> queue = new TFakeThreadPool;
    TPosixWorkerPtr posix = new TPosixWorker(queue);
    return posix;
}

void EnsureDoesNotExist(const TString& path) {
    if (NFs::Exists(path)) {
        try {
            NFs::RemoveRecursive(path);
        } catch(...) {
            NFs::Remove(path);
        }
    }
}

Y_UNIT_TEST(RenameFileSuccess) {
    CreateFile("flag");
    auto posix = CreatePosixWorker();
    auto result = posix->RenameAsync("flag", "flag.moved").GetValue();

    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(nullptr != ::fopen("flag.moved", "r"));
}

Y_UNIT_TEST(RenameFileFailure) {
    auto posix = CreatePosixWorker();
    auto result = posix->RenameAsync("flag", "flag.moved").GetValue();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(FileExistsSuccess) {
    TString fileName = "flag";
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    auto result = posix->ExistsAsync(fileName).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(result.Success());
}

Y_UNIT_TEST(MakeHardLinkSuccess) {
    TString fileName = "flag";
    TString linkName = "flag_hard_link";
    EnsureDoesNotExist(fileName);
    EnsureDoesNotExist(linkName);
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    auto result = posix->MakeHardLinkAsync(fileName, linkName).GetValue();
    UNIT_ASSERT((bool)result);
}

Y_UNIT_TEST(MakeSymLinkSuccess) {
    TString fileName = "flag";
    TString linkName = "flag_sym_link";
    EnsureDoesNotExist(fileName);
    EnsureDoesNotExist(linkName);
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    auto result = posix->MakeSymLinkAsync(fileName, linkName).GetValue();
    UNIT_ASSERT((bool)result);
}

Y_UNIT_TEST(FileExistsFailure) {
    TString fileName = "flag";
    EnsureDoesNotExist(fileName);
    auto posix = CreatePosixWorker();
    auto result = posix->ExistsAsync(fileName).GetValue().Success();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(MakeDirectorySuccess) {
    TString path = "directory";
    EnsureDoesNotExist(path);
    auto posix = CreatePosixWorker();
    auto result = posix->MakeDirectoryAsync(path).GetValue();
    UNIT_ASSERT((bool)result);

    auto existResult = posix->ExistsAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(existResult.Success());
}

Y_UNIT_TEST(MakeDirectoryRecursiveSuccess) {
    TString path = "directory/subdirectory";
    EnsureDoesNotExist("directory");
    auto posix = CreatePosixWorker();
    auto result = posix->MakeDirectoryRecursiveAsync(path).GetValue();
    UNIT_ASSERT((bool)result);

    auto existResult = posix->ExistsAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(existResult.Success());
}

Y_UNIT_TEST(IsDirectorySuccess) {
    TString path = "directory";
    EnsureDoesNotExist("directory");
    UNIT_ASSERT(NFs::MakeDirectory(path));
    auto posix = CreatePosixWorker();
    auto result = posix->IsDirectoryAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(result.Success());
}

Y_UNIT_TEST(IsDirectoryFailure) {
    TString path = "directory2";
    EnsureDoesNotExist("directory2");
    auto posix = CreatePosixWorker();
    auto result = posix->IsDirectoryAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(!result.Success());
}

Y_UNIT_TEST(ListSuccess) {
    TString path = "directory";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(path);
    NFs::MakeDirectory(path);
    TVector<TString> expectedList;
    for (i32 i = 0; i < 5; ++i) {
        expectedList.push_back(ToString(i));
        TString nestedPath = path + "/" + ToString(i);
        EnsureDoesNotExist(nestedPath);
        NFs::MakeDirectory(nestedPath);
    }

    auto result = posix->ListAsync(path).GetValue();
    UNIT_ASSERT((bool)result);

    Sort(result.Success().begin(), result.Success().end());
    UNIT_ASSERT(result.Success() == expectedList);
}

Y_UNIT_TEST(ListFailure) {
    TString path = "directory";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(path);
    auto result = posix->ListAsync(path);
    UNIT_ASSERT(!(bool)result.GetValue());
}

Y_UNIT_TEST(RemoveRecursiveSuccess) {
    TString path = "directory";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(path);
    NFs::MakeDirectory(path);
    auto result = posix->RemoveRecursiveAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(!NFs::Exists(path));
}

Y_UNIT_TEST(RemoveRecursiveSuccessFolderDoesNotExist) {
    TString path = "directory";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(path);
    auto result = posix->RemoveRecursiveAsync(path).GetValue();
    UNIT_ASSERT((bool)result);
}

Y_UNIT_TEST(RemoveRecursiveFailureFileInsteadOfFolder) {
    TString fileName = "file";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    CreateFile(fileName);
    auto result = posix->RemoveRecursiveAsync(fileName).GetValue();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(GetFileModificationTimeSuccess) {
    TString fileName = "file";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    CreateFile(fileName);
    auto result = posix->GetFileModificationTimeAsync(fileName).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(result.Success() <= Seconds());
}

Y_UNIT_TEST(GetModificationTimeWhenFileNotExist) {
    TString fileName = "file";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->GetFileModificationTimeAsync(fileName).GetValue();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(GetFileModificationTimeRecursiveSuccess) {
    TString path = "directory";
    auto posix = CreatePosixWorker();

    EnsureDoesNotExist(path);
    NFs::MakeDirectory(path);
    TFsPath dir(path);
    dir.Child("child_dir_1").MkDir();
    dir.Child("child_dir_2").MkDir();
    TString childFile = TStringBuilder() << path << "/child_dir_1/child_file_1";
    TString childFileSymLink = TStringBuilder() << path << "/child_dir_1/child_file_sym_link";
    CreateFile(childFile);
    NFs::SymLink(childFile, childFileSymLink);
    auto result = posix->GetFileModificationTimeRecursiveAsync(path).GetValue();
    auto modificationTimes = result.Success();
    auto modificationTimesSorted = modificationTimes;
    modificationTimesSorted.sort();

    UNIT_ASSERT_EQUAL(modificationTimes, modificationTimesSorted);
    UNIT_ASSERT_EQUAL(modificationTimes.size(), 4);
    UNIT_ASSERT((bool)result);
}

Y_UNIT_TEST(GetModificationTimeRecursiveWhenFileNotExist) {
    TString fileName = "file";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->GetFileModificationTimeRecursiveAsync(fileName).GetValue();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(SetFileModeRecursiveWhenFileNotExist) {
    TString fileName = "file";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->SetFileModeRecursiveAsync(fileName, EFileAccessMode::Mode_660).GetValue();
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(SetFileModeRecursiveSuccess) {
    TString path = "directory";
    auto posix = CreatePosixWorker();

    EnsureDoesNotExist(path);
    NFs::MakeDirectory(path);
    TFsPath dir(path);
    dir.Child("child_dir_1").MkDir();
    dir.Child("child_dir_2").MkDir();
    TString childFile = TStringBuilder() << path << "/child_dir_1/child_file_1";
    TString childFileSymLink = TStringBuilder() << path << "/child_dir_1/child_file_sym_link";
    CreateFile(childFile);
    NFs::SymLink(childFile, childFileSymLink);
    int intMode = 0600;
    auto result = posix->SetFileModeRecursiveAsync(path, EFileAccessMode::Mode_600).GetValue();

    UNIT_ASSERT((bool)result);

    // Check that mode where applied only to file
    UNIT_ASSERT_EQUAL((int)TFileStat(childFile).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(path).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(childFileSymLink).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory/child_dir_1").Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory/child_dir_2").Mode & 0x00000FFF, intMode);
    NFs::RemoveRecursive(path);
    EnsureDoesNotExist(path);
}

Y_UNIT_TEST(SetFileModeRecursiveWhenModeUnmodified) {
    TString path = "directory";
    auto posix = CreatePosixWorker();

    EnsureDoesNotExist(path);
    NFs::MakeDirectory(path);
    TFsPath dir(path);
    dir.Child("child_dir_1").MkDir();
    dir.Child("child_dir_2").MkDir();
    TString childFile = TStringBuilder() << path << "/child_dir_1/child_file_1";
    TString childFileSymLink = TStringBuilder() << path << "/child_dir_1/child_file_sym_link";
    CreateFile(childFile);
    NFs::SymLink(childFile, childFileSymLink);
    int intMode = 0664;
    auto result = posix->SetFileModeRecursiveAsync(path, EFileAccessMode::Mode_UNMODIFIED).GetValue();

    UNIT_ASSERT((bool)result);

    // Check that mode where not applied
    UNIT_ASSERT_EQUAL((int)TFileStat(childFile).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(path).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(childFileSymLink).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory/child_dir_1").Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory/child_dir_2").Mode & 0x00000FFF, intMode);
    NFs::RemoveRecursive(path);
    EnsureDoesNotExist(path);
}

Y_UNIT_TEST(RemoveFileSuccessWhenFileExist) {
    TString fileName = "flag";
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->RemoveFileAsync(fileName).GetValue();
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(!NFs::Exists(fileName));
}

Y_UNIT_TEST(RemoveFileSuccessWhenFileNotExist) {
    TString fileName = "flag";
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->RemoveFileAsync(fileName).GetValue();
    UNIT_ASSERT((bool)result);
}

Y_UNIT_TEST(ReadFileDataFailureWhenFileNotExist) {
    TString fileName = "file";
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    auto result = posix->ReadFileDataAsync(fileName).GetValue();
    UNIT_ASSERT(!(bool)result);
    UNIT_ASSERT(!NFs::Exists(fileName));
}

Y_UNIT_TEST(ReadFileDataSuccessWhenFileExist) {
    TString fileName = "file";
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    TString data = "data";
    auto writeResult = posix->WriteFileDataAsync(fileName, data).GetValue();
    UNIT_ASSERT((bool)writeResult);
    UNIT_ASSERT(NFs::Exists(fileName));

    auto readResult = posix->ReadFileDataAsync(fileName).GetValue();
    UNIT_ASSERT((bool)readResult);
    UNIT_ASSERT(readResult.Success().equal(data));
}

Y_UNIT_TEST(RepeatFileWriteShouldOverwriteFileData) {
    TString fileName = "file";
    CreateFile(fileName);
    auto posix = CreatePosixWorker();
    EnsureDoesNotExist(fileName);
    TString data1 = "data1";
    auto writeResult1 = posix->WriteFileDataAsync(fileName, data1).GetValue();
    UNIT_ASSERT((bool)writeResult1);
    TString data2 = "data2";
    auto writeResult2 = posix->WriteFileDataAsync(fileName, data2).GetValue();
    UNIT_ASSERT((bool)writeResult2);

    auto readResult = posix->ReadFileDataAsync(fileName).GetValue();
    UNIT_ASSERT((bool)readResult);
    UNIT_ASSERT(readResult.Success().equal(data2));
}

Y_UNIT_TEST(MakeFileShouldFailIfFileExists) {
    TString fileName = "file";
    CreateFile(fileName);
    UNIT_ASSERT(NFs::Exists(fileName));
    auto posix = CreatePosixWorker();
    auto createFileResult = posix->MakeFileAsync(fileName).GetValue();
    UNIT_ASSERT(!(bool)createFileResult);
}

Y_UNIT_TEST(MakeFileShouldBeSuccessIfFileNotExists) {
    TString fileName = "file";
    EnsureDoesNotExist(fileName);
    UNIT_ASSERT(!NFs::Exists(fileName));
    auto posix = CreatePosixWorker();
    auto createFileResult = posix->MakeFileAsync(fileName).GetValue();
    UNIT_ASSERT((bool)createFileResult);
    UNIT_ASSERT(NFs::Exists(fileName));
}

Y_UNIT_TEST(CheckSameINode) {
    const TString file = "file";
    const TString fileOther = "file_other";
    const TString fileHardLink = "file_hardlink";
    const TString nonExistentFile = "non_existent_file";
    EnsureDoesNotExist(file);
    EnsureDoesNotExist(fileOther);
    EnsureDoesNotExist(fileHardLink);
    EnsureDoesNotExist(nonExistentFile);
    auto posix = CreatePosixWorker();

    CreateFile(file);
    CreateFile(fileOther);
    UNIT_ASSERT(NFs::HardLink(file, fileHardLink));

    UNIT_ASSERT(!posix->CheckSameINodeAsync(file, fileOther).GetValue().Success());
    UNIT_ASSERT(posix->CheckSameINodeAsync(file, fileHardLink).GetValue().Success());
    UNIT_ASSERT(!(bool)posix->CheckSameINodeAsync(nonExistentFile, file).GetValue());
    UNIT_ASSERT(!(bool)posix->CheckSameINodeAsync(file, nonExistentFile).GetValue());
}

}

} // namespace NInfra::NPodAgent::NTestPosixWorker
