#include "push_event_mock.h"
#include "push_log_splitter.h"

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <util/string/split.h>

namespace NInfra::NPodAgent::NPushLogFormatterTest {
    static void PartOfSymbolsCanBeFormatedScenario(ui64 maxFormattedLogSize, ui64 maxFormattedRawLogSize) {
        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);

        TString firstLogPart = "\nstring_1\n\n";
        TString secondLogPart = "string_2\n";

        TString log = TStringBuilder() << firstLogPart << secondLogPart;

        TPushEventConstPtr pushEvent = new TMockPushEvent();
        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, 1, 1);

        UNIT_ASSERT((bool) logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();

        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, firstLogPart.size());

        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.size(), 1);
        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.at(0), "string_1");
    }

    static void FormatReturnsEmptyWhenMessagePackedOfMaxSizeScenario(ui64 maxFormattedLogSize, ui64 prevFormattedLogSize, ui64 maxFormattedRawLogSize, ui64 prevFormattedRawLogSize) {
        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);

        TString log = "string_1\n\n";

        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, prevFormattedLogSize, prevFormattedRawLogSize);

        UNIT_ASSERT(logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();

        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, 0);
        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.size(), 0);
    }

Y_UNIT_TEST_SUITE(PushLogSplitterSuite) {

    Y_UNIT_TEST(TestAllSymbolsCanBeFormated) {
        ui64 maxFormattedLogSize = 1000;
        ui64 maxFormattedRawLogSize = 1000;

        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);

        TString log = "\nstring_1\n\nstring_2\n";

        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, 1, 1);

        UNIT_ASSERT((bool)logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();
        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, log.size());

        TVector<TString> formattedLogStrings;
        StringSplitter(log).Split('\n').SkipEmpty().Collect(&formattedLogStrings);

        UNIT_ASSERT_EQUAL(formattedLogStrings.size(), formattedLog.FormattedData.size());
        UNIT_ASSERT_EQUAL( formattedLogStrings.at(0), formattedLog.FormattedData.at(0));
        UNIT_ASSERT_EQUAL( formattedLogStrings.at(1), formattedLog.FormattedData.at(1));
    }

    Y_UNIT_TEST(TestPartOfSymbolsCanBeFormated) {
        ui64 maxFormattedLogSize1 = 27;
        ui64 maxFormattedRawLogSize1 = 1000;

        ui64 maxFormattedLogSize2 = 1000;
        ui64 maxFormattedRawLogSize2 = 10;

        PartOfSymbolsCanBeFormatedScenario(maxFormattedLogSize1, maxFormattedRawLogSize1);
        PartOfSymbolsCanBeFormatedScenario(maxFormattedLogSize2, maxFormattedRawLogSize2);
    }

    Y_UNIT_TEST(TestFormatReturnsEmptyWhenInputLogIsEmpty) {
        ui64 maxFormattedLogSize = 400;
        ui64 maxFormattedRawLogSize = 400;

        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);
        TString log = "";

        TPushEventConstPtr pushEvent = new TMockPushEvent();
        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, 1, 1);

        UNIT_ASSERT(logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();

        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, 0);
        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.size(), 0);
    }

    Y_UNIT_TEST(TestFormatReturnsEmptyWhenMessagePackedOfMaxSize) {
        ui64 maxFormattedLogSize = 400;
        ui64 maxFormattedRawLogSize = 400;
        ui64 prevFormattedLogSize1 = maxFormattedLogSize - 1;
        ui64 prevFormattedRawLogSize1 = 0;

        ui64 prevFormattedLogSize2 = 0;
        ui64 prevFormattedRawLogSize2 = maxFormattedRawLogSize - 1;

        FormatReturnsEmptyWhenMessagePackedOfMaxSizeScenario(maxFormattedLogSize, prevFormattedLogSize1, maxFormattedRawLogSize, prevFormattedRawLogSize1);
        FormatReturnsEmptyWhenMessagePackedOfMaxSizeScenario(maxFormattedLogSize, prevFormattedLogSize2, maxFormattedRawLogSize, prevFormattedRawLogSize2);
    }

    Y_UNIT_TEST(TestFormatLogWithEmptyStrings) {
        ui64 maxFormattedLogSize = 400;
        ui64 maxFormattedRawLogSize = 400;

        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);

        TString log = "\n\n\n\n\n";

        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, 0, 0);

        UNIT_ASSERT(logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();
        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, log.size());
        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.size(), 0);
    }

    Y_UNIT_TEST(TestSkipLastLog) {
        ui64 maxFormattedLogSize = 10000;
        ui64 maxFormattedRawLogSize = 10000;

        TPushLogFormatterPtr logsFormatter = new TPushLogSplitter(maxFormattedLogSize, maxFormattedRawLogSize);

        TString firstLogPart = "\nstring_1\n\n";
        TString secondLogPart = "string_2";

        TString log = TStringBuilder() << firstLogPart << secondLogPart;

        TExpected<TFormattedLog, TPushClientError> logFormatterResult = logsFormatter->Format(log, 0, 0);

        UNIT_ASSERT(logFormatterResult);

        TFormattedLog formattedLog = logFormatterResult.Success();

        UNIT_ASSERT_EQUAL(formattedLog.NumBytesFormatted, firstLogPart.size());

        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.size(), 1);
        UNIT_ASSERT_EQUAL(formattedLog.FormattedData.at(0), "string_1");
    }
}

} //namespace NInfra::NPodAgent::NSimpleClientTest


