#include "push_session_impl.h"

#include <infra/pod_agent/libs/push_client/utils/push_client_utils.h>

namespace NInfra::NPodAgent {

TPushSessionImpl::TPushSessionImpl(
    TGrpcChannelPtr grpcChannel
    , const ELogType& typeOfLog
    , const TPushContainer& pushContainer
    , const TString& staticSecret
)
    : Stream_(nullptr)
    , Context_(nullptr)
    , StaticSecret_(staticSecret)
    , LogSeqNumber_(START_LOG_SEQ_NUMBER)
    , TypeOfLog_(typeOfLog)
    , PushContainer_(pushContainer)
{
    Stub_ = THolder<NUnifiedAgentProto::UnifiedAgentService::Stub>(NUnifiedAgentProto::UnifiedAgentService::NewStub(grpcChannel).release());
}

bool TPushSessionImpl::IsActive() const {
    return Stream_ != nullptr;
}

const TString& TPushSessionImpl::GetId() const {
    return SessionId_;
}

ui64 TPushSessionImpl::GetSeqNumber() const {
    return LogSeqNumber_;
}

TExpected<void, TPushClientError> TPushSessionImpl::SendLog(const TVector<TString>& payload, ui64 timestampMs) {
    TPushLogRequest logRequest;

    if (payload.empty()) {
        return TPushClientError{EPushClientError::NoData, "no data while log send when using new protocol"};
    }

    ui64 currentSeqNumber = LogSeqNumber_;

    for (const auto& batchPiece: payload) {
        logRequest.mutable_data_batch()->add_seq_no(currentSeqNumber);
        *(logRequest.mutable_data_batch()->add_payload()) = batchPiece;
        logRequest.mutable_data_batch()->add_timestamp(timestampMs);
        currentSeqNumber++;
    }

    TPushLogResponse logResponse = OUTCOME_TRYX(NPushClientUtils::MakeRequest(Stream_, logRequest));

    if (!logResponse.has_ack()) {
        return TPushClientError{EPushClientError::InvalidResponse, "not ack response while log send"};
    }

    return TExpected<void, TPushClientError>::DefaultSuccess();
}

TExpected<void, TPushClientError> TPushSessionImpl::Activate(const TString& logName, const TString& sessionId) {

    TPushLogRequest logRequest;

    TGrpcPushContextPtr context = new grpc::ClientContext;
    TGrpcPushStreamPtr stream = (Stub_->Session(context.Get())).release();

    if (!sessionId.empty()) {
        logRequest.mutable_initialize()->set_session_id(sessionId);
    }

    AddMetaObject(logRequest, DEPLOY_BOX_KEY, PushContainer_.BoxId);
    AddMetaObject(logRequest, DEPLOY_WORKLOAD_KEY, PushContainer_.WorkloadId);
    AddMetaObject(logRequest, DEPLOY_CONTAINER_ID_KEY, PushContainer_.Container);
    AddMetaObject(logRequest, DEPLOY_LOGGER_NAME_KEY, ToString(TypeOfLog_));

    logRequest.mutable_initialize()->set_shared_secret_key(StaticSecret_);

    TPushLogResponse logResponse = OUTCOME_TRYX(NPushClientUtils::MakeRequest(stream, logRequest));

    if (!logResponse.has_initialized()) {
        return TPushClientError{EPushClientError::InvalidResponse, TStringBuilder() << "not init response while " << TypeOfLog_ << " session init. " << "logname: " << logName << ", sessionId: " << sessionId};
    }

    TString newSessionId = logResponse.initialized().session_id();

    if (newSessionId.empty()) {
        return TPushClientError{EPushClientError::InvalidResponse, TStringBuilder() << "session id is empty while " << TypeOfLog_ << " session init. " << "logname: " << logName << ", sessionId: " << sessionId};
    }

    Stream_ = stream;
    Context_ = context;
    SessionId_ = newSessionId;
    LogSeqNumber_ = logResponse.initialized().last_seq_no() + 1;

    return TExpected<void, TPushClientError>::DefaultSuccess();
}

void TPushSessionImpl::IncreaseLogSeqNumber(ui32 increaseFactor) {
    LogSeqNumber_ += increaseFactor;
}

void TPushSessionImpl::Deactivate() {
    Stream_.Reset();
    Context_.Reset();
    Stream_ = nullptr;
    Context_ = nullptr;
}

void TPushSessionImpl::AddMetaObject(TPushLogRequest& request, const TString& key, const TString& value) {
    auto meta = request.mutable_initialize()->add_meta();
    meta->set_name(key);
    meta->set_value(value);
}


} // namespace NInfra::NPodAgent
