#include "simple_client.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/utils/logs_transmitter_utils.h>
#include <infra/pod_agent/libs/push_client/utils/push_client_utils.h>
#include <util/string/split.h>

namespace NInfra::NPodAgent {

TSimplePushClient::TSimplePushClient(
    TPushLogFormatterPtr pushLogFormatter
    , TSessionHolderPtr sessionHolder
    , ui64 maxLogSizeToRead
    , TLogsTransmitterStatisticsPtr statistics
)
    : PushLogFormatter_(pushLogFormatter)
    , SessionHolder_(sessionHolder)
    , MaxLogSizeToRead_(maxLogSizeToRead)
    , Statistics_(statistics)
{}

TExpected<TSendLogResult, TPushClientError> TSimplePushClient::Send(TPushEventConstPtr pushEvent) {
    const TPushContainer& pushContainer = pushEvent->GetPushContainer();
    TPushSessionPtr session = OUTCOME_TRYX(GetSession(pushContainer));

    TInstant now = TInstant::Now();

    auto formattedLog = OUTCOME_TRYX(ReadAndFormatLogs(pushEvent));

    ui32 formattedWithMetaLogSize = FormattedLogWithMetaSizeBytes(formattedLog.FormattedData);
    if (!formattedWithMetaLogSize) {
        return TSendLogResult{formattedLog.NumBytesFormatted, formattedWithMetaLogSize};
    }

    const auto sendLogResult = session->SendLog(formattedLog.FormattedData, now.MicroSeconds());

    if (!(bool)sendLogResult) {
        Statistics_->ErrorFromPushAgentOccured();
        OUTCOME_TRYV(SessionHolder_->DeactivateSession(pushContainer));
        return sendLogResult.Error();
    }

    OUTCOME_TRYV(SessionHolder_->IncreaseLogSeqNumber(formattedLog.FormattedData.size(), pushContainer));

    return TSendLogResult{formattedLog.NumBytesFormatted, formattedWithMetaLogSize};
}

TExpected<TPushSessionPtr, TPushClientError> TSimplePushClient::GetSession(const TPushContainer& pushContainer) {
    if (!OUTCOME_TRYX(SessionHolder_->IsActive(pushContainer))) {
        return SessionHolder_->ActivateSession(pushContainer);
    }

    return SessionHolder_->GetSession(pushContainer);
}

TExpected<TFormattedLog, TPushClientError> TSimplePushClient::ReadAndFormatLogs(TPushEventConstPtr pushEvent) {
    TFormattedLog formattedLog{};

    while (true) {
        TString log = OUTCOME_TRYX(NPushClientUtils::ReadLog(pushEvent->GetFileStream(), pushEvent->GetOffsetToReadFrom() + formattedLog.NumBytesFormatted, MaxLogSizeToRead_));
        TFormattedLog logConvertion = OUTCOME_TRYX(PushLogFormatter_->Format(log, FormattedLogWithMetaSizeBytes(formattedLog.FormattedData), FormattedLogWithoutMetaSizeBytes(formattedLog.FormattedData)));

        //NumBytesFormatted equals 0 when 1) file was read to the end, 2) message packed to the max size
        if (!logConvertion.NumBytesFormatted) {
            break;
        }

        formattedLog += logConvertion;
    }

    return formattedLog;
}

} // namespace NInfra::NPodAgent
