#include "push_event_mock.h"
#include "simple_client.h"

#include <contrib/libs/grpc/include/grpcpp/create_channel.h>

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_impl.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/holders/session_holder_mock.h>
#include <infra/pod_agent/libs/pod_agent/logs_transmitter/statistics/logs_transmitter_statistics_impl.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_impl.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_mock.h>
#include <infra/pod_agent/libs/push_client/sessions/push_session_factory_impl.h>
#include <infra/pod_agent/libs/push_client/unified_agent_mock.h>
#include <infra/pod_agent/libs/push_client/utils/push_client_utils.h>

#include <util/generic/stack.h>
#include <util/system/condvar.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NSimpleClientTest {

static const ui64 maxLogSizePerOneIteration = 100;
static const ui64 maxFormattedLogSize = 500;
static const TString defaultLog = "defaultLog";
static const TVector<TString> formattedLog = {"formattedLog"};
static const TString defaultPath = "defaultPath";
static const ui64 numBytesFormatted = defaultLog.size();
static const TPushContainer defaultPushContainer = {{{"boxId"}, "workloadId"}, "boxId", "workloadId"};

struct TTestPushEvent: public TMockPushEvent {
    TTestPushEvent(TFileStreamRotatedPtr fileStream)
        : FileStream(fileStream)
    {}

    TFileStreamRotatedPtr GetFileStream() const override {
        return FileStream;
    }

    const TPortoContainerName& GetContainerName() const override {
        return ContainerName;
    }

    const TPushContainer& GetPushContainer() const override {
        return PushContainer;
    }

private:
    TFileStreamRotatedPtr FileStream;
    TPortoContainerName ContainerName = defaultPushContainer.Container;
    TPushContainer PushContainer = defaultPushContainer;
};

struct TTestSessionHolder: public TMockSessionHolder {
    TExpected<TPushSessionPtr, TPushClientError> GetSession(const TPushContainer& pushContainer) const override {
        ++GetSessionCalls;
        return Sessions.at(pushContainer);
    }

    void Update(const THashSet<TPushContainer>& pushContainers) override {
        for (const auto& pushContainer: pushContainers) {
            TPushSessionPtr session = new TPushSessionMock;
            Sessions[pushContainer] = session;
        }
    }

    const THashMap<TPushContainer, TPushSessionPtr>& GetSessions() const {
        return Sessions;
    }

    TExpected<TPushSessionPtr, TPushClientError> ActivateSession(const TPushContainer&) override {
        ++ActivateSessionCalls;
        return TExpected<TPushSessionPtr, TPushClientError>::DefaultSuccess();
    }

    TExpected<void, TPushClientError> DeactivateSession(const TPushContainer&) override {
        ++DeactivateSessionCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<void, TPushClientError> IncreaseLogSeqNumber(ui32, const TPushContainer&) override {
        ++IncreaseLogSeqNumberCalls;
        return TExpected<void, TPushClientError>::DefaultSuccess();
    }

    TExpected<i32, TPushClientError> IsActive(const TPushContainer&) const override {
        ++CheckActiveCalls;
        return TExpected<i32, TPushClientError>::DefaultSuccess();
    }

    mutable size_t GetSessionCalls = 0;
    size_t ActivateSessionCalls = 0;
    size_t DeactivateSessionCalls = 0;
    size_t IncreaseLogSeqNumberCalls = 0;
    mutable size_t CheckActiveCalls = 0;
    THashMap<TPushContainer, TPushSessionPtr> Sessions;
};

struct TTestFileStreamRotated: public TMockFileStreamRotated {

    TTestFileStreamRotated(const TString& log = defaultLog)
        : Log(log)
    {}

    TExpected<TString, TPushClientError> Read(size_t, size_t) override {
        ++ReadCalls;
        return Log;
    }

    TString GetFilePath() const override {
        return defaultPath;
    }

    TExpected<ui64, TPushClientError> GetFileSize() const override {
        return Log.size();
    }

    size_t ReadCalls = 0;
    TString Log;
};

struct TTestPushLogsFormatter: public IPushLogFormatter {

    TTestPushLogsFormatter(const TVector<TString>& formattedLogVec = formattedLog, ui64 numBytesFormattedUint = numBytesFormatted)
        : FormattedLog(formattedLogVec)
        , NumBytesFormatted(numBytesFormattedUint)
    {}
    TExpected<TFormattedLog, TPushClientError> Format(const TString& log, ui64, ui64) override {
        ++FormatCalls;
        if (log.empty()) {
            return TFormattedLog{};
        }
        return TFormattedLog{FormattedLog, NumBytesFormatted};
    }

    TVector<TString> FormattedLog;
    ui64 NumBytesFormatted;
    size_t FormatCalls = 0;
};

Y_UNIT_TEST_SUITE(SimplePushClientSuite) {

    Y_UNIT_TEST(SendFailsWhenNoSessionTest) {
        struct TTestSessionHolderFails: TTestSessionHolder{
            TExpected<i32, TPushClientError> IsActive(const TPushContainer& pushContainer) const override {
                ++CheckActiveCalls;
                return TPushClientError{EPushClientError::NoSessionForWorkload, TStringBuilder() << "no stdout push session for workload: " << pushContainer.Container};
            }
        };

        TSessionHolderPtr sessionHolder = new TTestSessionHolderFails();
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;

        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(1, ((TTestSessionHolder*)sessionHolder.Get())->CheckActiveCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->GetSessionCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->ActivateSessionCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->DeactivateSessionCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->IncreaseLogSeqNumberCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestPushLogsFormatter*)pushLogsFormatter.Get())->FormatCalls);

        UNIT_ASSERT(!(bool)sendResult);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::NoSessionForWorkload);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, TStringBuilder() << "no stdout push session for workload: " << defaultPushContainer.Container);
    }

    Y_UNIT_TEST(SendFailsWhenActivateSessionReturnError) {
        struct TTestSessionHolderFails: TTestSessionHolder{
            TExpected<TPushSessionPtr, TPushClientError> ActivateSession(const TPushContainer&) override {
                ++ActivateSessionCalls;
                return TPushClientError{EPushClientError::Unspecified, TStringBuilder() << "activation error"};
            }
        };

        TSessionHolderPtr sessionHolder = new TTestSessionHolderFails();
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});
        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(1, ((TTestSessionHolder*)sessionHolder.Get())->CheckActiveCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->GetSessionCalls);
        UNIT_ASSERT_EQUAL(1, ((TTestSessionHolder*)sessionHolder.Get())->ActivateSessionCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->DeactivateSessionCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestSessionHolder*)sessionHolder.Get())->IncreaseLogSeqNumberCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);
        UNIT_ASSERT_EQUAL(0, ((TTestPushLogsFormatter*)pushLogsFormatter.Get())->FormatCalls);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::Unspecified);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, "activation error");
    }

    Y_UNIT_TEST(SendFailsWhenOffsetGreaterThanFileSize) {
        struct TTestPushEventReturnsOffsetGreaterThanFileSize: public TTestPushEvent {
            TTestPushEventReturnsOffsetGreaterThanFileSize(TFileStreamRotatedPtr fileStream)
                 : TTestPushEvent(fileStream)
            {}

            ui64 GetOffsetToReadFrom() const override {
                return GetFileStream()->GetFileSize().Success() + 1;
            }
        };

        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);
        unifiedAgentModel.AddResponse(validSessionActivationResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        const auto sendResult = client->Send(new TTestPushEventReturnsOffsetGreaterThanFileSize(fileStream));

        UNIT_ASSERT_EQUAL(0, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::InvalidPushLogEvent);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, TStringBuilder() << "Event offset is greater than file size. Path: " << defaultPath);
    }

    Y_UNIT_TEST(SendNothingWhenEmptyDataAfterFormatting) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);

        TString log = "\n\n\n\n";
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter({}, log.size());

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        TPushLogResponse validLogSendResponse1;
        validLogSendResponse1.mutable_ack()->set_seq_no(1);

        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(validLogSendResponse1);

        const auto sendResult1 = client->Send(new TTestPushEvent(new TTestFileStreamRotated(log)));

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);

        UNIT_ASSERT((bool)sendResult1);

        UNIT_ASSERT_EQUAL(sendResult1.Success().NumBytesBeforeWrapping, log.size());
        UNIT_ASSERT_EQUAL(sendResult1.Success().NumBytesAfterWrapping, 0);

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 1);

        auto activationRequest1 = requests.at(0);
        UNIT_ASSERT(activationRequest1.has_initialize());
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(0).GetName(), TPushSessionImpl::DEPLOY_BOX_KEY);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(0).GetValue(), defaultPushContainer.BoxId);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(1).GetName(), TPushSessionImpl::DEPLOY_WORKLOAD_KEY);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(1).GetValue(), defaultPushContainer.WorkloadId);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(2).GetName(), TPushSessionImpl::DEPLOY_CONTAINER_ID_KEY);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(2).GetValue(), ToString(defaultPushContainer.Container));
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(3).GetName(), TPushSessionImpl::DEPLOY_LOGGER_NAME_KEY);
        UNIT_ASSERT_EQUAL(activationRequest1.initialize().meta(3).GetValue(), ToString(ELogType::Stdout));

        UNIT_ASSERT(activationRequest1.initialize().session_id().empty());
    }

    Y_UNIT_TEST(SendFailsWhenReadReturnError) {
        struct TTestFileStreamRotatedFails: TTestFileStreamRotated {
            TExpected<TString, TPushClientError> Read(size_t, size_t) override {
                ++ReadCalls;
                return TPushClientError{EPushClientError::FileReadingError, "reading error"};
            }
        };

        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);
        unifiedAgentModel.AddResponse(validSessionActivationResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotatedFails;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::FileReadingError);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, "reading error");
    }

    Y_UNIT_TEST(SendFailsWhenFormatConvertionReturnsError) {
        struct TTestPushLogsFormatterFails: TTestPushLogsFormatter {
            TExpected<TFormattedLog, TPushClientError> Format(const TString&, ui64, ui64) override {
                ++FormatCalls;
                return TPushClientError{EPushClientError::LogFormatConvertionError, "formatting error"};
            }
        };

        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatterFails();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);
        unifiedAgentModel.AddResponse(validSessionActivationResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::LogFormatConvertionError);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, "formatting error");
    }

    Y_UNIT_TEST(RepeatFormatWhenFormatReturnsNullBytesConvertedOk) {
        struct TTestPushLogsFormatterRepeatFormat: public TTestPushLogsFormatter {
            TExpected<TFormattedLog, TPushClientError> Format(const TString&, ui64, ui64) override {
                ++FormatCalls;
                if (FormatCalls == 1) {
                    return TFormattedLog{formattedLog, maxLogSizePerOneIteration};
                }
                return TFormattedLog{};
            }
        };

        struct TTestFileStreamRotatedRepeatFormat: public TTestFileStreamRotated {
            TExpected<ui64, TPushClientError> GetFileSize() const override {
                return maxFormattedLogSize;
            }
        };

        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatterRepeatFormat();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        TPushLogResponse validLogSendResponse;
        validLogSendResponse.mutable_ack()->set_seq_no(1);

        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(validLogSendResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotatedRepeatFormat;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));
        UNIT_ASSERT(sendResult);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);
        UNIT_ASSERT_EQUAL(2, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);
        UNIT_ASSERT_EQUAL(2, ((TTestPushLogsFormatter*)pushLogsFormatter.Get())->FormatCalls);
    }

    Y_UNIT_TEST(SendFailsWhenReadReturnEmpty) {
        struct TTestFileStreamRotatedEmpty: TTestFileStreamRotated {
            TExpected<TString, TPushClientError> Read(size_t, size_t) override {
                ++ReadCalls;
                return TString();
            }
        };

        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);
        unifiedAgentModel.AddResponse(validSessionActivationResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotatedEmpty;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);

        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::FileReadingError);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, TStringBuilder() << "Empty log was read. Path: " << defaultPath);
    }

    Y_UNIT_TEST(LogSendFailsWhenNotAckAnswer) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        TPushLogResponse invalidLogSendResponse;

        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(invalidLogSendResponse);

        TFileStreamRotatedPtr fileStream = new TTestFileStreamRotated;
        const auto sendResult = client->Send(new TTestPushEvent(fileStream));

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 1);

        UNIT_ASSERT(!(bool)sendResult);
        UNIT_ASSERT_EQUAL(sendResult.Error().Errno, EPushClientError::InvalidResponse);
        UNIT_ASSERT_STRING_CONTAINS(sendResult.Error().Message, "not ack response while log send");

        UNIT_ASSERT_EQUAL(1, ((TTestFileStreamRotated*)fileStream.Get())->ReadCalls);

        auto sessions = ((TSessionHolderImpl*)sessionHolder.Get())->GetSessions();

        UNIT_ASSERT_EQUAL(sessions.size(), 1);
        auto session = sessions.at(defaultPushContainer);
        UNIT_ASSERT_EQUAL(session->GetSeqNumber(), 1);
        UNIT_ASSERT_EQUAL(session->GetId(), "session_id");

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 2);

        auto activationRequest = requests.at(0);
        UNIT_ASSERT(activationRequest.has_initialize());
        UNIT_ASSERT(activationRequest.initialize().session_id().empty());

        auto logSendRequest = requests.at(1);
        UNIT_ASSERT(logSendRequest.has_data_batch());
        UNIT_ASSERT_EQUAL(logSendRequest.data_batch().seq_no(0), 1);
        UNIT_ASSERT_EQUAL(logSendRequest.data_batch().payload(0), formattedLog.at(0));
    }

    Y_UNIT_TEST(LogSendOkAfterRepeatSessionActivation) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();
        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter();

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        TPushLogResponse invalidLogSendResponse;
        TPushLogResponse validLogSendResponse;
        validLogSendResponse.mutable_ack()->set_seq_no(1);

        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(invalidLogSendResponse);
        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(validLogSendResponse);

        const auto sendResult1 = client->Send(new TTestPushEvent(new TTestFileStreamRotated));
        const auto sendResult2 = client->Send(new TTestPushEvent(new TTestFileStreamRotated));

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 1);

        UNIT_ASSERT(!(bool) sendResult1);
        UNIT_ASSERT((bool) sendResult2);

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 4);

        auto activationRequest1 = requests.at(0);
        UNIT_ASSERT(activationRequest1.has_initialize());
        UNIT_ASSERT(activationRequest1.initialize().session_id().empty());

        auto logSendRequest1 = requests.at(1);
        UNIT_ASSERT(logSendRequest1.has_data_batch());
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().seq_no(0), 1);
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().payload(0), formattedLog.at(0));

        auto activationRequest2 = requests.at(2);
        UNIT_ASSERT(activationRequest2.has_initialize());
        UNIT_ASSERT_EQUAL(activationRequest2.initialize().session_id(), "session_id");

        auto logSendRequest2 = requests.at(3);
        UNIT_ASSERT(logSendRequest2.has_data_batch());
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().seq_no(0), 1);
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().payload(0), formattedLog.at(0));
    }

    Y_UNIT_TEST(LogSendOkScenario) {
        NTesting::TPortHolder port = NTesting::GetFreePort();

        TUnifiedAgentModelMock unifiedAgentModel;
        TUnifiedAgentMock service(&unifiedAgentModel, port);
        service.Start();

        TGrpcChannelPtr grpcChannel = grpc::CreateChannel(TStringBuilder() << "localhost:" << port, grpc::InsecureChannelCredentials());
        TLogsTransmitterStatisticsPtr statistics = new TLogsTransmitterStatisticsImpl();

        TSessionHolderPtr sessionHolder = new TSessionHolderImpl("logName", ELogType::Stdout, new TPushSessionFactoryImpl(ELogType::Stdout, grpcChannel, ""), statistics);
        TVector<TString> formattedLog = {"formattedPart1", "formattedPart2"};
        TPushLogFormatterPtr pushLogsFormatter = new TTestPushLogsFormatter(formattedLog);

        TSimplePushClientPtr client = new TSimplePushClient(pushLogsFormatter, sessionHolder, maxLogSizePerOneIteration, statistics);

        sessionHolder->Update({defaultPushContainer});

        TPushLogResponse validSessionActivationResponse;
        validSessionActivationResponse.mutable_initialized()->set_session_id("session_id");
        validSessionActivationResponse.mutable_initialized()->set_last_seq_no(0);

        TPushLogResponse validLogSendResponse1;
        validLogSendResponse1.mutable_ack()->set_seq_no(1);

        TPushLogResponse validLogSendResponse2;
        validLogSendResponse2.mutable_ack()->set_seq_no(2);

        unifiedAgentModel.AddResponse(validSessionActivationResponse);
        unifiedAgentModel.AddResponse(validLogSendResponse1);
        unifiedAgentModel.AddResponse(validLogSendResponse2);

        const auto sendResult1 = client->Send(new TTestPushEvent(new TTestFileStreamRotated));
        const auto sendResult2 = client->Send(new TTestPushEvent(new TTestFileStreamRotated));

        UNIT_ASSERT_EQUAL(statistics->GetNumErrorsFromPushAgent(), 0);

        UNIT_ASSERT((bool)sendResult1);
        UNIT_ASSERT((bool)sendResult2);
        UNIT_ASSERT_EQUAL(sendResult1.Success().NumBytesBeforeWrapping, numBytesFormatted);

        UNIT_ASSERT_EQUAL(sendResult1.Success().NumBytesAfterWrapping, formattedLog.at(0).size() + formattedLog.at(1).size() + 2*NSessionMetaDescription::LOG_CHUNK_META_SIZE);

        auto requests = unifiedAgentModel.GetRequests();
        UNIT_ASSERT_EQUAL(requests.size(), 3);

        auto activationRequest1 = requests.at(0);
        UNIT_ASSERT(activationRequest1.has_initialize());
        UNIT_ASSERT(activationRequest1.initialize().session_id().empty());

        auto logSendRequest1 = requests.at(1);
        UNIT_ASSERT(logSendRequest1.has_data_batch());
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().seq_no(0), 1);
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().payload(0), formattedLog.at(0));
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().seq_no(1), 2);
        UNIT_ASSERT_EQUAL(logSendRequest1.data_batch().payload(1), formattedLog.at(1));

        auto logSendRequest2 = requests.at(2);
        UNIT_ASSERT(logSendRequest2.has_data_batch());
        UNIT_ASSERT_EQUAL(logSendRequest2.data_batch().seq_no(0), 3);
        UNIT_ASSERT_EQUAL(logSendRequest2.data_batch().payload(0), formattedLog.at(0));
        UNIT_ASSERT_EQUAL(logSendRequest2.data_batch().seq_no(1), 4);
        UNIT_ASSERT_EQUAL(logSendRequest2.data_batch().payload(1), formattedLog.at(1));
    }
}

} //namespace NInfra::NPodAgent::NSimpleClientTest
