#pragma once

#include <infra/pod_agent/libs/porto_client/porto_types.h>

#include <grpcpp/channel.h>

#include <logbroker/unified_agent/plugins/grpc_input/proto/unified_agent.grpc.pb.h>
#include <logbroker/unified_agent/plugins/grpc_input/proto/unified_agent.pb.h>

#include <util/digest/numeric.h>
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/string/builder.h>

namespace NInfra::NPodAgent {

using TPushLogRequest = NUnifiedAgentProto::Request;
using TPushLogResponse = NUnifiedAgentProto::Response;

using TGrpcPushStreamPtr = TAtomicSharedPtr<grpc::ClientReaderWriter<TPushLogRequest, TPushLogResponse> >;

using TGrpcPushContextPtr = TAtomicSharedPtr<grpc::ClientContext>;
using TGrpcChannelPtr = std::shared_ptr<grpc::Channel>;

namespace NSessionMetaDescription {
    static constexpr const ui32 TIMESTAMP_SIZE = 8;
    static constexpr const ui32 SEQ_NO_SIZE = 8;
    static constexpr const ui32 LOG_CHUNK_META_SIZE = TIMESTAMP_SIZE + SEQ_NO_SIZE;
}

enum class ELogType: ui8 {
    Stdout = 0 /* "stdout" */,
    Stderr = 1 /* "stderr" */
};

enum class EPushClientError: ui8 {
    Unspecified                   = 0  /* "unspecifed" */,
    GrpcStreamBroken              = 1  /* "grpc stream broken" */,
    InvalidResponse               = 2  /* "invalid grpc response" */,
    NoSessionForWorkload          = 3  /* "no session for workload" */,
    ExceptionWhileStreamOperation = 5  /* "exception while stream read or write" */,
    BadRequest                    = 6 /* "bad request to push agent" */,
    Unauthorized                  = 7 /* "unauthorized request" */,
    InternalError                 = 8 /* "internal error" */,
    Aborted                       = 9 /* "request aborted" */,
    Unavalailable                 = 10 /* "request unavailable" */,
    Overloaded                    = 11 /* "overloaded" */,
    SchemeError                   = 12 /* "scheme error" */,
    GenericError                  = 13 /* "generic error" */,
    Timeout                       = 14 /* "generic timeout" */,
    BadSession                    = 15 /* "bad session" */,
    PreconditionFailed            = 16 /* "precondition failed"*/,
    AlreadyExists                 = 17 /* "already exists" */,
    NotFound                      = 18 /* "not found" */,
    SessionExpired                = 19 /* "session expired" */,
    Cancelled                     = 20 /* "cancelled " */,
    LogFormatConvertionError      = 21 /* "log format convertion error " */,
    NoOffsetForWorkload           = 22 /* "no offset for workload" */,
    NoLogsFileStreamForWorkload   = 23 /* "no file stream for workload" */,
    FileReadingError              = 24 /* "reading file error" */,
    FileNotExists                 = 25 /* "file not exists" */,
    FileRotateError               = 26 /* "file rotate error" */,
    FileInodeChanged              = 27 /* "file inode changed" */,
    InvalidRotateParams           = 28 /* "invalid rotate params" */,
    FileStatError                 = 29 /* "file stat error" */,
    FileStreamCreationError       = 30 /* "file stream creation error" */,
    InvalidPushLogEvent           = 31 /* "invalid push log event" */,
    EndOfFileWarning              = 32 /* "end of file warning" */,
    SignalSendError               = 33 /* "metrics signal send error" */,
    DirectoryCreatingError        = 34 /* "directory creating error" */,
    SerializeError                = 35 /* "serialize error" */,
    DeserializeError              = 36 /* "deserialize error" */,
    NoData                        = 37 /* "no data error" */
};

struct TPushClientError {
    const EPushClientError Errno = EPushClientError::Unspecified;
    TString Message = "";
};

struct TPushContainer {
    TPortoContainerName Container = TPortoContainerName(TString(""));
    TString BoxId = TString("");
    TString WorkloadId = TString("");

    bool operator == (const TPushContainer& other) const {
        return this->Container == other.Container
           && this->BoxId == other.BoxId
           && this->WorkloadId == other.WorkloadId;
    }
};

struct TFormattedLog {
    TVector<TString> FormattedData;
    ui64 NumBytesFormatted = 0;

    TFormattedLog& operator += (const TFormattedLog& rhs) {
        for (const auto& sample: rhs.FormattedData) {
            FormattedData.push_back(sample);
        }

        NumBytesFormatted += rhs.NumBytesFormatted;
        return *this;
    }
};

struct TSendLogResult {
    ui64 NumBytesBeforeWrapping = 0;
    ui64 NumBytesAfterWrapping = 0;
};

}

template <>
struct THash<NInfra::NPodAgent::TPushContainer> {
    size_t operator()(const NInfra::NPodAgent::TPushContainer& pushContainer) const {
        size_t result = ComputeHash((TString)pushContainer.Container);
        result = CombineHashes(result, ComputeHash(pushContainer.WorkloadId));
        result = CombineHashes(result, ComputeHash(pushContainer.BoxId));
        return result;
    }
};
