#include "push_client_utils.h"

#include <infra/pod_agent/libs/pod_agent/logs_transmitter/file_stream/file_stream_rotated_mock.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestPushClientUtils {

static const ui64 defaultFileSize = 100;
static const ui64 defaultMaxSizetoRead = 50;
static const TString defaultReadedLog("log");

struct TTestFileStreamRotated: public TMockFileStreamRotated {
    TExpected<ui64, TPushClientError> GetFileSize() const override {
        return defaultFileSize;
    }

    TExpected<TString, TPushClientError> Read(ui64, ui64) override {
        return defaultReadedLog;
    }
};

Y_UNIT_TEST_SUITE(PushClientTypesUtilsSuite) {

    Y_UNIT_TEST(ReadLogReturnsEmptyWhenOffsetEqualsFileSize) {
        ui64 offsetEqualsFileSize = defaultFileSize;
        auto readResult = NPushClientUtils::ReadLog(new TTestFileStreamRotated, offsetEqualsFileSize, defaultMaxSizetoRead);

        UNIT_ASSERT(readResult);
        UNIT_ASSERT_STRING_CONTAINS("", readResult.Success());
    }

    Y_UNIT_TEST(ReadLogReturnsErrorWhenOffsetGreaterThanFileSize) {
        ui64 offsetGreaterThanFileSize = defaultFileSize + 1;
        auto readResult = NPushClientUtils::ReadLog(new TTestFileStreamRotated, offsetGreaterThanFileSize, defaultMaxSizetoRead);

        UNIT_ASSERT(!readResult);
        UNIT_ASSERT_EQUAL(EPushClientError::InvalidPushLogEvent, readResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("Event offset is greater than file size. Path: ", readResult.Error().Message);
    }

    Y_UNIT_TEST(ReadLogReturnsErrorWhenReadReturnsEmpty) {
        struct TTestFileStreamRotatedEmptyRead: public TTestFileStreamRotated {
            TExpected<TString, TPushClientError> Read(ui64, ui64) override {
                return TString();
            }
        };
        ui64 offsetLessThanFileSize = defaultFileSize - 1;
        auto readResult = NPushClientUtils::ReadLog(new TTestFileStreamRotatedEmptyRead, offsetLessThanFileSize, defaultMaxSizetoRead);

        UNIT_ASSERT(!readResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileReadingError, readResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("Empty log was read. Path: ", readResult.Error().Message);
    }

    Y_UNIT_TEST(ReadLogReturnsErrorWhenReadReturnsError) {
        struct TTestFileStreamRotatedEmptyRead: public TTestFileStreamRotated {
            TExpected<TString, TPushClientError> Read(ui64, ui64) override {
                return TPushClientError{EPushClientError::FileReadingError, "File reading error"};
            }
        };
        ui64 offsetLessThanFileSize = defaultFileSize - 1;
        auto readResult = NPushClientUtils::ReadLog(new TTestFileStreamRotatedEmptyRead, offsetLessThanFileSize, defaultMaxSizetoRead);

        UNIT_ASSERT(!readResult);
        UNIT_ASSERT_EQUAL(EPushClientError::FileReadingError, readResult.Error().Errno);
        UNIT_ASSERT_STRING_CONTAINS("File reading error", readResult.Error().Message);
    }

    Y_UNIT_TEST(ReadLogOk) {
        ui64 offsetLessThanFileSize = defaultFileSize - 1;
        auto readResult = NPushClientUtils::ReadLog(new TTestFileStreamRotated, offsetLessThanFileSize, defaultMaxSizetoRead);

        UNIT_ASSERT(readResult);
        UNIT_ASSERT_STRING_CONTAINS(defaultReadedLog, readResult.Success());
    }
}

}
