#include "service.h"

#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>
#include <infra/pod_agent/libs/router_api/router_api.h>

#include <library/cpp/build_info/build_info.h>
#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/svnversion/svnversion.h>
#include <library/cpp/watchdog/watchdog.h>

#include <util/folder/dirut.h>
#include <util/folder/filelist.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/vector.h>
#include <util/system/fstat.h>
#include <util/system/hostname.h>
#include <util/system/mlock.h>
#include <util/system/sysstat.h>

namespace NInfra::NPodAgent {

namespace {

const ui32 UNISTAT_INTERVALS_SIZE_LIMIT = 50;

const ui32 TRUNK_BRANCH_NUMBER = 100;
const ui32 UNKNOWN_BRANCH_NUMBER = 200;
const ui32 SPECIAL_BRANCH_NUMBER_BARRIER = 300;
const ui32 FIRST_REAL_BRANCH_NUMBER = 66000;
const ui32 MAJOR_BRANCH_MULTIPLIER = 1000;

TString GetVersion() {
    TStringBuilder builder;
    builder << GetProgramSvnVersion() << "\n";
    builder << GetBuildInfo() << "\n";
    return builder;
}

TMaybe<ui32> ParseReleaseMachineBranchNumber(const TString& branch) {
    try {
        // Try to parse "pod_agent/stable-<major>-<minor>"
        TVector<TString> parts = SplitString(branch, "/");

        if (parts.size() == 2u && parts[0] == "pod_agent") {
            TVector<TString> versionParts = SplitString(parts[1], "-");

            if (versionParts.size() == 3u && versionParts[0] == "stable") {
                ui32 major = FromString<ui32>(versionParts[1]);
                ui32 minor = FromString<ui32>(versionParts[2]);

                return MAJOR_BRANCH_MULTIPLIER * major + minor;
            }
        }

        return Nothing();
    } catch (...) {
        return Nothing();
    }
}

NUnistat::TIntervals GetBranchNumberIntervals() {
    NUnistat::TIntervals ret;

    // Start value
    ret.push_back(0);

    // Special branch numbers
    ret.push_back(TRUNK_BRANCH_NUMBER);
    ret.push_back(UNKNOWN_BRANCH_NUMBER);

    // Special branch numbers barrier
    ret.push_back(SPECIAL_BRANCH_NUMBER_BARRIER);

    // Legacy branch numbers is [SPECIAL_BRANCH_NUMBER_BARRIER; FIRST_REAL_BRANCH_NUMBER)
    ret.push_back(FIRST_REAL_BRANCH_NUMBER);
    while (ret.size() < UNISTAT_INTERVALS_SIZE_LIMIT) {
        double lastValue = ret.back();
        ret.push_back(lastValue + MAJOR_BRANCH_MULTIPLIER);
    }

    return ret;
}

ui32 GetBranchNumber() {
    const TString branch = GetBranch();

    if (const TMaybe<ui32> version = ParseReleaseMachineBranchNumber(branch); version.Defined()) {
        return *version;
    } else if (branch == "trunk") {
        return TRUNK_BRANCH_NUMBER;
    } else {
        return UNKNOWN_BRANCH_NUMBER;
    }
}

} // namespace

THolder<TService> TService::MakeService(
    const TString& selfBinaryFilePath
    , const TConfig& config
    , TLogger& stderrLogger
) {
    TString dom0PodRoot;
    TMap<TString, TString> virtualDisksToPlace;
    TMap<TString, TString> placesToDownloadVolumePath;
    InitPortoAndGetPlaces(
        selfBinaryFilePath
        , config
        , dom0PodRoot
        , virtualDisksToPlace
        , placesToDownloadVolumePath
        , stderrLogger
    );
    return THolder<TService>(
        new TService(
            config
            , dom0PodRoot
            , virtualDisksToPlace
            , placesToDownloadVolumePath
        )
    );
}

TService::TService(
    const TConfig& config
    , const TString& dom0PodRoot
    , const TMap<TString, TString>& virtualDisksToPlace
    , const TMap<TString, TString>& placesToDownloadVolumePath
)
    : Logger_(config.GetLogger())
    , TreeTraceLogger_(config.GetTreeTraceLogger())
    , LogsTransmitterJobWorkerEventsLogger_(config.GetLogsTransmitter().GetJobWorkerEventsLogger())
    , Config_(config)
    , CoreService_(
        Logger_
        , TreeTraceLogger_
        , LogsTransmitterJobWorkerEventsLogger_
        , config
        , dom0PodRoot
        , virtualDisksToPlace
        , placesToDownloadVolumePath
    )
    , LastSuccessfulPodAgentRequestTime_(TInstant::Now())
{
    InitSignals();
    if (config.GetMemoryLock().GetLockSelfMemory()) {
        LockSelfMemory();
    }
}

void TService::Start() {
    auto logFrame = Logger_.SpawnFrame();

    CoreService_.Start();

    if (Config_.HasGrpcService()) {
        GrpcService_ = MakeHolder<TGrpcService>(Config_.GetGrpcService(), *this);
        GrpcService_->Start(logFrame);
    }

    if (Config_.HasHttpPublicService()) {
        HttpService_ = MakeHolder<THttpService>(Config_.GetHttpPublicService(), CreateRouter(*this, false /*withPrivateAPI*/));
        HttpService_->Start(logFrame);
    }

    StartTime_ = TInstant::Now();
    TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, SIGNAL_ALIVE_DAEMONS_NAME, 1);
}

void TService::Shutdown(TRequestPtr<TReqShutdown>, TReplyPtr<TRspShutdown>) {
    IncTotalCounter("Shutdown");
    static THolder<IWatchDog> abortWatchDog = THolder<IWatchDog>(CreateAbortByTimeoutWatchDog(CreateAbortByTimeoutWatchDogOptions(TDuration::Minutes(10)), "Ooops!"));

    if (GrpcService_) {
        GrpcService_->Shutdown();
    }

    if (HttpService_) {
        HttpService_->ShutDown();
    }

    CoreService_.Shutdown();
    IncProcessedCounter("Shutdown");
}

void TService::Config(TRequestPtr<TReqConfig>, TReplyPtr<TRspConfig> reply) {
    IncTotalCounter("Config");

    TRspConfig rsp;
    *rsp.MutableData() = NProtobufJson::Proto2Json<TConfig>(Config_);

    reply->Set(rsp);
    IncProcessedCounter("Config");
}

void TService::Version(TRequestPtr<TReqVersion>, TReplyPtr<TRspVersion> reply) {
    IncTotalCounter("Version");
    TRspVersion version;
    *version.MutableData() = GetVersion();

    reply->Set(version);
    IncProcessedCounter("Version");
}

void TService::Wait() {
    auto logFrame = Logger_.SpawnFrame();

    if (GrpcService_) {
        GrpcService_->Wait(logFrame);
    }

    if (HttpService_) {
        HttpService_->Wait(logFrame);
    }

    CoreService_.Wait();
}

void TService::CommonSensors(TMultiUnistat::ESignalNamespace signalNamespace, TRequestPtr<TReqSensors> request, TReplyPtr<TRspSensors> reply) {
    auto signalPriority = FromStringWithDefault<TMultiUnistat::ESignalPriority>(
        request->Get().GetPriorityLevel()
        , TMultiUnistat::ESignalPriority::USER_INFO
    );

    CoreService_.UpdateConditionSensors();

    TRspSensors result;
    *result.MutableData() = TMultiUnistat::Instance().CreateJsonDump(signalNamespace, signalPriority, true);

    reply->Set(result);
}

void TService::Sensors(TRequestPtr<TReqSensors> request, TReplyPtr<TRspSensors> reply) {
    IncTotalCounter("Sensors");

    TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, SIGNAL_UPTIME_NAME, (TInstant::Now() - StartTime_).MicroSeconds() / 1000.);
    TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, SIGNAL_LAST_SUCCESSFUL_POD_AGENT_REQUEST_LAG_NAME, (TInstant::Now() - LastSuccessfulPodAgentRequestTime_).MicroSeconds() / 1000.);
    CommonSensors(TMultiUnistat::ESignalNamespace::INFRA, request, reply);

    IncProcessedCounter("Sensors");
}

void TService::UserSensors(TRequestPtr<TReqSensors> request, TReplyPtr<TRspSensors> reply) {
    IncTotalCounter("UserSensors");

    CommonSensors(TMultiUnistat::ESignalNamespace::USER, request, reply);

    IncProcessedCounter("UserSensors");
}

void TService::Ping(TRequestPtr<TReqPing>, TReplyPtr<TRspPing> reply) {
    IncTotalCounter("Ping");
    TRspPing rsp;
    *rsp.MutableData() = "pong";

    reply->Set(rsp);
    IncProcessedCounter("Ping");
}

void TService::SetLogLevel(TRequestPtr<TReqSetLogLevel> request, TReplyPtr<TRspSetLogLevel>) {
    IncTotalCounter("SetLogLevel");
    Logger_.SetLevel(FromString<ELogPriority>(request->Get().GetLevel()));
    IncProcessedCounter("SetLogLevel");
}

void TService::ReopenLog(TRequestPtr<TReqReopenLog>, TReplyPtr<TRspReopenLog>) {
    IncTotalCounter("ReopenLog");
    Logger_.ReopenLog();
    LogsTransmitterJobWorkerEventsLogger_.ReopenLog();
    IncProcessedCounter("ReopenLog");
}

void TService::UpdatePodAgentRequest(TRequestPtr<API::TPodAgentRequest> request, TReplyPtr<API::TPodAgentStatus> reply) {
    IncTotalCounter("UpdatePodAgentRequest");
    auto& spec = request->Get();
    API::TPodAgentStatus rsp = CoreService_.UpdatePodAgentRequest(spec);
    LastSuccessfulPodAgentRequestTime_ = TInstant::Now();
    reply->Set(rsp);
    IncProcessedCounter("UpdatePodAgentRequest");
}

void TService::GetPodAgentStatus(TRequestPtr<TReqGetPodAgentStatus> /* request */, TReplyPtr<API::TPodAgentStatus> reply) {
    IncTotalCounter("GetPodAgentStatus");

    API::TPodAgentStatus rsp = CoreService_.GetPodAgentStatus();
    reply->Set(rsp);

    IncProcessedCounter("GetPodAgentStatus");
}

void TService::PodAttributesJson(TRequestPtr<TReqPodAttributesJson> /* request */, TReplyPtr<TRspPodAttributesJson> reply) {
    IncTotalCounter("PodAttributesJson");
    TRspPodAttributesJson rsp;
    *rsp.MutableData() = CoreService_.PodAttributesJson();
    reply->Set(rsp);
    IncProcessedCounter("PodAttributesJson");
}

void TService::PodStatusJson(TRequestPtr<TReqPodStatusJson> /* request */, TReplyPtr<TRspPodStatusJson> reply) {
    IncTotalCounter("PodStatusJson");
    TRspPodStatusJson rsp;
    *rsp.MutableData() = CoreService_.PodStatusJson();
    reply->Set(rsp);
    IncProcessedCounter("PodStatusJson");
}

void TService::IncCounter(const TString& name) {
    if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, name, 1)) {
        auto logFrame = Logger_.SpawnFrame();
        logFrame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TServiceSignalError(name, "PushSingalUnsafe(" + name + ", 1) returned false."));
    }
}

void TService::IncTotalCounter(const TString& name) {
    IncCounter(COUNTER_PREFIX + name + COUNTER_TOTAL_REQUESTS_SUFFIX);
}

void TService::IncProcessedCounter(const TString& name) {
    IncCounter(COUNTER_PREFIX + name + COUNTER_PROCESSED_REQUESTS_SUFFIX);
}

void TService::InitSignals() {
    // If pod_agent is started in a separate thread and restarted many times, signals between restarts are not reseted
    // so we have to reset them
    TMultiUnistat::Instance().ResetAllSignals();

    InitVersionSignals();
    InitStatusSignals();
}

void TService::InitStatusSignals() {
    const TVector<TString> signalNames = {
        "Config"
        , "GetPodAgentStatus"
        , "Ping"
        , "PodAttributesJson"
        , "PodStatusJson"
        , "ReopenLog"
        , "Sensors"
        , "SetLogLevel"
        , "Shutdown"
        , "UpdatePodAgentRequest"
        , "UserSensors"
        , "Version"
    };

    for (const auto& name : signalNames) {
        for (auto suffix : TVector<TString>{COUNTER_TOTAL_REQUESTS_SUFFIX, COUNTER_PROCESSED_REQUESTS_SUFFIX}) {
            TString curName = COUNTER_PREFIX + name + suffix;

            TMultiUnistat::Instance().DrillFloatHole(
                TMultiUnistat::ESignalNamespace::INFRA
                , curName
                , "deee"
                , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
                , NUnistat::TStartValue(0)
                , EAggregationType::Sum
            );
        }
    }

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , SIGNAL_ALIVE_DAEMONS_NAME
        , "aeev"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , SIGNAL_UPTIME_NAME
        , "ahhh"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , SIGNAL_MEMORY_LOCK_FAIL_NAME
        , "aeev"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , SIGNAL_LAST_SUCCESSFUL_POD_AGENT_REQUEST_LAG_NAME
        , "ahhh"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
}

void TService::InitVersionSignals() {
    {
        TMultiUnistat::Instance().DrillFloatHole(
            TMultiUnistat::ESignalNamespace::INFRA
            , SIGNAL_SVN_REVISION_NAME
            , "ahhh"
            , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
            , NUnistat::TStartValue(0)
            , EAggregationType::LastValue
        );
        TMultiUnistat::Instance().PushSignalUnsafe(
            TMultiUnistat::ESignalNamespace::INFRA
            , SIGNAL_SVN_REVISION_NAME
            , GetProgramSvnRevision()
        );
    }

    {
        TMultiUnistat::Instance().DrillHistogramHole(
            TMultiUnistat::ESignalNamespace::INFRA
            , SIGNAL_BRANCH_NAME
            , "ahhh"
            , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
            , GetBranchNumberIntervals()
            , EAggregationType::HostHistogram
        );
        TMultiUnistat::Instance().PushSignalUnsafe(
            TMultiUnistat::ESignalNamespace::INFRA
            , SIGNAL_BRANCH_NAME
            , GetBranchNumber()
        );
    }
}

void TService::LockSelfMemory() {
    try {
        LockAllMemory(ELockAllMemoryFlag::LockCurrentMemory);
    } catch (const yexception& e) {
        NLogEvent::TServiceMemoryLockError ev;
        ev.SetError("Failed to lock memory: " + CurrentExceptionMessage());
        Logger_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, ev);
        TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, SIGNAL_MEMORY_LOCK_FAIL_NAME, 1);
    }
}

void TService::InitPortoAndGetPlaces(
    const TString& selfBinaryFilePath
    , const TConfig& config
    , TString& dom0PodRoot
    , TMap<TString, TString>& virtualDisksToPlace
    , TMap<TString, TString>& placesToDownloadVolumePath
    , TLogger& stderrLogger
) {
    TPortoConnectionPoolPtr pool = new TPortoConnectionPool(8);
    TPortoClientPtr porto = new TSimplePortoClient(stderrLogger.SpawnFrame(), pool, PORTO_CLIENT_DEFAULT_TIMEOUT);

    {
        // Dom0 pod root
        // For CI tests. When we is inside sandbox porto isolation, root path will be available in parent container.
        TPortoContainerName containerWithRoot = NPortoTestLib::IsInsideSandboxPortoIsolation()
            ? TPortoContainerName::NoEscape("..")
            : TPortoContainerName::NoEscape(".");
        dom0PodRoot = porto->GetProperty(containerWithRoot, EPortoContainerProperty::Root).Success();
    }

    if (config.GetPodAgentMode() == EPodAgentMode::BOX_MODE) {
        NFs::MakeDirectoryRecursive(config.GetCache().GetVolumePath());
        NFs::MakeDirectoryRecursive(config.GetPublicVolume().GetVolumePath());
        return;
    }

    // Full mode
    {
        // Cache volume
        GetOrCreateVolumeAndStorage(config.GetCache().GetVolumePath(), config.GetCache().GetStorage(), "", porto, CACHE_VOLUME_DESCRIPTION);
    }

    {
        // Public volume
        GetOrCreateVolumeAndStorage(config.GetPublicVolume().GetVolumePath(), config.GetPublicVolume().GetStorage(), "", porto, PUBLIC_VOLUME_DESCRIPTION);
        if (!config.GetPublicVolume().GetPodAgentBinaryFileName().empty()) {
            const TString podAgentPublicVolumePath = config.GetPublicVolume().GetVolumePath() + "/" + config.GetPublicVolume().GetPodAgentBinaryFileName();
            NFs::Copy(selfBinaryFilePath, podAgentPublicVolumePath);
            Chmod(podAgentPublicVolumePath.c_str(), MODE0755);
        }
    }

    {
        // Virtual disks
        GetAndCreateVirtualDisksPlacesAndPaths(
            config.GetVirtualDisks()
            , config.GetResources().GetDownloadVolumePath()
            , virtualDisksToPlace
            , placesToDownloadVolumePath
        );

        for (const auto& it : placesToDownloadVolumePath) {
            GetOrCreateVolumeAndStorage(it.second, config.GetResources().GetDownloadStoragePrefix() + "_" + MD5::Calc(it.first), it.first, porto, RESOURCES_VOLUME_DESCRIPTION);
        }
    }
}

void TService::GetOrCreateVolumeAndStorage(
    const TString& volumePath
    , const TString& volumeStorage
    , const TString& volumePlace
    , TPortoClientPtr porto
    , const TString& volumeDescription
) {
    NFs::MakeDirectoryRecursive(volumePath);
    const TString volumePathFull = RealPath(volumePath);

    auto result = porto->ListVolumes(volumePathFull);
    if (!result && result.Error().Code != EPortoError::VolumeNotFound) {
        result.Success();
    }
    bool needCreate = true;
    if (result) {
        Y_ENSURE(result.Success().size() == 1, TStringBuilder() << "Only one " << volumeDescription << " expected for path: " << volumePathFull);
        auto& volume = result.Success().front();
        Y_ENSURE(volume.path() == volumePathFull, "Unexpected path for " << volumeDescription << ": " << volume.path());
        needCreate = false;
        bool needRemove = false;
        TString checkValue = volume.storage();
        if (checkValue != volumeStorage) {
            needRemove = true;
        }
        TString portoNamespaceRootContainer = porto->GetProperty(TPortoContainerName::NoEscape("."), EPortoContainerProperty::AbsoluteName).Success();
        bool foundContainer = false;
        for (auto& link: volume.links()) {
            // "." is alias to namespace root container, but ListVolumes returns absulute_name's
            // so we can't use "." here
            foundContainer |= (link.container() == portoNamespaceRootContainer);
        }
        needRemove |= !foundContainer;
        if (needRemove) {
            porto->UnlinkVolume(volumePathFull, TPortoContainerName::NoEscape("***"), "***", true).Success();
            needCreate = true;
        }
    }
    if (needCreate) {
        porto->CreateVolume(volumePathFull, volumeStorage, volumePlace, {}, 0, "", EPortoVolumeBackend::Auto, TPortoContainerName::NoEscape("."), {}, false).Success();
    }
}

void TService::CreatePortoPlace(const TString& path) {
    const TVector<TString> dirs = {
        path + "/porto_volumes/"
        , path + "/porto_storage/"
        , path + "/porto_layers/"
    };

    for (const TString& dir : dirs) {
        if (!NFs::Exists(dir)) {
            NFs::MakeDirectoryRecursive(dir);
        }
    }
}

void TService::GetAndCreateVirtualDisksPlacesAndPaths(
    const TVirtualDisksConfig& virtualDisksConfig
    , const TString& downloadVolumePath
    , TMap<TString, TString>& virtualDisksToPlace
    , TMap<TString, TString>& placesToDownloadVolumePath
) {
    if (!virtualDisksConfig.GetEnable()) {
        // TODO(chegoryu) DEPLOY-562 enable by default in config

        // When there are no virtual disks, we use default porto place
        // and we assume that it corresponds to the disk with empty id
        virtualDisksToPlace[""] = "";
        placesToDownloadVolumePath[""] = TStringBuilder() << downloadVolumePath << RESOURCES_VOLUME_DEFAULT_PATH;
        return;
    }

    TString virtualDisksRootPath = virtualDisksConfig.GetRootPath();
    if (!virtualDisksRootPath.StartsWith('/')) {
        virtualDisksRootPath.prepend(NFs::CurrentWorkingDirectory() + '/');
    }
    if (!virtualDisksRootPath.EndsWith('/')) {
        virtualDisksRootPath.push_back('/');
    }

    Y_ENSURE(NFs::Exists(virtualDisksRootPath),  "Virtual disks root path does not exist: '" << virtualDisksRootPath << "'");

    TDirsList dirList;
    dirList.Fill(virtualDisksRootPath);
    for (const char* curDirName = dirList.Next(); curDirName; curDirName = dirList.Next()) {
        // We need to use syntax:
        // portoctl vcreate /db/iss3 place=///vdisk_storage/NONROOT
        // "///" is porto magic for chrooted path
        const TString curPlacePath = TStringBuilder() << "//" << virtualDisksRootPath << curDirName;
        const TString curDownloadVolumePath = TStringBuilder() << downloadVolumePath << RESOURCES_VOLUME_SPECIFIC_PATH << RESOURCES_VOLUME_VIRTUAL_DISK_PREFIX << curDirName;

        Y_ENSURE(TFileStat(curPlacePath).IsDir(), "Virtual disk path '" << curPlacePath << "' is not a directory");

        CreatePortoPlace(curPlacePath);
        virtualDisksToPlace[curDirName] = curPlacePath;
        placesToDownloadVolumePath[curPlacePath] = curDownloadVolumePath;
    }

    Y_ENSURE(virtualDisksToPlace.size() > 0, "No virtual disks provided in '" << virtualDisksRootPath << "'");
}

} // namespace NInfra::NPodAgent
