#include "system_logs_statistics_printer_impl.h"

#include <infra/libs/logger/log_frame.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_types.h>

#include <util/generic/ptr.h>

namespace NInfra::NPodAgent {

bool TSystemLogsStatisticsPrinter::IsInitialized_ = false;
TRWMutex TSystemLogsStatisticsPrinter::SignalMutex_;

TSystemLogsStatisticsPrinter::TSystemLogsStatisticsPrinter(
    TSystemLogsStatisticsPtr statistics
    , TSystemLogsSessionCollectionPtr boxSystemLogSessionCollection
    , TSystemLogsSessionCollectionPtr workloadSystemLogSessionCollection
    , TLogFramePtr logFrame
    , ui32 printPeriodSec
)
    : LastPrintTime_(Now().Seconds())
    , Statistics_(statistics)
    , BoxSystemLogsSessionCollection_(boxSystemLogSessionCollection)
    , WorkloadSystemLogsSessionCollection_(workloadSystemLogSessionCollection)
    , LogFrame_(logFrame)
    , PrintPeriodSec_(printPeriodSec) {

    InitSignals();
}

void TSystemLogsStatisticsPrinter::Print() {
    ui64 now = Now().Seconds();

    if (now >= (LastPrintTime_ + PrintPeriodSec_)) {

        if (Statistics_->GetNumBytesSent())
            SendSignal(COUNTER_TRANSFERED_LOGS_BYTES, Statistics_->GetNumBytesSent());

        if (Statistics_->GetNumBytesDropped())
            SendSignal(COUNTER_DROPPED_LOGS_BYTES, Statistics_->GetNumBytesDropped());

        SendSignal(COUNTER_NUM_OF_SYSTEM_LOGS_ENABLED_BOX_SESSIONS,BoxSystemLogsSessionCollection_->GetEnabledSessionsCount());
        SendSignal(COUNTER_NUM_OF_SYSTEM_LOGS_DISABLED_BOX_SESSIONS,BoxSystemLogsSessionCollection_->GetDisabledSessionsCount());
        SendSignal(COUNTER_NUM_OF_SYSTEM_LOGS_ENABLED_WORKLOAD_SESSIONS,WorkloadSystemLogsSessionCollection_->GetEnabledSessionsCount());
        SendSignal(COUNTER_NUM_OF_SYSTEM_LOGS_DISABLED_WORKLOAD_SESSIONS,WorkloadSystemLogsSessionCollection_->GetDisabledSessionsCount());

        Statistics_->Reset();

        LastPrintTime_ = now;
    }
}

void TSystemLogsStatisticsPrinter::InitSignals() {
    {
        TReadGuard guard(SignalMutex_);
        if (IsInitialized_) {
            return;
        }
    }

    TWriteGuard guard(SignalMutex_);
    if (IsInitialized_) {
        return;
    }

    IsInitialized_ = true;

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_TRANSFERED_LOGS_BYTES
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_DROPPED_LOGS_BYTES
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_SYSTEM_LOGS_ENABLED_BOX_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_SYSTEM_LOGS_DISABLED_BOX_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_SYSTEM_LOGS_ENABLED_WORKLOAD_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NUM_OF_SYSTEM_LOGS_DISABLED_WORKLOAD_SESSIONS
        , "aeee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::LastValue
    );
}

void TSystemLogsStatisticsPrinter::SendSignal(const TString& signalType, i64 value) {
    if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, signalType, value)) {
        LogFrame_->LogEvent(ELogPriority::TLOG_ERR, ConstructExceptionEvent(
            TSystemLogsSenderError{
                ESystemLogsSenderError::PushSignalError
                , TStringBuilder() << "PushSingalUnsafe(" << signalType << ", " << ToString(value) << ") returned false"
            }
        ));
    }
}

} //namespace NInfra::NPodAgent
