#include "system_logs_sender_impl.h"

namespace NInfra::NPodAgent {
    
void TSystemLogsSender::Add(const TString& objectId, TSystemLogsSessionPtr session) {
    if (AtomicGet(MarkEnabled_)) {
        session->MarkEnabled(true);
    }

    ObjectIdToSession_->Add(objectId, session);
}

void TSystemLogsSender::Remove(const TString& objectId) {
    ObjectIdToSession_->Remove(objectId);
}

void TSystemLogsSender::RemoveAll() {
    ObjectIdToSession_->RemoveAll();
}

bool TSystemLogsSender::Has(const TString& objectId) const {
    return (bool)(ObjectIdToSession_->Get(objectId));
}

void TSystemLogsSender::MarkEnabled(bool enable) { 
    if (enable == AtomicGet(MarkEnabled_)) {
        return;
    } 
    
    TVector<TSystemLogsSessionPtr> sessions = ObjectIdToSession_->GetAll();

    for (const auto& session: sessions) {
        session->MarkEnabled(enable);
    }

    AtomicSet(MarkEnabled_, enable);
}

bool TSystemLogsSender::GetMarkEnabled() const {
    return AtomicGet(MarkEnabled_);
}

TFuture<TExpected<void, TSystemLogsSenderError>> TSystemLogsSender::SendMessage(const TString& objectId, const TString& message) {
    return NThreading::Async(
        [=] () {
            return SendMessageImpl(objectId, message);
        }
        , *ThreadPool_
    );    
}

TExpected<void, TSystemLogsSenderError> TSystemLogsSender::SendMessageImpl(const TString& objectId, const TString& message) {
    if (SystemLogsFilter_->DropMessageFrom(objectId)) {
        return TExpected<void, TSystemLogsSenderError>::DefaultSuccess();
    }

    TSystemLogsSessionPtr session = OUTCOME_TRYX(ObjectIdToSession_->Get(objectId));
    session->SendMessage(message);
    Statistics_->AddNumBytesSent(session->GetBytesSent());
    Statistics_->AddNumBytesDropped(session->GetBytesDropped());

    return TExpected<void, TSystemLogsSenderError>::DefaultSuccess();
}

} // namespace NInfra::NPodAgent
