import argparse
import json
import random

DEFAULT_NODES = {
    "Switch",
    "StaticSwitch",
    "Case",
    "Succeeder",
    "Failer",
    "Error",

    "MemSelector",
    "MemSequenceGenerator",
    "MemSequence",

    "Inverter",

    "TimedCache",
}

MAIN_TREE_PATHS = [
    "../../libs/behaviour/trees/base/tree_check_and_create_container.json",
    "../../libs/behaviour/trees/base/tree_check_and_create_resource_gang_meta_container.json",
    "../../libs/behaviour/trees/base/tree_create_volume.json",
    "../../libs/behaviour/trees/base/tree_destroy_container.json",
    "../../libs/behaviour/trees/base/tree_init.json",
    "../../libs/behaviour/trees/base/tree_init_container_ready.json",
    "../../libs/behaviour/trees/base/tree_layers_ready.json",
    "../../libs/behaviour/trees/base/tree_volumes_ready.json",

    "../../libs/behaviour/trees/box/box_tree_template.json",

    "../../libs/behaviour/trees/layer/layer_tree_template.json",
    "../../libs/behaviour/trees/layer/tree_remove_layer_files.json",
    "../../libs/behaviour/trees/layer/tree_verify_layer.json",

    "../../libs/behaviour/trees/static_resource/static_resource_tree_template.json",
    "../../libs/behaviour/trees/static_resource/tree_verify_static_resource.json",

    "../../libs/behaviour/trees/volume/volume_tree_template.json",

    "../../libs/behaviour/trees/workload/base/tree_workload_check_box.json",
    "../../libs/behaviour/trees/workload/base/tree_workload_clear_hook.json",
    "../../libs/behaviour/trees/workload/base/workload_tree_template.json",

    "../../libs/behaviour/trees/workload/destroy/tree_container_destroy_hook.json",
    "../../libs/behaviour/trees/workload/destroy/tree_http_destroy_hook.json",
    "../../libs/behaviour/trees/workload/destroy/tree_workload_destroy.json",

    "../../libs/behaviour/trees/workload/start/tree_workload_start.json",

    "../../libs/behaviour/trees/workload/status/tree_container_status_hook.json",
    "../../libs/behaviour/trees/workload/status/tree_network_status_hook.json",

    "../../libs/behaviour/trees/workload/stop/tree_container_stop_hook.json",
    "../../libs/behaviour/trees/workload/stop/tree_http_stop_hook.json",
    "../../libs/behaviour/trees/workload/stop/tree_unix_signal_stop_hook.json",
    "../../libs/behaviour/trees/workload/stop/tree_workload_stop.json",
]

TEST_TREE_PATHS = [
    "../../libs/behaviour/loaders/ut/tree.json",
    "../../libs/behaviour/loaders/ut/tree_check_create.json",
    "../../libs/behaviour/loaders/ut/tree_verify.json",
]

ALL_TREE_PATHS = MAIN_TREE_PATHS + TEST_TREE_PATHS


def remove_unused_nodes(file_data):
    current_json = json.loads(file_data)

    exist_nodes = DEFAULT_NODES.copy()

    nodes = current_json["nodes"]
    for key in nodes.keys():
        exist_nodes.add(nodes[key]["name"])

    file_lines = file_data.split("\n")
    new_file_lines = []

    parse_custom_nodes = False
    current_custom_node = []
    custom_nodes = set()
    for line in file_lines:
        if line == " " * len(line):
            continue

        if line.find('  "custom_nodes": [') != -1:
            new_file_lines.append(line)
            parse_custom_nodes = True
        elif parse_custom_nodes and line[:len("  ]")] == "  ]":
            new_nodes = exist_nodes - custom_nodes - DEFAULT_NODES

            if new_nodes:
                if new_file_lines[-1][:len("    }")] == "    }":
                    new_file_lines[-1] = "    },"

                for name in new_nodes:
                    new_file_lines.append('    {')
                    new_file_lines.append('      "version": "0.3.0",')
                    new_file_lines.append('      "scope": "node",')
                    new_file_lines.append('      "name": "{}",'.format(name))
                    new_file_lines.append('      "category": "action",')
                    new_file_lines.append('      "title": "",')
                    new_file_lines.append('      "description": null,')
                    new_file_lines.append('      "properties": {}')
                    new_file_lines.append('    },')

            if new_file_lines[-1][:len("    },")] == "    },":
                new_file_lines[-1] = "    }"

            new_file_lines.append(line)
            parse_custom_nodes = False
        elif not parse_custom_nodes:
            new_file_lines.append(line)
        else:
            current_custom_node.append(line)

            if line[:len("    }")] == "    }":
                current_node = json.loads("\n".join(current_custom_node[:len(current_custom_node) - 1] + ["  }"]))

                if current_node["name"] in exist_nodes:
                    custom_nodes.add(current_node["name"])
                    if current_node["title"] != "":
                        for i in range(len(current_custom_node)):
                            if current_custom_node[i].find('      "title": ') != -1:
                                current_custom_node[i] = '      "title": "",'
                                break

                    new_file_lines.extend(current_custom_node)

                current_custom_node = []

    new_file_data = "\n".join(new_file_lines) + "\n"
    return new_file_data


def regen_ids(file_data):
    current_json = json.loads(file_data)

    exist_ids = set()
    exist_ids.add(current_json["id"])

    nodes = current_json["nodes"]
    for key in nodes.keys():
        assert key == nodes[key]["id"]
        exist_ids.add(nodes[key]["id"])

    new_ids = set()
    new_ids_by_old_id = dict()
    for id in exist_ids:
        #  same as generation in BT editor
        new_id = "{0:016}".format(random.randint(0, 10**16 - 1))
        while new_id in new_ids:
            new_id = "{0:016}".format(random.randint(0, 10**16 - 1))

        new_ids.add(new_id)
        new_ids_by_old_id[id] = new_id

    new_file_data = file_data
    for id in exist_ids:
        new_file_data = new_file_data.replace(id, new_ids_by_old_id[id])

    return new_file_data


def remove_useless_node_titles(file_data):
    for default_node_id in DEFAULT_NODES:
        file_data = file_data.replace("\"title\": \"{}\"".format(default_node_id), "\"title\": \"\"")

    return file_data


def main(arguments):
    if (arguments.file_name or arguments.override) and arguments.patch_all:
        raise Exception("You can't specify file name or override with patch_all")

    if arguments.pre_commit_patch:
        if arguments.file_name \
           or arguments.override \
           or arguments.patch_all \
           or arguments.remove_unused_nodes \
           or arguments.regen_ids \
           or arguments.remove_useless_node_titles:
            raise Exception("You can't specify anything else with pre_commit_patch")

        arguments.patch_all = True
        arguments.remove_unused_nodes = True
        arguments.remove_useless_node_titles = True

    files_to_patch = []
    override = False

    if arguments.patch_all:
        files_to_patch = ALL_TREE_PATHS
        override = True
    elif arguments.file_name:
        files_to_patch = [arguments.file_name]
        override = arguments.override
    else:
        raise Exception("You must specify file name or patch_all")

    if not arguments.remove_unused_nodes and not arguments.regen_ids and not arguments.remove_useless_node_titles:
        raise Exception("You must specify at least one option for patch")

    for file in files_to_patch:
        try:
            with open(file, 'r') as f:
                file_data = f.read()

            if arguments.remove_unused_nodes:
                file_data = remove_unused_nodes(file_data)
            if arguments.regen_ids:
                file_data = regen_ids(file_data)
            if arguments.remove_useless_node_titles:
                file_data = remove_useless_node_titles(file_data)

            if override:
                with open(file, 'w') as f:
                    f.write(file_data)
            else:
                print(file_data)
        except Exception as e:
            raise Exception("At file %s: %s" % (file, e))


def parse_arguments():
    parser = argparse.ArgumentParser(description="Patch tree json files.")
    parser.add_argument(
        "-f", "--file-name",
        dest="file_name",
        default=None,
        help='file name to patch, print patched file to stdout.'
    )
    parser.add_argument(
        "--override",
        action="store_true",
        dest="override",
        default=False,
        help="override file after patch.",
    )
    parser.add_argument(
        "--patch-all",
        action="store_true",
        dest="patch_all",
        default=False,
        help="patch all tree files and override them.",
    )

    parser.add_argument(
        "--remove-unused-nodes",
        action="store_true",
        dest="remove_unused_nodes",
        default=False,
        help="remove unused nodes from file.",
    )
    parser.add_argument(
        "--regen-ids",
        action="store_true",
        dest="regen_ids",
        default=False,
        help="generate new random id for every node and tree root.",
    )
    parser.add_argument(
        "--remove-useless-node-titles",
        action="store_true",
        dest="remove_useless_node_titles",
        default=False,
        help="remove useless node titles from file.",
    )
    parser.add_argument(
        "--pre-commit-patch",
        action="store_true",
        dest="pre_commit_patch",
        default=False,
        help="enable --patch-all --remove-unused-nodes --remove-useless-node-titles.",
    )

    return parser.parse_args()


if __name__ == "__main__":
    main(parse_arguments())
