#!/usr/bin/env bash

set -e

# Easy script for profiling

print_usage() {
    echo "Run program with given arguments and profile it"
    echo "Docs: https://mvel.at.yandex-team.ru/3405"
    echo "Usage:"
    echo "$0 [--text] <program> <program-params>"
    exit 1
}

# do not do anyting except usage printing if run without options
if [ $# -lt 1 ] || [ "$1" == "-h" ] || [ "$1" == "--help" ] ; then
    print_usage
fi

profile_mode="--dot"
if [ "$1" == "--text" ] ; then
    profile_mode="$1"
    shift || true
fi

if grep -q 14.0 /etc/*release > /dev/null ; then
    echo "Ubuntu 14.04 detected, falling back to text mode profiling, see SEARCH-1209 for details"
    profile_mode="--text"
fi

binary_name="$1"
if [ -z "$binary_name" ] ; then
    print_usage
fi

dt="$(date +%Y-%m-%d-%H-%M-%S)"
profile_name="./profile/$dt/$(basename "$binary_name")"
dot_name="./profile/$dt/dot.out.$(basename $profile_name)"
cg_name="./profile/$dt/callgrind.out.$(basename $profile_name)"
png_name="./profile/$dt/dot.out.$(basename $profile_name).png"

# configure profiler

myself="$(readlink -f "$0")"
my_dir="$(dirname "$myself")"
ya_tool="ya"
echo "Using ya tool from '$ya_tool'"
$ya_tool tool gpt_perf
gpt_path="$($ya_tool tool gpt_perf --print-path)/gpt"

# make profile directory
mkdir -p "$(dirname $profile_name)"

# ser profile file name to dump into
export CPUPROFILE="$profile_name"

# set interruption frequency (greater values -> more precise profiling)
export CPUPROFILE_FREQUENCY=1000

rm -f $profile_name*

# run program
echo Executing program $@ using $gpt_path profiler tools
time ( LD_LIBRARY_PATH="$gpt_path/lib/" LD_PRELOAD=libprofiler.so "$@" )

# generate profile info for callgrind
echo "Annotating profile $profile_name to $dot_name"
LC_ALL=en_US.UTF-8 $gpt_path/bin/pprof $profile_mode $binary_name $profile_name > $dot_name
dot -Tpng $dot_name -o $png_name

echo "Annotating profile $profile_name to $cg_name"
LC_ALL=en_US.UTF-8 $gpt_path/bin/pprof --callgrind $binary_name $profile_name > $cg_name

