import argparse
import os
import logging

import library.python.init_log

from yp.client import YpClient
from yp.common import YpNoSuchObjectError

from sandbox.common import rest


YP_TOKEN_ENV = "YP_TOKEN"
YP_TOKEN_FILE = os.path.expanduser("~/.yp/token")


def get_token(token_env, token_path):
    token = os.getenv(token_env)
    if token:
        logging.info("Use yp token from env '{}'".format(token_env))
        return token
    if os.path.isfile(token_path):
        logging.info("Use yp token from file '{}'".format(token_path))
        with open(token_path, 'r') as f:
            return f.read().strip()
    raise RuntimeError("No yp token provided")


def parse_arguments():
    parser = argparse.ArgumentParser(description="Print pod_sets/stages with deprecated components.")
    parser.add_argument("-c", "--cluster", default=['sas-test', 'man-pre'], nargs='*')
    return parser.parse_args()


def is_bad_pod(pod_agent_payload_meta, pod_agent_rbtorrent_white_list):
    return pod_agent_payload_meta['url'] not in pod_agent_rbtorrent_white_list


def parse_owners(acls):
    owners = set()
    for acl in acls:
        for subject in acl['subjects']:
            if ('robot' not in subject) and (not subject.startswith('abc:')) and (not subject.startswith('deploy:')):
                owners.add(subject)
    return owners


def select_pods_page(client, continuation_token):
    LIMIT = 500
    options = {"limit": LIMIT}
    if continuation_token is not None:
        options["continuation_token"] = continuation_token
    logging.info("selecting %s pods", LIMIT)
    return client.select_objects(
        "pod",
        filter='[/spec/pod_agent_payload/meta]!=#',
        selectors=["/meta/id", "/meta/pod_set_id", "/spec/pod_agent_payload/meta"],
        options=options,
        enable_structured_response=True,
    )


def iter_pods(client):
    response = select_pods_page(client, None)
    while response["results"]:
        for pod in response["results"]:
            yield pod
        response = select_pods_page(client, response["continuation_token"])


def scan_cluster(cluster, yp_token, pod_agent_rbtorrent_white_list):
    logging.info("scanning cluster %s", cluster)
    client = YpClient(cluster, config={"token": yp_token})

    bad_pod_sets = set()
    for fields in iter_pods(client):
        if is_bad_pod(fields[2]["value"], pod_agent_rbtorrent_white_list):
            logging.info("bad %s on %s", fields[0]["value"], fields[1]["value"])
            bad_pod_sets.add(fields[1]["value"])

    YP_STAGE_CLUSTERS = ['sas-test', 'man-pre', 'xdc']
    yp_clients = {cl: YpClient(cl, config={"token": yp_token}) for cl in YP_STAGE_CLUSTERS}

    by_user = {}
    bad_pod_sets = list(bad_pod_sets)
    for i in range(len(bad_pod_sets)):
        logging.info("%s / %s", i, len(bad_pod_sets))
        pod_set_id = bad_pod_sets[i]

        owners = set()

        fields = client.get_object("pod_set", pod_set_id, selectors=["/meta/acl"])
        owners |= parse_owners(fields[0])

        replica_set_id = None
        try:
            fields = client.get_object("replica_set", pod_set_id, selectors=["/meta/acl"])
            replica_set_id = pod_set_id
            owners |= parse_owners(fields[0])
        except YpNoSuchObjectError:
            pass

        STAGE_CLUSTERS = ['sas-test', 'man-pre'] if cluster in ['sas-test', 'man-pre'] else ['xdc']

        mcrs_cluster = None
        mcrs_id = None
        for candidate_cluster in STAGE_CLUSTERS:
            mcrs_client = yp_clients[candidate_cluster]
            try:
                candidate_id = pod_set_id.split('.')[0]
                fields = mcrs_client.get_object("multi_cluster_replica_set", candidate_id, selectors=["/meta/acl"])
                mcrs_cluster = candidate_cluster
                mcrs_id = candidate_id
                owners |= parse_owners(fields[0])
            except YpNoSuchObjectError:
                pass

        stage_cluster = None
        stage_id = None
        for candidate_cluster in STAGE_CLUSTERS:
            stage_client = yp_clients[candidate_cluster]
            try:
                candidate_id = pod_set_id.split('.')[0]
                fields = stage_client.get_object("stage", candidate_id, selectors=["/meta/acl"])
                stage_cluster = candidate_cluster
                stage_id = candidate_id
                owners |= parse_owners(fields[0])
            except YpNoSuchObjectError:
                pass

        if stage_id and stage_id.startswith('pod-agent-test-stage-on'):
            continue  # skip pod agent test stages

        owner = list(owners)[0] if owners else 'unknown'
        if stage_id:
            by_user.setdefault(owner, {}).setdefault(stage_cluster, {}).setdefault("stages", set()).add(stage_id)
        elif mcrs_id:
            by_user.setdefault(owner, {}).setdefault(mcrs_cluster, {}).setdefault("multi_cluster_replica_sets", set()).add(mcrs_id)
        elif replica_set_id:
            by_user.setdefault(owner, {}).setdefault(cluster, {}).setdefault("replica_sets", set()).add(replica_set_id)
        else:
            by_user.setdefault(owner, {}).setdefault(cluster, {}).setdefault("pod_sets", set()).add(pod_set_id)
    return by_user


def format_messages(by_user):
    for user, cl_objs in by_user.iteritems():
        message = "привет!\nможешь пожалуйста удалить/обновить:\n"
        for cluster, objs in cl_objs.iteritems():
            for pod_set_id in objs.get("pod_sets", []):
                message += "ya tool yp remove --address %s pod_set %s\n" % (cluster, pod_set_id)
            for replica_set_id in objs.get("replica_sets", []):
                message += "ya tool yp remove --address %s replica_set %s\n" % (cluster, replica_set_id)
            for mcrs_id in objs.get("multi_cluster_replica_sets", []):
                message += "ya tool dctl remove mcrs -c %s %s\n" % (cluster, mcrs_id)
            for stage_id in objs.get("stages", []):
                message += "https://%s.yandex-team.ru/project/%s\n" % (
                    {"sas-test": "test.deploy", "man-pre": "man-pre.deploy", "xdc": "yd"}[cluster],
                    stage_id
                )
                message += "  удалить:       ya tool dctl remove stage -c %s %s\n" % (cluster, stage_id)
                message += "  либо обновить: ya tool dctl get stage -c %s %s > stage.yaml && ya tool dctl put stage -c %s stage.yaml\n" % (
                    cluster,
                    stage_id,
                    cluster
                )
        message += "?\nподчищаем старые pod_agent в YD"
        logging.info("%s : %s", user, message)


def main(arguments):
    library.python.init_log.init_log(level='INFO')

    pod_agent_rbtorrent_white_list = set()
    pod_agent_white_list = rest.Client().resource.read(type='POD_AGENT_BINARY', limit=1000, attrs={'released_sas_test': True})
    for it in pod_agent_white_list['items']:
        MAJOR_THRESHOLD = 79
        prefix = 'pod_agent-'
        major_version = it['description'][len(prefix):len(prefix) + 2]
        if int(major_version) >= MAJOR_THRESHOLD:
            logging.info("pod_agent %s : %s", major_version, it['skynet_id'])
            pod_agent_rbtorrent_white_list.add(it['skynet_id'])

    yp_token = get_token(YP_TOKEN_ENV, YP_TOKEN_FILE)

    by_user = {}
    for yp_cluster in arguments.cluster:
        res = scan_cluster(yp_cluster, yp_token, pod_agent_rbtorrent_white_list)
        for user, cl_objs in res.iteritems():
            for cluster, objects in cl_objs.iteritems():
                for merge_key in objects.keys():
                    cur = by_user.setdefault(user, {}).setdefault(cluster, {}).setdefault(merge_key, set())
                    cur |= objects[merge_key]

    format_messages(by_user)


if __name__ == "__main__":
    main(parse_arguments())
