import argparse
import logging
import os
import time

import yp.client
import yp.data_model as data_model
import yt.yson as yson
from yp_proto.yp.client.api.proto import object_service_pb2

import infra.dctl.src.consts as consts

YP_TOKEN_ENV = "YP_TOKEN"
YP_TOKEN_FILE = os.path.expanduser("~/.yp/token")

DEFAULT_TRIES = 50
DEFAULT_TRY_INTERVAL = 20


def get_cluster_list():
    ret = []
    for cluster in consts.CLUSTER_CONFIGS.keys():
        ret.append(cluster)

    return ret


def get_token(token_env, token_path):
    token = os.getenv(token_env)
    if token:
        logging.info("Use yp token from env '{}'".format(token_env))
        return token

    if os.path.isfile(token_path):
        logging.info("Use yp token from file '{}'".format(token_path))
        with open(token_path, 'r') as f:
            return f.read().strip()

    raise RuntimeError("No yp token provided")


def get_stage_info(yp_client_stub, stage):
    req = object_service_pb2.TReqGetObject()
    req.object_type = data_model.OT_STAGE
    req.object_id = stage

    req.selector.paths.append("/spec")
    req.selector.paths.append("/status")

    resp = yp_client_stub.GetObject(req)
    return yson.loads(resp.result.values[0]), yson.loads(resp.result.values[1])


def check_stage_for_ready(yp_client_stub, stage_id):
    try:
        logging.info("Get stage info")
        spec, status = get_stage_info(yp_client_stub, stage_id)
    except Exception as e:
        logging.error("Exception while getting stage info: '{}'".format(str(e)))
        return

    if "deploy_units" not in spec:
        logging.warning("No deploy units in spec")
        return False

    if "deploy_units" not in status:
        logging.warning("No deploy units in status")
        return False

    deploy_unit_specs = spec["deploy_units"]
    deploy_unit_statuses = status["deploy_units"]

    logging.info("Stage has '{}' deploy units in spec".format(len(deploy_unit_specs)))
    logging.info("Stage has '{}' deploy units in status".format(len(deploy_unit_statuses)))

    for deploy_unit_id, deploy_unit_spec in deploy_unit_specs.iteritems():
        if deploy_unit_id not in deploy_unit_statuses:
            logging.warning("Deploy unit from spec '{}' not found in status".format(deploy_unit_id))
            return False

    for deploy_unit_id, deploy_unit_status in deploy_unit_statuses.iteritems():
        if deploy_unit_id not in deploy_unit_specs:
            logging.warning("Deploy unit from status '{}' not found in spec".format(deploy_unit_id))
            return False

    all_ready = True
    for deploy_unit_id, deploy_unit_status in deploy_unit_statuses.iteritems():
        if "target_revision" not in deploy_unit_status:
            logging.warning("'target_revision' not found in deploy unit status '{}'".format(deploy_unit_id))
            all_ready = False
            continue

        deploy_unit_spec_revision = deploy_unit_specs[deploy_unit_id].get("revision", 0)
        deploy_unit_status_target_revision = deploy_unit_status["target_revision"]
        if deploy_unit_status_target_revision != deploy_unit_spec_revision:
            logging.info(
                "target revision mismatch expected '{}' got '{}' in '{}'".format(
                    deploy_unit_spec_revision,
                    deploy_unit_status_target_revision,
                    deploy_unit_id
                )
            )
            all_ready = False
            continue

        logging.info("target revision '{}' matched in '{}'".format(deploy_unit_spec_revision, deploy_unit_id))

        if "ready" not in deploy_unit_status:
            logging.warning("'ready' not found in deploy unit status '{}'".format(deploy_unit_id))
            all_ready = False
            continue

        ready = deploy_unit_status["ready"]

        if "status" not in ready:
            logging.warning("'ready/status' not found in deploy unit status '{}'".format(deploy_unit_id))
            all_ready = False
            continue

        if ready["status"] != "true":
            if "reason" not in ready:
                logging.warning("'ready/reason' not fount in deploy unit status '{}'".format(deploy_unit_id))
                reason = "No reason"
            else:
                reason = ready["reason"]

            logging.info("Deploy unit '{}' not ready, reason '{}'".format(deploy_unit_id, reason))
            all_ready = False
            continue

        logging.info("deploy unit ready '{}'".format(deploy_unit_id))

    return all_ready


def main(arguments):
    cluster_config = consts.CLUSTER_CONFIGS[arguments.cluster]

    yp_token = get_token(YP_TOKEN_ENV, YP_TOKEN_FILE)
    yp_client = yp.client.YpClient(
        address=cluster_config.address,
        config={
            'token': yp_token,
        }
    )
    yp_client_stub = yp_client.create_grpc_object_stub()

    is_stage_ready = False
    for i in range(arguments.tries):
        is_stage_ready = check_stage_for_ready(yp_client_stub, arguments.stage_id)
        if is_stage_ready:
            break
        else:
            logging.info("Sleep for {} seconds".format(arguments.interval))
            time.sleep(arguments.interval)

    if not is_stage_ready:
        raise RuntimeError("Stage not ready after {} tries with interval {}".format(arguments.tries, arguments.interval))


def parse_arguments():
    parser = argparse.ArgumentParser(description="Wait stage for ready state.")
    parser.add_argument(
        "cluster",
        metavar="cluster",
        type=str,
        choices=get_cluster_list(),
        help="cluster"
    )
    parser.add_argument(
        "stage_id",
        metavar="stage_id",
        type=str,
        help="stage id"
    )
    parser.add_argument(
        "-t", "--tries",
        dest="tries",
        type=int,
        default=DEFAULT_TRIES,
        help="number of tries."
    )
    parser.add_argument(
        "-i", "--interval",
        dest="interval",
        type=int,
        default=DEFAULT_TRY_INTERVAL,
        help="interval between tries (in seconds)."
    )

    return parser.parse_args()


if __name__ == "__main__":
    logging.basicConfig(format='%(asctime)s %(levelname)s:%(message)s', level=logging.INFO)
    main(parse_arguments())
