package porto

import (
	pb "a.yandex-team.ru/infra/porto/proto"
	"bufio"
	"encoding/binary"
	"fmt"
	"github.com/golang/protobuf/proto"
	"io"
	"math"
	"net"
	"os"
	"strconv"
	"syscall"
	"time"
)

const PortoSocket = "/run/portod.socket"

const (
	InfiniteTimeout    time.Duration = -1
	DefaultTimeout                   = 5 * time.Minute
	DefaultDiskTimeout               = 15 * time.Minute
	DefaultStopTimeout               = 5 * time.Minute
)

type PortoError struct {
	Code    pb.EError
	Message string
}

func (err *PortoError) Error() string {
	return fmt.Sprintf("%s: %s", pb.EError_name[int32(err.Code)], err.Message)
}

type PortoProperty struct {
	Name        string
	Description string
}

type PortoAPI interface {
	/* Connection */

	Connect() error
	Close() error
	IsConnected() bool
	SetAutoReconnect(bool)

	/* Timeout */

	GetTimeout() time.Duration
	SetTimeout(timeout time.Duration)
	GetDiskTimeout() time.Duration
	SetDiskTimeout(timeout time.Duration)

	/* Place */

	GetPlace() string
	SetPlace(place string) string

	/* Transport  */

	WriteRequest(*pb.TPortoRequest) error
	ReadResponse() (*pb.TPortoResponse, error)
	CallTimeout(*pb.TPortoRequest, time.Duration) (*pb.TPortoResponse, error)
	Call(*pb.TPortoRequest) (*pb.TPortoResponse, error)

	/* System */

	GetVersion() (tag string, rev string, _ error)
	GetSystem() (*pb.TGetSystemResponse, error)

	/* Container */

	ListProperties() ([]PortoProperty, error)

	List() ([]string, error)
	ListContainers(mask string) ([]string, error)

	GetProperty(name string, property string) (string, error)
	SetProperty(name string, property string, value string) error

	GetPropertyIndex(name string, property string, index string) (string, error)
	SetPropertyIndex(name string, property string, index string, value string) error

	GetIntProperty(name string, property string) (uint64, error)
	SetIntProperty(name string, property string, value uint64) error

	GetIntPropertyIndex(name string, property string, index string) (uint64, error)
	SetIntPropertyIndex(name string, property string, index string, value uint64) error

	GetLabel(name string, label string) (string, error)
	SetLabel(name string, label string, value string, prev string) (string, error)
	IncLabel(name string, label string, add int64) (int64, error)

	Get(names []string, properties []string) (*pb.TGetResponse, error)
	GetProperties(names []string, properties []string) (map[string]map[string]string, error)

	Create(name string) error
	CreateWeak(name string) error
	Destroy(name string) error

	Start(name string) error
	Stop(name string) error
	Kill(name string, sig syscall.Signal) error
	Pause(name string) error
	Resume(name string) error
	Respawn(name string) error

	Wait(names []string, timeout time.Duration) (*pb.TWaitResponse, error)
	WaitLabels(names []string, labels []string, timeout time.Duration) (*pb.TWaitResponse, error)
	WaitContainer(name string, timeout time.Duration) (state string, _ error)
	WaitContainers(names []string, timeout time.Duration) (name string, state string, _ error)

	AttachProcess(name string, pid uint32, comm string) error
	AttachThread(name string, pid uint32, comm string) error
	LocateProcess(pid uint32, comm string) (string, error)

	ConvertPath(path string, src string, dest string) (string, error)

	/* Volume */

	ListVolumeProperties() ([]PortoProperty, error)

	ListVolumes(path string, container string) ([]*pb.TVolumeDescription, error)
	GetVolume(path string) (*pb.TVolumeDescription, error)
	CreateVolume(path string, properties map[string]string) (*pb.TVolumeDescription, error)
	TuneVolume(path string, properties map[string]string) error

	LinkVolume(path string, container string) error
	UnlinkVolume(path string, container string) error
	LinkVolumeTarget(path string, container string, target string, readOnly bool) error
	UnlinkVolumeTarget(path string, container string, target string) error
	UnlinkVolumeStrict(path string, container string, target string, strict bool) error

	/* Layer */

	ListLayers(mask string) ([]*pb.TLayer, error)
	ImportLayer(layer string, tarball string, merge bool, private string) error
	ExportLayer(volume string, tarball string) error
	RemoveLayer(layer string) error

	GetLayerPrivate(layer string) (string, error)
	SetLayerPrivate(layer string, private string) error

	/* Storage */

	ListStorages(mask string) ([]*pb.TStorage, error)
	RemoveStorage(name string) error
}

type client struct {
	conn        net.Conn
	reader      *bufio.Reader
	noReconnect bool
	timeout     time.Duration
	diskTimeout time.Duration
	place       string
}

func Dial() (PortoAPI, error) {
	c := new(client)
	return c, nil
}

func (c *client) IsConnected() bool {
	return c.conn != nil
}

func (c *client) SetAutoReconnect(reconnect bool) {
	c.noReconnect = !reconnect
}

// Connect connects to the address of portod unix domain socket
//
// For testing purposes inside containers default path to Portod
// socket could be replaced by environment variable PORTO_SOCKET.
func (c *client) Connect() error {
	_ = c.Close()

	portoSocketPath := PortoSocket
	rv, ok := os.LookupEnv("PORTO_SOCKET")
	if ok {
		portoSocketPath = rv
	}
	conn, err := net.DialTimeout("unix", portoSocketPath, c.GetTimeout())
	if err == nil {
		c.conn = conn
		c.reader = bufio.NewReader(c.conn)
	}
	return err
}

func (c *client) Close() error {
	if c.conn == nil {
		return nil
	}
	err := c.conn.Close()
	c.conn = nil
	c.reader = nil
	return err
}

func (c *client) GetTimeout() time.Duration {
	if c.timeout == 0 {
		return DefaultTimeout
	}
	return c.timeout
}

func (c *client) SetTimeout(timeout time.Duration) {
	c.timeout = timeout
}

func (c *client) GetDiskTimeout() time.Duration {
	if c.diskTimeout == 0 {
		return DefaultDiskTimeout
	}
	return c.diskTimeout
}

func (c *client) SetDiskTimeout(timeout time.Duration) {
	c.diskTimeout = timeout
}

func (c *client) GetPlace() string {
	return c.place
}

func (c *client) SetPlace(place string) string {
	old := c.place
	c.place = place
	return old
}

func optString(val string) *string {
	if val == "" {
		return nil
	}
	return &val
}

func (c *client) WriteRequest(req *pb.TPortoRequest) error {
	buf, err := proto.Marshal(req)
	if err != nil {
		return err
	}

	len := len(buf)
	hdr := make([]byte, 64)
	hdrLen := binary.PutUvarint(hdr, uint64(len))

	_, err = c.conn.Write(hdr[:hdrLen])
	if err != nil {
		_ = c.Close()
		return err
	}

	_, err = c.conn.Write(buf)
	if err != nil {
		_ = c.Close()
		return err
	}

	return nil
}

func (c *client) ReadResponse() (*pb.TPortoResponse, error) {
	len, err := binary.ReadUvarint(c.reader)
	if err != nil {
		_ = c.Close()
		return nil, err
	}

	buf := make([]byte, len)
	_, err = io.ReadFull(c.reader, buf)
	if err != nil {
		_ = c.Close()
		return nil, err
	}

	rsp := new(pb.TPortoResponse)
	err = proto.Unmarshal(buf, rsp)
	if err != nil {
		_ = c.Close()
		return nil, err
	}

	if rsp.GetError() != pb.EError_Success {
		err = &PortoError{
			Code:    rsp.GetError(),
			Message: rsp.GetErrorMsg(),
		}
		return rsp, err
	}

	return rsp, nil
}

func (c *client) CallTimeout(req *pb.TPortoRequest, timeout time.Duration) (*pb.TPortoResponse, error) {
	if !c.IsConnected() {
		if c.noReconnect {
			err := &PortoError{
				Code:    pb.EError_SocketError,
				Message: "Socket not connected",
			}
			return nil, err
		}
		err := c.Connect()
		if err != nil {
			return nil, err
		}
	}

	reqTimeout := c.GetTimeout()
	if reqTimeout < 0 {
		err := c.conn.SetDeadline(time.Time{})
		if err != nil {
			return nil, err
		}
	} else {
		deadline := time.Now().Add(reqTimeout)
		err := c.conn.SetWriteDeadline(deadline)
		if err != nil {
			return nil, err
		}

		if timeout < 0 {
			deadline = time.Time{}
		} else {
			deadline = deadline.Add(timeout)
		}

		err = c.conn.SetReadDeadline(deadline)
		if err != nil {
			return nil, err
		}
	}

	err := c.WriteRequest(req)
	if err != nil {
		return nil, err
	}

	return c.ReadResponse()
}

func (c *client) Call(req *pb.TPortoRequest) (*pb.TPortoResponse, error) {
	return c.CallTimeout(req, 0)
}

// System

func (c *client) GetVersion() (tag string, rev string, err error) {
	req := &pb.TPortoRequest{
		Version: &pb.TVersionRequest{},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return "", "", err
	}
	return rsp.GetVersion().GetTag(), rsp.GetVersion().GetRevision(), nil
}

func (c *client) GetSystem() (*pb.TGetSystemResponse, error) {
	req := &pb.TPortoRequest{
		GetSystem: &pb.TGetSystemRequest{},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}
	return rsp.GetGetSystem(), nil
}

// Container

func (c *client) Create(name string) error {
	req := &pb.TPortoRequest{
		Create: &pb.TCreateRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) CreateWeak(name string) error {
	req := &pb.TPortoRequest{
		CreateWeak: &pb.TCreateRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Destroy(name string) error {
	req := &pb.TPortoRequest{
		Destroy: &pb.TDestroyRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Start(name string) error {
	req := &pb.TPortoRequest{
		Start: &pb.TStartRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Stop(name string) error {
	return c.StopTimeout(name, DefaultStopTimeout)
}

func (c *client) StopTimeout(name string, timeout time.Duration) error {
	req := &pb.TPortoRequest{
		Stop: &pb.TStopRequest{
			Name: &name,
		},
	}

	if timeout >= 0 {
		if timeout/time.Millisecond > math.MaxUint32 {
			return fmt.Errorf("timeout must be less than %d ms", math.MaxUint32)
		}

		timeoutms := uint32(timeout / time.Millisecond)
		req.Stop.TimeoutMs = &timeoutms
	} else {
		timeout = 0
	}

	_, err := c.CallTimeout(req, timeout)
	return err
}

func (c *client) Kill(name string, sig syscall.Signal) error {
	signum := int32(sig)
	req := &pb.TPortoRequest{
		Kill: &pb.TKillRequest{
			Name: &name,
			Sig:  &signum,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Pause(name string) error {
	req := &pb.TPortoRequest{
		Pause: &pb.TPauseRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Resume(name string) error {
	req := &pb.TPortoRequest{
		Resume: &pb.TResumeRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) Respawn(name string) error {
	req := &pb.TPortoRequest{
		Respawn: &pb.TRespawnRequest{
			Name: &name,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) WaitLabels(names []string, labels []string, timeout time.Duration) (*pb.TWaitResponse, error) {
	req := &pb.TPortoRequest{
		Wait: &pb.TWaitRequest{
			Name:  names,
			Label: labels,
		},
	}

	if timeout >= 0 {
		if timeout/time.Millisecond > math.MaxUint32 {
			return nil, fmt.Errorf("timeout must be less than %d ms", math.MaxUint32)
		}

		timeoutms := uint32(timeout / time.Millisecond)
		req.Wait.TimeoutMs = &timeoutms
	}

	rsp, err := c.Call(req)
	return rsp.GetWait(), err
}

func (c *client) Wait(names []string, timeout time.Duration) (*pb.TWaitResponse, error) {
	return c.WaitLabels(names, nil, timeout)
}

func (c *client) WaitContainer(name string, timeout time.Duration) (string, error) {
	res, err := c.Wait([]string{name}, timeout)
	return res.GetState(), err
}

func (c *client) WaitContainers(names []string, timeout time.Duration) (string, string, error) {
	res, err := c.Wait(names, timeout)
	return res.GetName(), res.GetState(), err
}

func (c *client) List() ([]string, error) {
	return c.ListContainers("")
}

func (c *client) ListContainers(mask string) ([]string, error) {
	req := &pb.TPortoRequest{
		List: &pb.TListRequest{
			Mask: optString(mask),
		},
	}

	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}

	return rsp.GetList().GetName(), nil
}

func (c *client) ListProperties() (ret []PortoProperty, err error) {
	req := &pb.TPortoRequest{
		ListProperties: &pb.TListPropertiesRequest{},
	}
	rsp, err := c.Call(req)
	for _, prop := range rsp.GetListProperties().GetList() {
		var p = PortoProperty{
			Name:        prop.GetName(),
			Description: prop.GetDesc(),
		}
		ret = append(ret, p)
	}
	return ret, err
}

func (c *client) Get(containers []string, properties []string) (ret *pb.TGetResponse, err error) {
	req := &pb.TPortoRequest{
		Get: &pb.TGetRequest{
			Name:     containers,
			Variable: properties,
		},
	}

	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}

	return rsp.GetGet(), nil
}

func (c *client) GetProperties(containers []string, properties []string) (ret map[string]map[string]string, _ error) {
	ret = make(map[string]map[string]string)
	res, err := c.Get(containers, properties)
	if err != nil {
		return nil, err
	}
	for _, item := range res.GetList() {
		ret[item.GetName()] = make(map[string]string)
		for _, value := range item.GetKeyval() {
			ret[item.GetName()][value.GetVariable()] = value.GetValue()
		}
	}
	return ret, nil
}

func (c *client) GetProperty(name string, property string) (string, error) {
	req := &pb.TPortoRequest{
		GetProperty: &pb.TGetPropertyRequest{
			Name:     &name,
			Property: &property,
		},
	}

	rsp, err := c.Call(req)
	if err != nil {
		return "", err
	}

	return rsp.GetGetProperty().GetValue(), nil
}

func (c *client) SetProperty(name string, property string, value string) error {
	req := &pb.TPortoRequest{
		SetProperty: &pb.TSetPropertyRequest{
			Name:     &name,
			Property: &property,
			Value:    &value,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) GetPropertyIndex(name string, property string, index string) (string, error) {
	return c.GetProperty(name, property+"["+index+"]")
}

func (c *client) SetPropertyIndex(name string, property string, index string, value string) error {
	return c.SetProperty(name, property+"["+index+"]", value)
}

func (c *client) GetIntProperty(name string, property string) (uint64, error) {
	return c.GetIntPropertyIndex(name, property, "")
}

func (c *client) SetIntProperty(name string, property string, value uint64) error {
	return c.SetIntPropertyIndex(name, property, "", value)
}

func (c *client) GetIntPropertyIndex(name string, property string, index string) (uint64, error) {
	str, err := c.GetPropertyIndex(name, property, index)
	if err == nil {
		return strconv.ParseUint(str, 10, 64)
	}

	return 0, err
}

func (c *client) SetIntPropertyIndex(name string, property string, index string, value uint64) error {
	return c.SetPropertyIndex(name, property, index, strconv.FormatUint(value, 10))
}

func (c *client) GetLabel(name string, label string) (string, error) {
	return c.GetPropertyIndex(name, "labels", label)
}

func (c *client) SetLabel(name string, label string, value string, prev string) (string, error) {
	req := &pb.TPortoRequest{
		SetLabel: &pb.TSetLabelRequest{
			Name:      &name,
			Label:     &label,
			Value:     &value,
			PrevValue: optString(prev),
		},
	}
	rsp, err := c.Call(req)
	return rsp.GetSetLabel().GetPrevValue(), err
}

func (c *client) IncLabel(name string, label string, add int64) (int64, error) {
	req := &pb.TPortoRequest{
		IncLabel: &pb.TIncLabelRequest{
			Name:  &name,
			Label: &label,
			Add:   &add,
		},
	}
	rsp, err := c.Call(req)
	return rsp.GetIncLabel().GetResult(), err
}

func (c *client) AttachProcess(name string, pid uint32, comm string) error {
	req := &pb.TPortoRequest{
		AttachProcess: &pb.TAttachProcessRequest{
			Name: &name,
			Pid:  &pid,
			Comm: &comm,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) AttachThread(name string, pid uint32, comm string) error {
	req := &pb.TPortoRequest{
		AttachThread: &pb.TAttachProcessRequest{
			Name: &name,
			Pid:  &pid,
			Comm: &comm,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) LocateProcess(pid uint32, comm string) (string, error) {
	req := &pb.TPortoRequest{
		LocateProcess: &pb.TLocateProcessRequest{
			Pid:  &pid,
			Comm: &comm,
		},
	}
	res, err := c.Call(req)
	if err != nil {
		return "", err
	}
	return res.GetLocateProcess().GetName(), nil
}

func (c *client) ConvertPath(path string, src string, dest string) (string, error) {
	req := &pb.TPortoRequest{
		ConvertPath: &pb.TConvertPathRequest{
			Path:        &path,
			Source:      &src,
			Destination: &dest,
		},
	}

	rsp, err := c.Call(req)
	if err != nil {
		return "", err
	}

	return rsp.GetConvertPath().GetPath(), nil
}

// VolumeAPI

func (c *client) ListVolumeProperties() (ret []PortoProperty, err error) {
	req := &pb.TPortoRequest{
		ListVolumeProperties: &pb.TListVolumePropertiesRequest{},
	}

	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}

	for _, prop := range rsp.GetListVolumeProperties().GetList() {
		var p = PortoProperty{
			Name:        prop.GetName(),
			Description: prop.GetDesc(),
		}
		ret = append(ret, p)
	}
	return ret, err
}

func (c *client) CreateVolume(path string, properties map[string]string) (vol *pb.TVolumeDescription, err error) {
	req := &pb.TPortoRequest{
		CreateVolume: &pb.TCreateVolumeRequest{
			Properties: properties,
			Path:       optString(path),
		},
	}

	if _, ok := properties["place"]; !ok && c.place != "" {
		properties["place"] = c.place
	}

	rsp, err := c.CallTimeout(req, c.diskTimeout)
	if err != nil {
		return nil, err
	}
	return rsp.GetCreateVolume(), nil
}

func (c *client) TuneVolume(path string, properties map[string]string) error {
	req := &pb.TPortoRequest{
		TuneVolume: &pb.TTuneVolumeRequest{
			Path:       &path,
			Properties: properties,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) LinkVolume(path string, container string) error {
	req := &pb.TPortoRequest{
		LinkVolume: &pb.TLinkVolumeRequest{
			Path:      &path,
			Container: &container,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) LinkVolumeTarget(path string, container string, target string, readOnly bool) error {
	req := &pb.TPortoRequest{
		LinkVolumeTarget: &pb.TLinkVolumeRequest{
			Path:      &path,
			Container: &container,
			Target:    optString(target),
			ReadOnly:  &readOnly,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) UnlinkVolume(path string, container string) error {
	req := &pb.TPortoRequest{
		UnlinkVolume: &pb.TUnlinkVolumeRequest{
			Path:      &path,
			Container: optString(container),
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) UnlinkVolumeTarget(path string, container string, target string) error {
	req := &pb.TPortoRequest{
		UnlinkVolumeTarget: &pb.TUnlinkVolumeRequest{
			Path:      &path,
			Container: optString(container),
			Target:    optString(target),
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) UnlinkVolumeStrict(path string, container string, target string, strict bool) error {
	req := &pb.TPortoRequest{
		UnlinkVolumeTarget: &pb.TUnlinkVolumeRequest{
			Path:      &path,
			Container: optString(container),
			Target:    optString(target),
			Strict:    &strict,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) ListVolumes(path string, container string) ([]*pb.TVolumeDescription, error) {
	req := &pb.TPortoRequest{
		ListVolumes: &pb.TListVolumesRequest{
			Path:      optString(path),
			Container: optString(container),
		},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}
	return rsp.GetListVolumes().GetVolumes(), nil
}

func (c *client) GetVolume(path string) (*pb.TVolumeDescription, error) {
	req := &pb.TPortoRequest{
		ListVolumes: &pb.TListVolumesRequest{
			Path: &path,
		},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}
	return rsp.GetListVolumes().GetVolumes()[0], nil
}

// LayerAPI

func (c *client) ImportLayer(layer string, tarball string, merge bool, private string) error {
	req := &pb.TPortoRequest{
		ImportLayer: &pb.TImportLayerRequest{
			Place:        optString(c.place),
			Layer:        &layer,
			Tarball:      &tarball,
			Merge:        &merge,
			PrivateValue: &private,
		},
	}
	_, err := c.CallTimeout(req, c.GetDiskTimeout())
	return err
}

func (c *client) ExportLayer(volume string, tarball string) error {
	req := &pb.TPortoRequest{
		ExportLayer: &pb.TExportLayerRequest{
			Volume:  &volume,
			Tarball: &tarball,
		},
	}
	_, err := c.CallTimeout(req, c.GetDiskTimeout())
	return err
}

func (c *client) RemoveLayer(layer string) error {
	req := &pb.TPortoRequest{
		RemoveLayer: &pb.TRemoveLayerRequest{
			Place: optString(c.place),
			Layer: &layer,
		},
	}
	_, err := c.CallTimeout(req, c.GetDiskTimeout())
	return err
}

func (c *client) ListLayers(mask string) ([]*pb.TLayer, error) {
	req := &pb.TPortoRequest{
		ListLayers: &pb.TListLayersRequest{
			Place: optString(c.place),
			Mask:  optString(mask),
		},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}
	return rsp.GetListLayers().GetLayers(), nil
}

func (c *client) GetLayerPrivate(layer string) (string, error) {
	req := &pb.TPortoRequest{
		GetLayerPrivate: &pb.TGetLayerPrivateRequest{
			Place: optString(c.place),
			Layer: &layer,
		},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return "", err
	}
	return rsp.GetGetLayerPrivate().GetPrivateValue(), nil
}

func (c *client) SetLayerPrivate(layer string, private string) error {
	req := &pb.TPortoRequest{
		SetLayerPrivate: &pb.TSetLayerPrivateRequest{
			Place:        optString(c.place),
			Layer:        &layer,
			PrivateValue: &private,
		},
	}
	_, err := c.Call(req)
	return err
}

func (c *client) ListStorages(mask string) ([]*pb.TStorage, error) {
	req := &pb.TPortoRequest{
		ListStorages: &pb.TListStoragesRequest{
			Place: optString(c.place),
			Mask:  optString(mask),
		},
	}
	rsp, err := c.Call(req)
	if err != nil {
		return nil, err
	}
	return rsp.GetListStorages().GetStorages(), nil
}

func (c *client) RemoveStorage(name string) error {
	req := &pb.TPortoRequest{
		RemoveStorage: &pb.TRemoveStorageRequest{
			Place: optString(c.place),
			Name:  &name,
		},
	}
	_, err := c.CallTimeout(req, c.GetDiskTimeout())
	return err
}
