package porto

import (
	"os"
	"os/exec"
	"strings"
	"syscall"
	"testing"
)

func FailOnError(t *testing.T, conn PortoAPI, err error) {
	if err != nil {
		t.Error(err)
	}
}

func ConnectToPorto(t *testing.T) PortoAPI {
	conn, err := Dial()
	if err != nil {
		t.Error(err)
		t.FailNow()
	}
	return conn
}

const testContainer string = "golang_testContainer"
const testVolume string = "/tmp/golang_testVolume"
const testLayer string = "golang_testLayer"
const testTarball = "/tmp/" + testLayer + ".tgz"
const testStorage string = "go_abcde"
const testPlace string = "/tmp/golang_place"

func TestGetVersion(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	maj, min, err := conn.GetVersion()
	FailOnError(t, conn, err)
	t.Logf("Porto version %s.%s", maj, min)
}

func TestListProperties(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	plist, err := conn.ListProperties()
	FailOnError(t, conn, err)
	for i := range plist {
		if plist[i].Name == "command" {
			t.Logf("Porto supports command property: %s", plist[i].Description)
			return
		}
	}
	t.Error("Porto doesn't support command property")
	t.FailNow()
}

func TestCreateWeak(t *testing.T) {
	conn := ConnectToPorto(t)
	FailOnError(t, conn, conn.CreateWeak(testContainer))
	_ = conn.Close()

	conn = ConnectToPorto(t)
	defer conn.Close()
	err := conn.Destroy(testContainer)
	if err == nil {
		t.Fail()
	}
}

func TestCreate(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Create(testContainer))
}

func TestList(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	list, err := conn.List()
	FailOnError(t, conn, err)
	for i := range list {
		if list[i] == testContainer {
			return
		}
	}
	t.Error("Created container isn't presented in a list")
	t.FailNow()
}

func TestSetProperty(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.SetProperty(testContainer, "command", "sleep 10000"))
}

func TestGetProperty(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	value, err := conn.GetProperty(testContainer, "command")
	FailOnError(t, conn, err)
	if value != "sleep 10000" {
		t.Error("Got a wrong command value")
		t.FailNow()
	}
}

func TestStart(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Start(testContainer))
}

func TestPause(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Pause(testContainer))
}

func TestResume(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Resume(testContainer))
}

func TestKill(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Kill(testContainer, syscall.SIGTERM))
}

func TestWait(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	containers := []string{testContainer}
	container, state, err := conn.WaitContainers(containers, -1)
	FailOnError(t, conn, err)
	if container != testContainer {
		t.Error("Wait returned a wrong container")
		t.FailNow()
	}
	if state != "dead" {
		t.Error("Wait returned a wrong state")
		t.FailNow()
	}
}

func TestGetProperties(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	containers := []string{testContainer}
	variables := []string{"state", "exit_status"}
	res, err := conn.GetProperties(containers, variables)
	FailOnError(t, conn, err)
	if res[testContainer]["state"] != "dead" {
		t.Error("Got a wrong state value")
		t.FailNow()
	}
	if res[testContainer]["exit_status"] != "15" {
		t.Error("Got a wrong exit_status value")
		t.FailNow()
	}
}

func TestConvertPath(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	resp, err := conn.ConvertPath("/", testContainer, testContainer)
	FailOnError(t, conn, err)
	if resp != "/" {
		t.Error("Got wrong path conversion")
		t.FailNow()
	}
}

func TestStop(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Stop(testContainer))
}

func TestDestroy(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Destroy(testContainer))
}

// VolumeAPI
func TestListVolumeProperties(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	properties, err := conn.ListVolumeProperties()
	FailOnError(t, conn, err)
	for i := range properties {
		if properties[i].Name == "backend" {
			return
		}
	}
	t.Error("Porto doesn't list backend volume property")
	t.FailNow()
}

func TestCreateVolume(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	_ = os.Remove(testVolume)
	_ = os.Mkdir(testVolume, 0755)
	config := make(map[string]string)
	config["private"] = "golang test volume"
	_, err := conn.CreateVolume(testVolume, config)
	FailOnError(t, conn, err)
}

func TestListVolumes(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	volumes, err := conn.ListVolumes("", "")
	FailOnError(t, conn, err)
	for i := range volumes {
		if volumes[i].GetPath() == testVolume {
			return
		}
	}
	t.Error("Porto doesn't list previously created volume")
	t.FailNow()
}

func TestLinkVolume(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.Create(testContainer))
	FailOnError(t, conn, conn.LinkVolume(testVolume, testContainer))
}

func TestExportLayer(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.ExportLayer(testVolume, "/tmp/goporto.tgz"))
	_ = os.Remove("/tmp/goporto.tgz")
}

func TestUnlinkVolume(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.UnlinkVolume(testVolume, testContainer))
	FailOnError(t, conn, conn.UnlinkVolume(testVolume, "/"))
	FailOnError(t, conn, conn.Destroy(testContainer))
	_ = os.Remove(testVolume)
}

// LayerAPI
func TestImportLayer(t *testing.T) {
	// Prepare tarball
	_ = os.Mkdir("/tmp/golang_testTarball", 0755)
	defer os.Remove("/tmp/golang_testTarball")
	_ = os.Mkdir("/tmp/golang_testTarball/dir", 0755)
	defer os.Remove("/tmp/golang_testTarball/dir")

	cmd := exec.Command("tar", "-czf", testTarball,
		"/tmp/golang/testTarball")
	err := cmd.Start()
	if err != nil {
		t.Error("Can't prepare test tarball")
		t.SkipNow()
	}
	_ = cmd.Wait()
	defer os.Remove(testTarball)

	// Import
	conn := ConnectToPorto(t)
	defer conn.Close()

	FailOnError(t, conn, conn.ImportLayer(testLayer, testTarball, false, "test"))
}

func TestLayerPrivate(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()

	FailOnError(t, conn, conn.SetLayerPrivate(testLayer, "456"))
	private, err := conn.GetLayerPrivate(testLayer)
	FailOnError(t, conn, err)

	if private == "456" {
		return
	}

	t.Error("Can't get/set new layer private")
	t.FailNow()
}

func TestListLayers(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	layers, err := conn.ListLayers("")
	FailOnError(t, conn, err)
	for i := range layers {
		if layers[i].GetName() == testLayer &&
			layers[i].GetPrivateValue() == "456" &&
			layers[i].GetLastUsage() < 10 {

			return
		}
	}
	t.Error("Porto doesn't list previously imported layer as object")
	t.FailNow()
}

func TestRemoveLayer(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.RemoveLayer(testLayer))
}

func TestCreateStorage(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()

	config := make(map[string]string)
	config["backend"] = "native"
	config["storage"] = testStorage
	config["private"] = "12345"

	volume, err := conn.CreateVolume("", config)
	FailOnError(t, conn, err)
	FailOnError(t, conn, conn.UnlinkVolume(volume.GetPath(), ""))
}

func TestListStorage(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()

	storages, err := conn.ListStorages("")
	FailOnError(t, conn, err)

	for i := range storages {
		if storages[i].GetName() == testStorage &&
			storages[i].GetPrivateValue() == "12345" {
			return
		}
	}

	t.Error("Failed listing storages")
	t.FailNow()
}

func TestRemoveStorage(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()
	FailOnError(t, conn, conn.RemoveStorage(testStorage))
}

func TestPlace(t *testing.T) {
	conn := ConnectToPorto(t)
	defer conn.Close()

	_ = os.RemoveAll(testPlace)
	_ = os.Mkdir(testPlace, 0755)
	_ = os.Mkdir(testPlace+"/porto_volumes", 0755)
	_ = os.Mkdir(testPlace+"/porto_layers", 0755)
	_ = os.Mkdir(testPlace+"/porto_storage", 0755)

	config := make(map[string]string)
	config["private"] = "golang test volume"
	config["storage"] = "abcd"

	conn.SetPlace(testPlace)

	volume, err := conn.CreateVolume("", config)
	FailOnError(t, conn, err)

	if !strings.Contains(volume.GetPath(), testPlace) {
		t.Error("Volume does not use desired place")
		t.FailNow()
	}

	_, err = os.Stat(volume.GetPath())
	FailOnError(t, conn, err)

	storages, err := conn.ListStorages("")
	FailOnError(t, conn, err)

	if len(storages) == 0 || storages[0].GetName() != "abcd" {
		t.Error("Storage failed to be created in place")
		t.FailNow()
	}

	FailOnError(t, conn, conn.UnlinkVolume(volume.GetPath(), ""))
	FailOnError(t, conn, conn.RemoveStorage("abcd"))
	FailOnError(t, conn, os.Remove(testPlace+"/porto_volumes"))
	FailOnError(t, conn, os.Remove(testPlace+"/porto_layers"))
	FailOnError(t, conn, os.Remove(testPlace+"/porto_storage"))
	FailOnError(t, conn, os.Remove(testPlace))
}
