#!/usr/bin/env python3

import subprocess
import contextlib
import argparse
import shutil
import sys
import os


@contextlib.contextmanager
def run(args, **kwargs):
    print("+ '" + "' '".join(args) + "'", flush=True)
    return subprocess.check_call(args, **kwargs)

BUILD_DEPS = [
    'devscripts',
    'equivs',
    'git'
]


def install_deps(control):
    cmd = ['sudo', 'apt-get', 'install', '--no-install-recommends', '-y'] + BUILD_DEPS
    run(['sudo', 'apt-get', 'update', '-y'])
    run(cmd)
    run(['mk-build-deps', '--install', '--tool=apt-get -o Debug::pkgProblemResolver=yes --no-install-recommends --yes', control])


def build_bin(args):
    build_dir = "/tmp/build_porto"
    run(['mkdir', build_dir])

    run(['tar', '--strip-components={}'.format(args.archive_components), '-xf', os.path.abspath(args.src)], cwd=build_dir)

    if args.install_deps:
        install_deps(build_dir + "/debian/control")

    run(['cmake', '.', '-DUSE_PORTO_KERNEL=ON'], cwd=build_dir)
    run(['make', '-j', str(os.sysconf('SC_NPROCESSORS_ONLN'))], cwd=build_dir)

    run(['tar', '-caf', os.path.abspath(args.out), build_dir])


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible extra actions", dest='command')

    build = subparsers.add_parser(name="build")
    build.set_defaults(handle=build_bin)
    build.add_argument("--dry-run", default=False, action='store_true')
    build.add_argument("--install-deps", default=False, action='store_true')
    build.add_argument("--archive-components", default="1")
    build.add_argument("src", help="tarball with porto sources")
    build.add_argument("out", help="tarball with built porto")

    args = parser.parse_args()
    args.handle(args)
