package internal

import (
	"context"
	"net"
	"os"

	"google.golang.org/grpc"

	"go.uber.org/zap"

	diskman_api "a.yandex-team.ru/infra/diskmanager/proto"
	porto_api "a.yandex-team.ru/infra/porto/api_go"
	nvgpu_api "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
)

const NvgpuSockPath = "/var/run/nvgpu-manager.sock"
const DiskmanSockPath = "/var/run/diskman.sock"

func isDaemonRunning(path string) bool {
	_, err := os.Stat(path)
	if err == nil {
		return true
	}
	if !os.IsNotExist(err) {
		zap.S().Errorf("Failed to stat %v: %v", path, err)
	}
	return false
}

func IsNvgpuRunning() bool {
	return isDaemonRunning(NvgpuSockPath)
}

func IsDiskmanRunning() bool {
	return isDaemonRunning(DiskmanSockPath)
}

func CreatePortoClient() (porto_api.PortoAPI, error) {
	conn, err := porto_api.Dial()
	if err != nil {
		zap.S().Errorf("Failed to dial Porto: %v", err)
		return nil, err
	}
	return conn, nil
}

func GrpcDialUnix(path string) (*grpc.ClientConn, error) {
	conn, err := grpc.DialContext(
		context.Background(),
		path,
		grpc.WithInsecure(),
		grpc.WithContextDialer(func(ctx context.Context, addr string) (net.Conn, error) {
			return net.Dial("unix", addr)
		}),
	)
	if err != nil {
		zap.S().Errorf("Failed to dial %v: %v", path, err)
		return nil, err
	}
	return conn, nil
}

func CreateNvgpuClient() (nvgpu_api.NvGpuManagerClient, error) {
	conn, err := GrpcDialUnix(NvgpuSockPath)
	if err != nil {
		return nil, err
	}
	return nvgpu_api.NewNvGpuManagerClient(conn), nil
}

func CreateDiskmanClient() (diskman_api.DiskManagerClient, error) {
	conn, err := GrpcDialUnix(DiskmanSockPath)
	if err != nil {
		return nil, err
	}
	return diskman_api.NewDiskManagerClient(conn), nil
}

func CreateHostClient() (*HostAPI, error) {
	return InitHostAPI()
}
