package internal

import (
	"errors"
	"fmt"

	"go.uber.org/zap"

	"a.yandex-team.ru/infra/gopcm/pkg/ipc"
	"a.yandex-team.ru/infra/gopcm/pkg/membw"
	"a.yandex-team.ru/infra/gopcm/pkg/pcm"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal/util"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/yasm"
)

var (
	BackboneIfname = ""
)

type HostAPI struct {
	pcm *pcm.Pcm

	membw *membw.MemBW
	ipc   *ipc.IPC

	resctrl   bool
	stats     *StatsStorage
	hoststats *HostStatsStorage
	signals   map[string]float64
}

func InitHostAPI() (*HostAPI, error) {
	var err error
	h := &HostAPI{
		stats: &StatsStorage{
			MemoryStats: &MemoryStats{},
			CPUStats:    &CPUStats{},
		},
		hoststats: &HostStatsStorage{
			HostStats: nil,
		},
	}
	// needed for NUMA remote and local bytes extraction
	err = util.MountResctrl()
	if err != nil {
		zap.S().Errorf("Failed resctrl mount: %v", err.Error())
		h.resctrl = false
	} else {
		h.resctrl = true
	}
	h.signals = make(map[string]float64)
	h.pcm, err = pcm.Init()
	if err != nil {
		return h, err
	}
	h.membw, err = membw.Init(h.pcm)
	if err != nil {
		return h, err
	}
	h.ipc, err = ipc.Init(h.pcm)
	if err != nil {
		return h, err
	}

	return h, nil
}

func (ha *HostAPI) Close() error {
	return ha.pcm.Close()
}

func (ha *HostAPI) Update(hostStats *HostStats) error {
	msg := ""

	err := ha.updateHostStats(hostStats)
	if err != nil {
		msg += err.Error()
	}

	if ha.pcm != nil {
		if ha.membw != nil {
			err = ha.updateMemoryStats()
			if err != nil {
				msg += err.Error()
			}
		}
		if ha.ipc != nil {
			err = ha.updateCPUStats()
			if err != nil {
				msg += err.Error()
			}
		}
	}

	if len(msg) != 0 {
		return errors.New(msg)
	}

	return nil
}

func (ha *HostAPI) updateHostStats(HostStats *HostStats) (err error) {

	HostStats.CPUHostCPUWait, err = util.GetCPUWait()
	if err != nil {
		zap.S().Error(err)
	}

	HostStats.NetHostQdiscDrops = util.GetQdiscDrops()

	if ha.resctrl {
		HostStats.MemHostNUMAStats, err = util.GetNUMAStats()
		if err != nil {
			zap.S().Errorf("Failed to collect NUMA stats: %v", err)
		}
	}

	if ha.hoststats.HostStats != nil {
		ha.signals["net_total_sack_reorder_tmmv"] = float64(HostStats.NetTotalSACKReorder - ha.hoststats.HostStats.NetTotalSACKReorder)
		ha.signals["net_total_retrans_segs_tmmv"] = float64(HostStats.NetTotalRetransSegs - ha.hoststats.HostStats.NetTotalRetransSegs)
		ha.signals["net_host_qdisc_drops_tmmv"] = float64(HostStats.NetHostQdiscDrops - ha.hoststats.HostStats.NetHostQdiscDrops)
		ha.signals["cpu_total_cpu_wait_tmmv"] = HostStats.CPUHostCPUWait - ha.hoststats.HostStats.CPUHostCPUWait
		if ha.resctrl {
			for node, vals := range HostStats.MemHostNUMAStats {
				if vals.Local != nil && ha.hoststats.HostStats.MemHostNUMAStats[node].Local != nil {
					ha.signals[fmt.Sprintf("mem_host_numa_%s_local_bw_tmmv", node)] = float64(Delta(*vals.Local, *ha.hoststats.HostStats.MemHostNUMAStats[node].Local))
				}
				if vals.Total != nil && ha.hoststats.HostStats.MemHostNUMAStats[node].Total != nil {
					ha.signals[fmt.Sprintf("mem_host_numa_%s_total_bw_tmmv", node)] = float64(Delta(*vals.Total, *ha.hoststats.HostStats.MemHostNUMAStats[node].Total))
				}
			}
		}

	}
	ha.hoststats.HostStats = HostStats
	return nil
}

func (ha *HostAPI) updateMemoryStats() error {
	err := ha.membw.Update(true)
	if err != nil {
		return err
	}
	ha.stats.MemoryStats.Bandwith = ha.membw.Data().Overall()
	zap.S().Debugf("Mem BW Overall: %f", ha.stats.MemoryStats.Bandwith)
	return nil
}

func (ha *HostAPI) updateCPUStats() error {
	err := ha.ipc.Update(true)
	if err != nil {
		return err
	}
	ha.stats.CPUStats.IPC = ha.ipc.Values()
	zap.S().Debugf("CPU IPC Values: %v", ha.stats.CPUStats.IPC)
	return nil
}

func (ha *HostAPI) formSignals() []yasm.Signal {
	signals := []yasm.Signal{}

	for name, val := range ha.signals {
		sig := yasm.Signal{
			Name: name,
			Val:  val,
		}
		signals = append(signals, sig)
	}

	memBw := yasm.Signal{
		Name: "memory_bw_mb_tmmv",
		Val:  ha.stats.MemoryStats.Bandwith,
	}
	signals = append(signals, memBw)

	// Push only 50 perc for now
	ipcPerc50 := yasm.Signal{
		Name: "cpu_ipc_perc_50_tmmv",
		Val:  ha.stats.CPUStats.IPC.Percentile(50),
	}
	signals = append(signals, ipcPerc50)
	return signals
}

func (ha *HostAPI) YasmPush(ttl uint) error {
	tags := make(yasm.TagsMap)
	tags["itype"] = "rtcsys"
	tags["prj"] = "portostatd"

	signals := ha.formSignals()
	r := yasm.NewReq(ttl, tags, signals...)
	err := r.Push(yasm.URL)
	if err != nil {
		return err
	}
	return nil
}
