package server

import (
	"context"
	"fmt"
	"syscall"

	node_pb "a.yandex-team.ru/infra/node_agent/go/proto/api"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
	porto_api "a.yandex-team.ru/infra/porto/proto"

	"go.uber.org/zap"
)

func extractCapacityStats(pvolumes *node_pb.TPortoVolumesMonitoringInfo, portoVolumes []*porto_api.TVolumeDescription) (map[string]*internal.CapStats, error) {
	res := make(map[string]*internal.CapStats)
	if len(portoVolumes) == 0 {
		return res, fmt.Errorf("portoVolumes is empty")
	}
	if pvolumes == nil {
		return res, fmt.Errorf("pvolumes is nil")
	}
	ctNamePVolumes := make(map[string][]*node_pb.TPortoVolumeMonitoringInfo)
	for _, pVolPod := range pvolumes.Pods {
		ctNamePVolumes[pVolPod.Container] = pVolPod.Volumes
	}

	for ctName, volumes := range ctNamePVolumes {
		volStats := []*internal.VolCapStat{}
		for _, volume := range volumes {
			var statfs syscall.Statfs_t
			if err := syscall.Statfs(volume.Id, &statfs); err != nil {
				// ENOENT if volume destroyed
				if err != syscall.ENOENT {
					zap.S().Errorf("Failed statfs('%v'): %v", volume.Id, err)
				}
				continue
			}

			v := &internal.VolCapStat{}

			if volume.VolumeType == node_pb.TPortoVolumeMonitoringInfo_CWD {
				v.VolName = "cwd"
			} else if volume.VolumeType == node_pb.TPortoVolumeMonitoringInfo_ROOT_FS {
				v.VolName = "root_fs"
			} else {
				v.VolName = volume.MountPoint
			}

			zap.S().Debugf("ctName %s VolName: %s", ctName, v.VolName)

			if statfs.Blocks == 0 {
				v.PercUsage = 0
			} else {
				v.PercUsage = uint64((float64(statfs.Blocks-statfs.Bfree) / float64(statfs.Blocks)) * 100)
			}
			volStats = append(volStats, v)

		}
		res[ctName] = &internal.CapStats{
			VolCapStats: volStats,
		}
	}
	for ctn, r := range res {
		zap.S().Debugf("ctName: %s", ctn)
		for _, v := range r.VolCapStats {
			zap.S().Debugf("stat: %#v", *v)
		}
	}
	return res, nil
}

func (s *PortostatdServer) GetCapacityStats(ctx context.Context, req *rpcpb.GetCapacityStatRequest) (*rpcpb.GetCapacityStatsResponse, error) {
	return internal.GetCapacityStatsStorage(req.CtName)
}
