package server

import (
	"context"
	"strconv"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

var CPUStatsProps = []string{
	"cpu_usage",
	"cpu_usage_system",
	"cpu_guarantee",
	"cpu_wait",
	"cpu_limit",
	"thread_count",
	"cpu.stat",
	"cpuacct.stat",
}

func parseCoresProp(ctProps map[string]string, prop string) (float64, error) {
	val := ctProps[prop]
	if len(val) == 0 {
		return 0.0, nil
	}
	if val[len(val)-1] == 'c' {
		val = val[:len(val)-1]
	}
	cores, err := strconv.ParseFloat(val, 64)
	if err != nil {
		return 0, status.Errorf(codes.Internal, "Can not convert to float string value of '%v' property: %v", prop, err)
	}
	return cores, nil
}

func extractCPUStats(ctProps map[string]string) (*internal.CPUStats, error) {
	cpuUsage, err := parseUintProp(ctProps, "cpu_usage")
	if err != nil {
		return nil, err
	}

	cpuUsageSystem, err := parseUintProp(ctProps, "cpu_usage_system")
	if err != nil {
		return nil, err
	}

	cpuWait, err := parseUintProp(ctProps, "cpu_wait")
	if err != nil {
		return nil, err
	}

	cpuGuarantee, err := parseCoresProp(ctProps, "cpu_guarantee")
	if err != nil {
		return nil, err
	}

	cpuLimit, err := parseCoresProp(ctProps, "cpu_limit")
	if err != nil {
		return nil, err
	}

	tc := ctProps["thread_count"]
	threadCount, err := strconv.ParseUint(tc, 10, 64)
	if err != nil {
		return nil, err
	}

	cpuStat, err := parseStat(ctProps["cpu.stat"], "\n", " ")
	if err != nil {
		return nil, err
	}

	cpuBurstUsage := cpuStat["burst_usage"]
	cpuBurstLoad := cpuStat["burst_load"]
	throttled := cpuStat["h_throttled_time"]

	cpuAcctStat, err := parseStat(ctProps["cpuacct.stat"], "\n", " ")
	if err != nil {
		return nil, err
	}

	instructions := cpuAcctStat["ipc.instructions"]
	cycles := cpuAcctStat["ipc.cycles"]

	return &internal.CPUStats{
		Usage:          cpuUsage,
		UsageSystem:    cpuUsageSystem,
		Wait:           cpuWait,
		BurstUsage:     cpuBurstUsage,
		BurstLoad:      cpuBurstLoad,
		Throttled:      throttled,
		GuaranteeCores: cpuGuarantee,
		LimitCores:     cpuLimit,
		ThreadCount:    threadCount,
		Instructions:   instructions,
		Cycles:         cycles,
	}, nil
}

func doGetCPUStatsCached(req *rpcpb.GetCPUStatRequest) (*rpcpb.GetCPUStatsResponse, error) {
	return internal.GetCPUStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetCPUStats(ctx context.Context, req *rpcpb.GetCPUStatRequest) (*rpcpb.GetCPUStatsResponse, error) {
	return doGetCPUStatsCached(req)
}

func getCPUStatCached(req *rpcpb.GetCPUStatRequest, cpuStatName string) (*rpcpb.GetCPUStatResponse, error) {
	cpuStats, err := doGetCPUStatsCached(req)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Can not get 'cpu_stats' of '%v' container: %v", req.CtName, err)
	}

	switch cpuStatName {
	case "cpu_usage":
		return &rpcpb.GetCPUStatResponse{Val: cpuStats.CpuUsage}, nil
	case "cpu_usage_system":
		return &rpcpb.GetCPUStatResponse{Val: cpuStats.CpuUsageSystem}, nil
	}

	return nil, status.Errorf(codes.Internal, "Invalid property '%v' for '%v' container: %v", cpuStatName, req.CtName, err)
}

func (s *PortostatdServer) GetCPUUsage(ctx context.Context, req *rpcpb.GetCPUStatRequest) (*rpcpb.GetCPUStatResponse, error) {
	return getCPUStatCached(req, "cpu_usage")
}

func (s *PortostatdServer) GetCPUUsageSystem(ctx context.Context, req *rpcpb.GetCPUStatRequest) (*rpcpb.GetCPUStatResponse, error) {
	return getCPUStatCached(req, "cpu_usage_system")
}
