package server

import (
	"context"
	"regexp"
	"strings"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"

	nvgpu_api "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
)

func extractNvidiaPaths(rspDevStr string) ([]string, error) {
	// 'rspDevStr' format is like:
	//     <device> [r][w][m][-][?] [path] [mode] [user] [group]|preset <preset>; ...
	//     /dev/nvidia0 rwm /dev/nvidia0 0666 root root; ...

	// if container doesn't have any Gpus - return rsp with empty array
	// if !strings.Contains(rspDevStr, "nvidia") {
	re, err := regexp.Compile(`nvidia\d`)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Regexp compilation failed: %v", err)
	}

	// split 'rspDevStr' to array of nvidia paths
	var nvidiaPaths []string

	rspDevArr := strings.Split(rspDevStr, "; ")

	for i := range rspDevArr {
		if re.FindString(rspDevArr[i]) == "" { // substring not found
			continue
		}

		devValArr := strings.Split(rspDevArr[i], " ")
		if len(devValArr) < 3 {
			return nil, status.Errorf(codes.Internal, "Can not parse %v (part of %v) for gpu_path: unknown format", rspDevArr[i], rspDevStr)
		}

		nvidiaPaths = append(nvidiaPaths, devValArr[2])
	}

	return nvidiaPaths, err
}

func getGpuStats(nvidiaPaths []string) (*internal.GpuStats, error) {
	var temperature []internal.GpuStat
	var utilization []internal.GpuStat
	var memoryUtilization []internal.GpuStat

	// as container does have some gpus - ask nvgpu-manager for info about gpus
	rspListDev, err := nvgpuClient.ListDevices(context.Background(), &nvgpu_api.Empty{})
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Failed to call nvgpu's ListDevices(): %v", err)
	}

	// try to find info about gpus from container's devices in response from nvgpu-manager's ListDevices()
	for _, nvidiaPath := range nvidiaPaths {
		isFound := false

		for _, device := range rspListDev.Devices {
			specDriver := device.Spec.GetNvidia()
			statusDriver := device.Status.GetNvidia()

			if specDriver != nil && statusDriver != nil {
				if nvidiaPath != specDriver.DevicePath {
					continue
				} else {
					temperature = append(temperature, internal.GpuStat{
						Dev: nvidiaPath,
						Val: uint64(statusDriver.Temperature)})
					utilization = append(utilization, internal.GpuStat{
						Dev: nvidiaPath,
						Val: uint64(statusDriver.GpuUtilization)})
					memoryUtilization = append(memoryUtilization, internal.GpuStat{
						Dev: nvidiaPath,
						Val: uint64(statusDriver.MemoryUtilization)})

					isFound = true

					break
				}
			} else {
				continue
			}
		}

		if !isFound {
			return nil, status.Errorf(codes.Internal, "Error: nvgpu-manager doesn't know about gpu '%v'", nvidiaPath)
		}
	}

	return &internal.GpuStats{
		Temperature:       temperature,
		Utilization:       utilization,
		MemoryUtilization: memoryUtilization,
	}, nil
}

func doGetGpuStatsCached(req *rpcpb.GetGpuStatRequest) (*rpcpb.GetGpuStatsResponse, error) {
	return internal.GetGpuStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetGpuStats(ctx context.Context, req *rpcpb.GetGpuStatRequest) (*rpcpb.GetGpuStatsResponse, error) {
	return doGetGpuStatsCached(req)
}

func getGpuStatCached(req *rpcpb.GetGpuStatRequest, gpuStatName string) (*rpcpb.GetGpuStatResponse, error) {
	gpuStats, err := doGetGpuStatsCached(req)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Can not get 'gpu_stats' of '%v' container: %v", req.CtName, err)
	}

	switch gpuStatName {
	case "gpu_temperature":
		return gpuStats.GpuTemperature, nil
	case "gpu_utilization":
		return gpuStats.GpuUtilization, nil
	case "gpu_memory_utilization":
		return gpuStats.GpuMemoryUtilization, nil
	}

	return nil, status.Errorf(codes.Internal, "Invalid property '%v' for '%v' container: %v", gpuStatName, req.CtName, err)
}

func (s *PortostatdServer) GetGpuTemperature(ctx context.Context, req *rpcpb.GetGpuStatRequest) (*rpcpb.GetGpuStatResponse, error) {
	return getGpuStatCached(req, "gpu_temperature")
}

func (s *PortostatdServer) GetGpuUtilization(ctx context.Context, req *rpcpb.GetGpuStatRequest) (*rpcpb.GetGpuStatResponse, error) {
	return getGpuStatCached(req, "gpu_utilization")
}

func (s *PortostatdServer) GetGpuMemoryUtilization(ctx context.Context, req *rpcpb.GetGpuStatRequest) (*rpcpb.GetGpuStatResponse, error) {
	return getGpuStatCached(req, "gpu_memory_utilization")
}
