package server

import (
	"context"
	"fmt"
	"path"

	"go.uber.org/zap"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal/cgroups/blkio"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/pkg/diskstat"
	"a.yandex-team.ru/infra/porto/plugins/portostatd/pkg/iss"

	diskman_api "a.yandex-team.ru/infra/diskmanager/proto"
	node_pb "a.yandex-team.ru/infra/node_agent/go/proto/api"

	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

const (
	cgroupBlkIOBasePath = "/sys/fs/cgroup/blkio"
)

// { ContainerName: []*TDiskAllocationDescription, }
type issCtAllocations map[string][]*node_pb.TDiskAllocationDescription

type CgroupAnswer struct {
	ReadOps    uint64
	WriteOps   uint64
	ReadBytes  uint64
	WriteBytes uint64
}

func formCgroupAnswer(ctName string, device blkio.Device) (*CgroupAnswer, error) {
	res := &CgroupAnswer{}
	cgroupBaseName := fmt.Sprintf("porto%%%s", ctName)
	opsPath := path.Join(cgroupBlkIOBasePath, cgroupBaseName, "blkio.throttle.io_serviced")
	bwPath := path.Join(cgroupBlkIOBasePath, cgroupBaseName, "blkio.throttle.io_service_bytes")

	ops, err := blkio.FromFile(opsPath)
	if err != nil {
		return nil, err
	}
	res.ReadOps = ops[device]["Read"]
	res.WriteOps = ops[device]["Write"]

	bw, err := blkio.FromFile(bwPath)
	if err != nil {
		return nil, err
	}
	res.ReadBytes = bw[device]["Read"]
	res.WriteBytes = bw[device]["Write"]
	return res, nil
}

func formIssCtAllocations() (issCtAllocations, error) {
	issAnsw, err := iss.IssAllocations()
	if err != nil {
		return nil, err
	}
	ctAllocations := make(issCtAllocations)
	for _, monInfo := range issAnsw.Pods {
		ctAllocations[monInfo.Container] = []*node_pb.TDiskAllocationDescription{}
		for _, a := range monInfo.Allocations {
			ctAllocations[monInfo.Container] = append(ctAllocations[monInfo.Container], a)
		}
	}
	return ctAllocations, nil
}

func ExtractAllocationsIO(ctName string, allocs issCtAllocations, diskmanClient diskman_api.DiskManagerClient) internal.IOAllocStatsMap {
	res := make(internal.IOAllocStatsMap)
	var answ *internal.IOAllocStats
	var err error

	procDiskStat, err := diskstat.ReadSystem()
	if err != nil {
		zap.S().Errorf("could not read /proc/diskstats, %s", err)
		return res
	}

	for _, a := range allocs[ctName] {
		switch a.Backend {
		case node_pb.TDiskAllocationDescription_DISKMAN:
			answ, err = formDiskmanAnswer(ctName, a, diskmanClient, procDiskStat)
			if err != nil {
				zap.S().Error(err)
				continue
			}
			zap.S().Debugf("IO ALLOC %s, %#v", ctName, answ)
			res[answ.AllocName] = answ
		case node_pb.TDiskAllocationDescription_PORTO:
			answ, err = formPortoAnswer(a, ctName)
			if err != nil {
				zap.S().Error(err)
				continue
			}
			zap.S().Debugf("IO ALLOC %s, %#v", ctName, answ)
			if iostat, ok := res[answ.AllocName]; ok {
				iostat.Add(answ)
			} else {
				res[answ.AllocName] = answ
			}
		default:
			zap.S().Debug("unknown type of backend for allocation: %s", a)
			continue
		}
	}
	return res
}

func formDiskmanAnswer(ctName string, allocation *node_pb.TDiskAllocationDescription, diskmanClient diskman_api.DiskManagerClient, procDiskStat diskstat.DiskStatMap) (*internal.IOAllocStats, error) {
	res := &internal.IOAllocStats{
		AllocName:     allocation.Alias,
		ReadBwLimit:   allocation.ReadBandwidthLimit,
		WriteBwLimit:  allocation.WriteBandwidthLimit,
		ReadOpsLimit:  allocation.ReadOperationRateLimit,
		WriteOpsLimit: allocation.WriteOperationRateLimit,
	}

	zap.S().Debugf("Forming Diskman Answer for %s", allocation.Storage)
	dResp, err := diskmanClient.ListDisks(
		context.Background(),
		&diskman_api.ListDisksRequest{
			DiskIds: []string{allocation.Storage},
		},
	)
	if err != nil {
		return nil, err
	}
	if len(dResp.Disks) != 1 {
		return nil, fmt.Errorf("multiple volume ids for %s, %v", allocation.Storage, dResp.Disks)
	}
	device := blkio.Device{
		Major: dResp.Disks[0].Spec.Major,
		Minor: dResp.Disks[0].Spec.Minor,
	}

	cgAnsw, err := formCgroupAnswer(ctName, device)
	if err != nil {
		return nil, fmt.Errorf("%w", err)
	}
	res.ReadOps = cgAnsw.ReadOps
	res.WriteOps = cgAnsw.WriteOps
	res.ReadBytes = cgAnsw.ReadBytes
	res.WriteBytes = cgAnsw.WriteBytes
	return res, nil
}

func formPortoAnswer(allocation *node_pb.TDiskAllocationDescription, ctName string) (*internal.IOAllocStats, error) {
	res := &internal.IOAllocStats{
		ReadBwLimit:   allocation.ReadBandwidthLimit,
		WriteBwLimit:  allocation.WriteBandwidthLimit,
		ReadOpsLimit:  allocation.ReadOperationRateLimit,
		WriteOpsLimit: allocation.WriteOperationRateLimit,
	}
	zap.S().Debugf("Froming Porto Answer for %s", allocation.Storage)
	volInfo, err := diskmanClient.ListVolumes(
		context.Background(),
		&diskman_api.ListVolumesRequest{MountPaths: []string{allocation.Storage}},
	)
	if err != nil {
		return nil, err
	}
	if len(volInfo.Volumes) != 1 {
		return nil, fmt.Errorf("multiple volume ids for %s, %v", allocation.Storage, volInfo.Volumes)
	}
	res.AllocName = volInfo.Volumes[0].Spec.StorageClass
	device := blkio.Device{
		Major: volInfo.Volumes[0].Spec.Major,
		Minor: volInfo.Volumes[0].Spec.Minor,
	}

	cgAnsw, err := formCgroupAnswer(ctName, device)
	if err != nil {
		return nil, fmt.Errorf("%w", err)
	}
	res.ReadOps = cgAnsw.ReadOps
	res.WriteOps = cgAnsw.WriteOps
	res.ReadBytes = cgAnsw.ReadBytes
	res.WriteBytes = cgAnsw.WriteBytes

	return res, nil
}

func doGetIOAllocStatsCached(req *rpcpb.GetIOAllocStatRequest) (*rpcpb.GetIOAllocStatsResponse, error) {
	return internal.GetIOAllocStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetIOAllocStats(ctx context.Context, req *rpcpb.GetIOAllocStatRequest) (*rpcpb.GetIOAllocStatsResponse, error) {
	return doGetIOAllocStatsCached(req)
}
