package server

import (
	"context"
	"strconv"
	"strings"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

var IoStatsProps = []string{
	"io_read",
	"io_write",
	"io_ops",
	"io_time",
}

func parseIoStatProp(ctProps map[string]string, prop string) ([]internal.IoStat, error) {
	rspStr := ctProps[prop]
	if len(rspStr) == 0 {
		return nil, nil
	}

	// 'rspStr' format is like "fs: 942080; hw: 942080; vda: 942080"
	rspArr := strings.Split(rspStr, "; ")
	stat := make([]internal.IoStat, 0, len(rspArr))
	for i := range rspArr {
		devValArr := strings.Split(rspArr[i], ": ")

		if len(devValArr) != 2 {
			return nil, status.Errorf(codes.Internal, "Can not parse '%v' (part of '%v') for '%v' property: unknown format", rspArr[i], rspStr, prop)
		}

		uintVal, err := strconv.ParseUint(devValArr[1], 10, 64)
		if err != nil {
			return nil, status.Errorf(codes.Internal, "Can not convert to uint string value of '%v' for '%v' property: %v", devValArr[0], prop, err)
		}

		devValStruct := internal.IoStat{Dev: devValArr[0], Val: uintVal}
		stat = append(stat, devValStruct)
	}

	return stat, nil
}

func extractIoStats(ctProps map[string]string) (*internal.IoStats, error) {
	var ioStats internal.IoStats
	stats := []struct {
		name  string
		stats *[]internal.IoStat
	}{
		{"io_read", &ioStats.Read},
		{"io_write", &ioStats.Write},
		{"io_ops", &ioStats.Ops},
		{"io_time", &ioStats.Time},
	}

	for _, s := range stats {
		rsp, err := parseIoStatProp(ctProps, s.name)
		if err != nil {
			return nil, err
		}

		*s.stats = rsp
	}

	return &ioStats, nil
}

func doGetIoStatsCached(req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatsResponse, error) {
	return internal.GetIoStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetIoStats(ctx context.Context, req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatsResponse, error) {
	return doGetIoStatsCached(req)
}

func getIoStatCached(req *rpcpb.GetIoStatRequest, ioStatName string) (*rpcpb.GetIoStatResponse, error) {
	ioStats, err := doGetIoStatsCached(req)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Can not get 'io_stats' of '%v' container: %v", req.CtName, err)
	}

	switch ioStatName {
	case "io_read":
		return ioStats.IoRead, nil
	case "io_write":
		return ioStats.IoWrite, nil
	case "io_ops":
		return ioStats.IoOps, nil
	case "io_time":
		return ioStats.IoTime, nil
	}

	return nil, status.Errorf(codes.Internal, "Invalid property '%v' for '%v' container: %v", ioStatName, req.CtName, err)
}

func (s *PortostatdServer) GetIoRead(ctx context.Context, req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatResponse, error) {
	return getIoStatCached(req, "io_read")
}

func (s *PortostatdServer) GetIoWrite(ctx context.Context, req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatResponse, error) {
	return getIoStatCached(req, "io_write")
}

func (s *PortostatdServer) GetIoOps(ctx context.Context, req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatResponse, error) {
	return getIoStatCached(req, "io_ops")
}

func (s *PortostatdServer) GetIoTime(ctx context.Context, req *rpcpb.GetIoStatRequest) (*rpcpb.GetIoStatResponse, error) {
	return getIoStatCached(req, "io_time")
}
