package server

import (
	"context"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

var MemoryStatsProps = []string{
	"memory_usage",
	"memory_guarantee",
	"memory_limit",
	"memory.stat",
}

func extractMemoryStats(ctProps map[string]string) (*internal.MemoryStats, error) {
	memUsage, err := parseUintProp(ctProps, "memory_usage")
	if err != nil {
		return nil, err
	}

	memGuarantee, err := parseUintProp(ctProps, "memory_guarantee")
	if err != nil {
		return nil, err
	}

	memLimit, err := parseUintProp(ctProps, "memory_limit")
	if err != nil {
		return nil, err
	}

	statStr := ctProps["memory.stat"]
	memStat, err := parseMemStat(statStr)
	if err != nil {
		return nil, err
	}

	return &internal.MemoryStats{
		Usage:     memUsage,
		Guarantee: memGuarantee,
		Limit:     memLimit,
		Stat:      memStat,
	}, nil
}

func doGetMemoryStatsCached(req *rpcpb.GetMemoryStatRequest) (*rpcpb.GetMemoryStatsResponse, error) {
	return internal.GetMemoryStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetMemoryStats(ctx context.Context, req *rpcpb.GetMemoryStatRequest) (*rpcpb.GetMemoryStatsResponse, error) {
	return doGetMemoryStatsCached(req)
}

func getMemoryStatCached(req *rpcpb.GetMemoryStatRequest, memoryStatName string) (*rpcpb.GetMemoryStatResponse, error) {
	memoryStats, err := doGetMemoryStatsCached(req)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Can not get 'memory_stats' of '%v' container: %v", req.CtName, err)
	}

	switch memoryStatName {
	case "memory_usage":
		return &rpcpb.GetMemoryStatResponse{Val: memoryStats.MemoryUsage}, nil
	}

	return nil, status.Errorf(codes.Internal, "Invalid property '%v' for '%v' container: %v", memoryStatName, req.CtName, err)
}

func (s *PortostatdServer) GetMemoryUsage(ctx context.Context, req *rpcpb.GetMemoryStatRequest) (*rpcpb.GetMemoryStatResponse, error) {
	return getMemoryStatCached(req, "memory_usage")
}
