package server

import (
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"reflect"
	"strconv"
	"strings"

	"go.uber.org/zap"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

type hgramBucket struct {
	leftEdge uint32
	count    uint64
}

const backboneGroupID = 1

var (
	NetStatsProps = []string{
		"net_rx_bytes",
		"net_tx_bytes",
		"net_rx_packets",
		"net_tx_packets",
		"net_rx_drops",
		"net_tx_drops",
		"net_rx_limit",
		"net_limit",
		"net_overlimits",
		"net_rx_overlimits",
		"net_netstat",
		"net_snmp",
		"net_rx_speed_hgram",
		"net_tx_speed_hgram",
	}
	//This is a list of interfaces, that are ignored when parsing tunnel metrics
	//They are ignored since they often exist and are not valuable, and Uplink is parsed separately
	//For more info: https://bb.yandex-team.ru/projects/PORTO/repos/porto/browse/src/property.cpp#5740
	IgnoredInterfaces = map[string]struct{}{
		"Uplink":   struct{}{},
		"Latency":  struct{}{},
		"SockDiag": struct{}{},
	}
	//This is a list of patterns, that are ignored when parsing tunnel metrics
	IgnoredPatterns = []string{
		"group ",
	}

	prevNetTxSpeedHgram map[string][]hgramBucket = make(map[string][]hgramBucket)
	prevNetRxSpeedHgram map[string][]hgramBucket = make(map[string][]hgramBucket)
)

func getCtNetStatsMap(ctProps map[string]string, propertyName string) (map[string]uint64, error) {
	netStat := ctProps[propertyName]
	netStatMap, err := parseNetStat(netStat)
	if err != nil {
		return nil, err
	}
	return netStatMap, nil
}

func checkIface(ifacename string) bool {
	if _, found := IgnoredInterfaces[ifacename]; found {
		return false
	}
	for _, pattern := range IgnoredPatterns {
		if strings.Contains(ifacename, pattern) {
			return false
		}
	}
	return true
}

func setInterfaceStat(fieldName string, propMap map[string]uint64, ifaceStatMap map[string]*internal.NetInterfaceStats) {
	for key, value := range propMap {
		if checkIface(key) {
			stat := ifaceStatMap[key]
			if stat == nil {
				stat = new(internal.NetInterfaceStats)
			}
			statValue := reflect.ValueOf(stat)
			statValue.Elem().FieldByName(fieldName).SetUint(value)
			ifaceStatMap[key] = stat
		}
	}
}

func getIfaceStatsMap(ctProps map[string]string) (map[string]*internal.NetInterfaceStats, error) {
	// Gets these properties:
	//		"net_rx_bytes"
	//		"net_tx_bytes"
	//		"net_rx_packets"
	//		"net_tx_packets"
	//		"net_tx_drops"
	//		"net_rx_drops"
	// For all network interfaces, ignoring some of them.
	// Creates a map where key is interface name, and value is a struct with all the property values for a corresponding interface

	ifaceStatMap := make(map[string]*internal.NetInterfaceStats)

	properties := []struct {
		src, dst string
	}{
		{"net_rx_bytes", "RxBytes"},
		{"net_tx_bytes", "TxBytes"},
		{"net_tx_packets", "TxPackets"},
		{"net_rx_packets", "RxPackets"},
		{"net_tx_drops", "TxDrops"},
		{"net_rx_drops", "RxDrops"},
	}
	for _, p := range properties {
		statMap, err := getCtNetStatsMap(ctProps, p.src)
		if err != nil {
			return nil, err
		}
		setInterfaceStat(p.dst, statMap, ifaceStatMap)
	}

	return ifaceStatMap, nil
}

func parseNetStatProp(ctProps map[string]string, prop string) (uint64, error) {
	var err error

	rspStr := ctProps[prop]
	// 'rspStr' format is like one of these:
	//     "Total: 0"
	//     "Latency: 3686; Uplink: 12"
	//     "Uplink: 1018; group default: 1799; ip6tnl0: 781; tun0: 0B; veth: 1018"
	// this function only parses net_limit, net_rx_limit, and parses only "Uplink" interface

	if len(rspStr) == 0 {
		return 0, status.Errorf(codes.Internal, "'%v' property is empty", prop)
	}

	if prop == "net_limit" || prop == "net_rx_limit" {
		split := strings.Split(rspStr, ": ")
		if len(split) != 2 {
			return 0, status.Errorf(codes.Internal, "Can not parse %v for '%v' property: unknown format", rspStr, prop)
		}
		limit, err := strconv.ParseUint(split[1], 10, 64)
		if err != nil {
			return 0, status.Errorf(codes.Internal, "Can not convert to uint string value of '%v' property: %v", prop, err)
		}
		return limit, nil
	}

	rspArr := strings.Split(rspStr, "; ")
	uplink := uint64(0)

	for i := range rspArr {
		rspKeyVal := strings.Split(rspArr[i], ": ")

		if rspKeyVal[0] == "Uplink" {
			if len(rspKeyVal) != 2 {
				return 0, status.Errorf(codes.Internal, "Can not parse %v (part of %v) for '%v' property: unknown format", rspArr[i], rspStr, prop)
			}

			uplink, err = strconv.ParseUint(rspKeyVal[1], 10, 64)
			if err != nil {
				return 0, status.Errorf(codes.Internal, "Can not convert to uint string value of 'Uplink' for '%v' property: %v", prop, err)
			}
			return uplink, nil
		}
	}

	return 0, status.Errorf(codes.Internal, "No Uplink found for '%v' property", prop)
}

func getHgramDiff(prev []hgramBucket, curr []hgramBucket) []hgramBucket {

	if len(prev) != len(curr) {
		return nil
	}

	diff := make([]hgramBucket, len(prev))

	for i, prevBucket := range prev {
		currBucket := curr[i]

		if prevBucket.leftEdge != currBucket.leftEdge || prevBucket.count > currBucket.count {
			return nil
		}

		diff[i] = hgramBucket{leftEdge: currBucket.leftEdge, count: currBucket.count - prevBucket.count}
	}

	return diff
}

func parseNetStatPropHgram(ctName string, ctProps map[string]string, prop string) (*rpcpb.GetNetStatHgramResponse, error) {
	var err error
	currStat := make([]hgramBucket, 0)

	// prop format is like '0:51;10:1;13:15....'
	propValue := ctProps[prop]
	if len(propValue) == 0 {
		return nil, nil
	}

	buckets := strings.Split(propValue, ";")

	leftEdge := uint64(0)
	count := uint64(0)

	for _, bucket := range buckets {
		keyVal := strings.Split(bucket, ":")
		if len(keyVal) != 2 {
			return nil, status.Errorf(codes.Internal, "Unknown hgram bucket format: '%s'", bucket)
		}

		leftEdge, err = strconv.ParseUint(keyVal[0], 10, 32)
		if err != nil {
			return nil, status.Errorf(codes.Internal, "Can not convert to uint string value of '%v' property: %s", prop, bucket)
		}

		count, err = strconv.ParseUint(keyVal[1], 10, 64)
		if err != nil {
			return nil, status.Errorf(codes.Internal, "Can not convert to uint string value of '%v' property: %s", prop, bucket)
		}
		currStat = append(currStat, hgramBucket{leftEdge: uint32(leftEdge), count: count})
	}

	prevStat := make([]hgramBucket, 0)

	if prop == "net_tx_speed_hgram" {
		prev, ok := prevNetTxSpeedHgram[ctName]
		if ok {
			prevStat = prev
		}
	} else if prop == "net_rx_speed_hgram" {
		prev, ok := prevNetRxSpeedHgram[ctName]
		if ok {
			prevStat = prev
		}
	}

	if prop == "net_tx_speed_hgram" {
		prevNetTxSpeedHgram[ctName] = currStat
	} else if prop == "net_rx_speed_hgram" {
		prevNetRxSpeedHgram[ctName] = currStat
	}

	// diff may be nil on psd or porto restart
	hgramDiff := getHgramDiff(prevStat, currStat)
	if hgramDiff == nil {
		return nil, nil
	}

	bins := make([]*rpcpb.HistogramBin, len(hgramDiff))

	for i, bucket := range hgramDiff {
		bins[i] = &rpcpb.HistogramBin{
			LeftEdge: float64(bucket.leftEdge),
			Count:    bucket.count,
		}
	}

	return &rpcpb.GetNetStatHgramResponse{Bins: bins}, nil
}

func ifIsPhy(ifname string) bool {
	_, err := os.Stat(fmt.Sprintf("/sys/class/net/%s/device/vendor", ifname))
	return err == nil
}

func ifIsBackbone(ifname string) bool {
	data, err := ioutil.ReadFile(fmt.Sprintf("/sys/class/net/%s/netdev_group", ifname))
	if err != nil {
		return false
	}
	group, err := strconv.ParseUint(strings.TrimSpace(string(data)), 10, 64)
	return err == nil && group == backboneGroupID
}

func ifIsUp(ifname string) bool {
	data, err := ioutil.ReadFile(fmt.Sprintf("/sys/class/net/%s/operstate", ifname))
	return err == nil && strings.TrimSpace(string(data)) == "up"
}

func readDirnames(path string) ([]string, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	defer f.Close()
	return f.Readdirnames(-1)
}

func backboneIfname() (string, error) {
	ifnames, err := readDirnames("/sys/class/net")
	if err != nil {
		return "", err
	}
	backboneIfname := ""
	for _, ifname := range ifnames {
		if !ifIsPhy(ifname) {
			continue
		}
		if !ifIsBackbone(ifname) {
			continue
		}
		if !ifIsUp(ifname) {
			continue
		}
		if len(backboneIfname) > 0 {
			return "", errors.New("more than one backbone interface found")
		}
		backboneIfname = ifname
	}
	if len(backboneIfname) == 0 {
		return "", errors.New("no backbone iface found")
	}
	return backboneIfname, nil
}

// backboneIfSpeed returns interfaces speed in bytes
func backboneIfSpeed() (uint64, error) {
	ifname := internal.BackboneIfname

	data, err := ioutil.ReadFile(fmt.Sprintf("/sys/class/net/%s/speed", ifname))
	if err != nil {
		return 0, fmt.Errorf("failed to read iface speed: %w", err)
	}
	speed, err := strconv.ParseUint(strings.TrimSpace(string(data)), 10, 64)
	if err != nil {
		return 0, fmt.Errorf("failed to parse iface speed: %w", err)
	}
	return speed * 1000000 / 8, nil
}

func extractNetStats(ctProps map[string]string, ctName string) (resp *internal.NetStats, err error) {
	resp = &internal.NetStats{}

	resp.NetStat, err = getCtNetStatsMap(ctProps, "net_netstat")
	if err != nil {
		return nil, err
	}

	resp.NetSnmp, err = getCtNetStatsMap(ctProps, "net_snmp")
	if err != nil {
		return nil, err
	}

	resp.UplinkRxBytes, err = parseNetStatProp(ctProps, "net_rx_bytes")
	if err != nil {
		return nil, err
	}

	resp.UplinkTxBytes, err = parseNetStatProp(ctProps, "net_tx_bytes")
	if err != nil {
		return nil, err
	}

	uplinkRxOverlimits, err := parseNetStatProp(ctProps, "net_rx_overlimits")
	if err == nil {
		resp.UplinkRxOverlimits = &uplinkRxOverlimits
	}

	uplinkTxOverlimits, err := parseNetStatProp(ctProps, "net_overlimits")
	if err == nil {
		resp.UplinkTxOverlimits = &uplinkTxOverlimits
	}

	resp.IfaceStat, err = getIfaceStatsMap(ctProps)
	if err != nil {
		return nil, err
	}

	resp.UplinkRxLimit, err = parseNetStatProp(ctProps, "net_rx_limit")
	if err != nil {
		zap.S().Debugf("Could not parse net_rx_limit, so set limit to ifSpeed=%d, err: %s", ifSpeed, err)
		resp.UplinkRxLimit = ifSpeed
	}

	resp.UplinkTxLimit, err = parseNetStatProp(ctProps, "net_limit")
	if err != nil {
		zap.S().Debugf("Could not parse net_limit, so set limit to ifSpeed=%d, err: %s", ifSpeed, err)
		resp.UplinkTxLimit = ifSpeed
	}

	resp.TxSpeedHgram, err = parseNetStatPropHgram(ctName, ctProps, "net_tx_speed_hgram")
	if err != nil {
		return nil, err
	}

	resp.RxSpeedHgram, err = parseNetStatPropHgram(ctName, ctProps, "net_rx_speed_hgram")
	if err != nil {
		return nil, err
	}

	return resp, nil
}

func doGetNetStatsCached(req *rpcpb.GetNetStatRequest) (*rpcpb.GetNetStatsResponse, error) {
	return internal.GetNetStatsStorage(req.CtName)
}

func (s *PortostatdServer) GetNetStats(ctx context.Context, req *rpcpb.GetNetStatRequest) (*rpcpb.GetNetStatsResponse, error) {
	return doGetNetStatsCached(req)
}

func getNetStatCached(req *rpcpb.GetNetStatRequest, netStatName string) (*rpcpb.GetNetStatResponse, error) {
	netStats, err := doGetNetStatsCached(req)
	if err != nil {
		return nil, status.Errorf(codes.Internal, "Can not get 'net_stats' of '%v' container: %v", req.CtName, err)
	}

	switch netStatName {
	case "net_rx_bytes", "net_uplink_rx_bytes":
		return netStats.NetUplinkRxBytes, nil
	case "net_tx_bytes", "net_uplink_tx_bytes":
		return netStats.NetUplinkTxBytes, nil
	}

	return nil, status.Errorf(codes.Internal, "Invalid property '%v' for '%v' container: %v", netStatName, req.CtName, err)
}

func (s *PortostatdServer) GetNetRxBytes(ctx context.Context, req *rpcpb.GetNetStatRequest) (*rpcpb.GetNetStatResponse, error) {
	return getNetStatCached(req, "net_rx_bytes")
}

func (s *PortostatdServer) GetNetTxBytes(ctx context.Context, req *rpcpb.GetNetStatRequest) (*rpcpb.GetNetStatResponse, error) {
	return getNetStatCached(req, "net_tx_bytes")
}
