package server

import (
	"context"
	"fmt"
	"math"
	"strings"

	"a.yandex-team.ru/infra/porto/plugins/portostatd/internal"
	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

const (
	baseBinWidth           = uint64(5)
	tempratureBinWidth     = baseBinWidth
	utilizationBinWidth    = baseBinWidth
	netUtilizationBinWidth = baseBinWidth

	totalWriteback = "total_writeback"
)

type hgram struct {
	bins     map[uint64]uint64
	binWidth uint64
}

func (h *hgram) add(v uint64) {
	// TODO: zero division here, must be fixed
	// creates key with default value 0, and then increments value
	h.bins[v-(v%h.binWidth)]++
}

func (h *hgram) minMax() (uint64, uint64) {
	var (
		min uint64 = math.MaxUint64
		max uint64 = 0
	)
	for leftEdge := range h.bins {
		if leftEdge < min {
			min = leftEdge
		}
		rightEdge := leftEdge + h.binWidth
		if rightEdge > max {
			max = rightEdge
		}
	}
	return min, max
}

func (h *hgram) protoMarshal() []*rpcpb.HistogramBin {
	min, max := h.minMax()
	if min >= max {
		return nil
	}
	bins := make([]*rpcpb.HistogramBin, (max-min)/h.binWidth+1) // +1 cause we need extra empy bin to indicate border
	for i := range bins {
		leftEdge := min + h.binWidth*uint64(i)
		bins[i] = &rpcpb.HistogramBin{
			LeftEdge: float64(leftEdge),
			Count:    uint64(h.bins[leftEdge]),
		}
	}
	return bins
}

func toGB(bytes uint64) float64 {
	return (float64)(bytes) / (1024 * 1024 * 1024)
}

func toCores(ns uint64) float64 {
	return float64(ns) / (1000000000 * 5)
}

func sanitizeVolName(name string) string {
	return strings.Replace(name, "-", "_", -1)
}

func (s *PortostatdServer) GetPortoinstStats(ctx context.Context, req *rpcpb.GetPortoinstStatsRequest) (*rpcpb.GetPortoinstStatsResponse, error) {
	portoinstStats := make([]*rpcpb.GetPortoinstStatsResponse_PortoinstStat, 0, len(req.CtNames))

	for _, ctName := range req.CtNames {
		statsStorage, ok := internal.GetStatsStorage(ctName)
		if ok {
			var (
				stats  []*rpcpb.GetPortoinstStatsResponse_ScalarStat
				hstats []*rpcpb.GetPortoinstStatsResponse_HistogramStat
			)

			cpuStats := statsStorage.CPUStats
			if cpuStats != nil {
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_guarantee_slot_cores_tmmv", Val: cpuStats.CpuGuaranteeCores})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_limit_slot_cores_tmmv", Val: cpuStats.CpuLimitCores})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_usage_slot_tmmv", Val: toCores(cpuStats.CpuUsage)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "thread_count_tmmv", Val: float64(cpuStats.ThreadCount)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "ipc_tmmv", Val: cpuStats.Ipc})
				if cpuHasBurst {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_burst_usage_slot_tmmv", Val: toCores(cpuStats.CpuBurstUsage)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_constrained_wait_slot_tmmv", Val: toCores(cpuStats.CpuConstrainedWait)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_unconstrained_wait_slot_tmmv", Val: toCores(cpuStats.CpuUnconstrainedWait)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "cpu_throttled_slot_tmmv", Val: toCores(cpuStats.CpuThrottled)})
				}
			}

			memStats := statsStorage.MemoryStats
			if memStats != nil {
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_guarantee_slot_gb_tmmv", Val: toGB(memStats.MemoryGuarantee)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_limit_slot_gb_tmmv", Val: toGB(memStats.MemoryLimit)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_usage_slot_tmmv", Val: (float64)(memStats.MemoryUsage)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_total_pgscan_kswapd_slot_tmmv", Val: (float64)(memStats.MemoryTotalPgscanKswapd)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_total_pgscan_direct_slot_tmmv", Val: (float64)(memStats.MemoryTotalPgscanDirect)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_total_written_tmmv", Val: (float64)(memStats.MemoryTotalWritten)})
				stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "memory_total_writeback_tmmv", Val: (float64)(memStats.MemoryTotalWriteback)})
			}

			netStats := statsStorage.NetStats
			if netStats != nil {
				for _, v := range []struct {
					name  string
					value uint64
				}{
					{name: "net_rx_utilization", value: netStats.NetUplinkRxUtilization.Val},
					{name: "net_tx_utilization", value: netStats.NetUplinkTxUtilization.Val},
				} {
					hgram := hgram{bins: map[uint64]uint64{}, binWidth: utilizationBinWidth}
					hgram.add(v.value)
					hstats = append(hstats, &rpcpb.GetPortoinstStatsResponse_HistogramStat{Name: fmt.Sprintf("%s_hgram", v.name), Bins: hgram.protoMarshal()})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_tmmv", v.name), Val: float64(v.value)})
				}
				if netStats.NetUplinkRxOverlimits != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_rx_overlimits_tmmv", Val: float64(netStats.NetUplinkRxOverlimits.Val)})
				}
				if netStats.NetUplinkTxOverlimits != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_tx_overlimits_tmmv", Val: float64(netStats.NetUplinkTxOverlimits.Val)})
				}
				if netStats.NetRetransSegs != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_retrans_segs_tmmv", Val: float64(netStats.NetRetransSegs.Val)})
				}
				if netStats.NetTcpBacklogDrop != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_tcp_backlog_drop_tmmv", Val: float64(netStats.NetTcpBacklogDrop.Val)})
				}
				if netStats.NetTcpSynRetrans != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_tcp_syn_retrans_tmmv", Val: float64(netStats.NetTcpSynRetrans.Val)})
				}
				// if netStats.NetTcpSackReorder != nil {
				// 	stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_tcp_sack_reorder_tmmv", Val: float64(netStats.NetTcpSackReorder.Val)})
				// }
				if netStats.NetCurrEstab != nil {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: "net_curr_estab_tmmv", Val: float64(netStats.NetCurrEstab.Val)})
				}
				for _, s := range netStats.NetInterfaceStats.NetInterfaces {
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_tx_bytes_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.TxBytes)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_rx_bytes_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.RxBytes)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_tx_packets_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.TxPackets)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_rx_packets_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.RxPackets)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_tx_drops_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.TxDrops)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("net_iface_%s_rx_drops_tmmv", strings.ReplaceAll(s.Ifname, " ", "_")), Val: float64(s.RxDrops)})
				}

				if netStats.NetRxSpeedHgram != nil {
					hstats = append(hstats, &rpcpb.GetPortoinstStatsResponse_HistogramStat{Name: "net_rx_speed_hgram", Bins: netStats.NetRxSpeedHgram.Bins})
				}
				if netStats.NetTxSpeedHgram != nil {
					hstats = append(hstats, &rpcpb.GetPortoinstStatsResponse_HistogramStat{Name: "net_tx_speed_hgram", Bins: netStats.NetTxSpeedHgram.Bins})
				}
			}

			gpuStats := statsStorage.GpuStats
			if gpuStats != nil {
				for _, v := range []struct {
					name     string
					binWidth uint64
					values   []*rpcpb.GetGpuStatResponse_GpuStatRsp
				}{
					{name: "gpu_temperature", binWidth: tempratureBinWidth, values: gpuStats.GpuTemperature.GpuStat},
					{name: "gpu_utilization", binWidth: utilizationBinWidth, values: gpuStats.GpuUtilization.GpuStat},
					{name: "gpu_memory_utilization", binWidth: utilizationBinWidth, values: gpuStats.GpuMemoryUtilization.GpuStat},
				} {
					hgram := hgram{bins: map[uint64]uint64{}, binWidth: v.binWidth}
					for _, stat := range v.values {
						metric := fmt.Sprintf("%s_%s_tmmv", v.name, stat.Dev)
						stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.Val)})
						hgram.add(stat.Val)
					}
					if len(hgram.bins) > 0 {
						metric := fmt.Sprintf("%s_hgram", v.name)
						hstats = append(hstats, &rpcpb.GetPortoinstStatsResponse_HistogramStat{Name: metric, Bins: hgram.protoMarshal()})
					}
				}
			}

			ioAllocResp := statsStorage.IOAllocStats
			if ioAllocResp != nil {
				for _, stat := range ioAllocResp.IoAllocStats {
					basename := fmt.Sprintf("io_alloc_%s", stat.AllocName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "read_bw_bytes"), Val: float64(stat.ReadBwBytes)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "write_bw_bytes"), Val: float64(stat.WriteBwBytes)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "read_ops"), Val: float64(stat.ReadOps)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "write_ops"), Val: float64(stat.WriteOps)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "read_bw_limit"), Val: float64(stat.ReadBwLimit)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "write_bw_limit"), Val: float64(stat.WriteBwLimit)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "read_ops_limit"), Val: float64(stat.ReadOpsLimit)})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_%s_tmmv", basename, "write_ops_limit"), Val: float64(stat.WriteOpsLimit)})
				}
			}

			lvmStats := statsStorage.LvmStats
			if lvmStats != nil {
				for _, stat := range lvmStats.VolStats {
					volName := sanitizeVolName(stat.VolName)

					metric := fmt.Sprintf("lvm-avail_space_gb_%s_tnnv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: toGB(stat.AvailableBytes)})
					metric = fmt.Sprintf("lvm-usage_space_gb_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: toGB(stat.UsedBytes)})

					metric = fmt.Sprintf("lvm-read_bytes_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.ReadBytes)})
					metric = fmt.Sprintf("lvm-write_bytes_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.WriteBytes)})

					metric = fmt.Sprintf("lvm-read_ops_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.ReadOps)})
					metric = fmt.Sprintf("lvm-write_ops_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.WriteOps)})

					metric = fmt.Sprintf("lvm-io_in_progress_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.IoInProgress)})

					metric = fmt.Sprintf("lvm-await_%s_tmmv", volName)
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: metric, Val: (float64)(stat.Await)})
				}
			}

			capacityStats := statsStorage.CapacityStats
			if capacityStats != nil {
				for _, stat := range capacityStats.CapacityStats {
					metric := fmt.Sprintf("capacity-perc_usage_%s", stat.VolName)

					hgram := hgram{bins: map[uint64]uint64{}, binWidth: baseBinWidth}
					hgram.add(stat.PercUsage)
					hstats = append(hstats, &rpcpb.GetPortoinstStatsResponse_HistogramStat{Name: fmt.Sprintf("%s_hgram", metric), Bins: hgram.protoMarshal()})
					stats = append(stats, &rpcpb.GetPortoinstStatsResponse_ScalarStat{Name: fmt.Sprintf("%s_txxx", metric), Val: float64(stat.PercUsage)})
				}
			}

			if len(stats)+len(hstats) > 0 {
				portoinstStats = append(portoinstStats, &rpcpb.GetPortoinstStatsResponse_PortoinstStat{CtName: ctName, ScalarStats: stats, HistogramStats: hstats})
			}
		}
	}

	return &rpcpb.GetPortoinstStatsResponse{PortoinstStats: portoinstStats}, nil
}
