package server

import (
	"errors"
	"fmt"
	"math/rand"
	"testing"
	"time"

	rpcpb "a.yandex-team.ru/infra/porto/plugins/portostatd/portostatd_rpc"
)

func makeBins(xs ...uint64) []*rpcpb.HistogramBin {
	if len(xs)%2 != 0 {
		panic(fmt.Sprintf("input length must be even"))
	}
	bins := make([]*rpcpb.HistogramBin, 0, len(xs)/2)
	for i := 0; i < len(xs); i += 2 {
		bin := &rpcpb.HistogramBin{
			LeftEdge: float64(xs[i]),
			Count:    xs[i+1],
		}
		bins = append(bins, bin)
	}
	return bins
}

func cmpBins(xs, ys []*rpcpb.HistogramBin) bool {
	if len(xs) != len(ys) {
		return false
	}
	for i := range xs {
		if xs[i].LeftEdge != ys[i].LeftEdge || xs[i].Count != ys[i].Count {
			return false
		}
	}
	return true
}

func checkBins(bins []*rpcpb.HistogramBin) error {
	if len(bins) == 0 {
		return nil
	}
	if bins[0].Count == 0 {
		return errors.New("first bin must be non-empty")
	}
	for i := 0; i < len(bins)-1; i++ {
		if bins[i+1].LeftEdge <= bins[i].LeftEdge {
			return errors.New("bin edges must be strictly ascending")
		}
	}
	if bins[len(bins)-1].Count != 0 {
		return errors.New("last bin must be non-empty")
	}
	return nil
}

func TestHgram(t *testing.T) {
	for _, tcase := range []struct {
		binw   uint64
		input  []uint64
		output []*rpcpb.HistogramBin
	}{
		{
			binw:   5,
			input:  []uint64{4, 20, 3, 15, 0},
			output: makeBins(0, 3, 5, 0, 10, 0, 15, 1, 20, 1),
		},
		{
			binw:   1,
			input:  []uint64{0},
			output: makeBins(0, 1),
		},
		{
			binw:   3,
			input:  []uint64{0},
			output: makeBins(0, 1),
		},
		{
			binw:   1,
			input:  []uint64{},
			output: makeBins(),
		},
	} {
		if len(tcase.output) > 0 {
			tcase.output = append(
				tcase.output,
				&rpcpb.HistogramBin{LeftEdge: tcase.output[len(tcase.output)-1].LeftEdge + float64(tcase.binw), Count: 0},
			)
		}
		h := hgram{bins: map[uint64]uint64{}, binWidth: tcase.binw}
		for _, v := range tcase.input {
			h.add(v)
		}
		output := h.protoMarshal()
		if !cmpBins(output, tcase.output) {
			t.Fatalf("failed for %v:\nexpected: %v\nactual: %v", tcase.input, tcase.output, output)
		}
	}
}

func TestHgramRandom(t *testing.T) {
	r := rand.New(rand.NewSource(time.Now().UnixNano()))
	for i := 0; i < 1000; i++ {
		xs := make([]uint64, r.Intn(1000))
		d := int(100. * r.Float64() * float64(len(xs)))
		b := uint64(r.Intn(d))
		h := hgram{bins: map[uint64]uint64{}, binWidth: b}
		for i := range xs {
			xs[i] = uint64(r.Intn(d))
			h.add(xs[i])
		}
		bins := h.protoMarshal()
		if err := checkBins(bins); err != nil {
			t.Fatalf("failed for %v:\n%v", xs, err)
		}
	}
}
